<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class EmployeeRequest extends FormRequest
{
    /**
     * تحديد ما إذا كان المستخدم مخولًا لتنفيذ هذا الطلب.
     *
     * @return bool
     */
    public function authorize()
    {
        return true; // يمكن ربطه بصلاحية معينة لاحقًا
    }

    /**
     * استرجاع قواعد التحقق الخاصة بالطلب.
     *
     * @return array
     */
    public function rules()
    {
        $isUpdate = $this->isMethod('PUT') || $this->isMethod('PATCH');
        $employeeId = $isUpdate ? $this->route('employee')->id : null;

        $rules = [
            // رقم الموظف (nullable للسماح بالتوليد التلقائي)
            'employee_id' => [
                'nullable', 
                'string',
                'max:20',
                             'regex:/^EMP\d{4}\d{4}$/', // ✅ إضافة التحقق باستخدام regex
                Rule::unique('employees', 'employee_id')->ignore($employeeId)
            ],
            'name' => 'required|string|max:150',
            'email' => [
                'nullable',
                'email',
                'max:191',
                Rule::unique('employees', 'email')->ignore($employeeId)
            ],
            'phone' => 'nullable|string|max:20',
            'user_id' => 'nullable|exists:users,id',
            'position_id' => 'nullable|exists:positions,id',
            'department_id' => 'nullable|exists:departments,id',
            'hire_date' => 'nullable|date',
            'salary' => 'nullable|numeric|min:0',
            'incentives' => 'nullable|numeric|min:0', // 🔄 تم التعديل إلى 'incentives'
            'status' => 'required|in:active,inactive,pending',
            'birth_date' => 'nullable|date|before_or_equal:today',
            'age' => 'nullable|integer|min:16|max:100',
            'qualification' => 'nullable|string|max:255',
            'national_id' => 'nullable|string|max:50|unique:employees,national_id,' . ($employeeId ?? 'NULL'),
            'address' => 'nullable|string|max:500',
            'scheduled_check_in' => 'nullable|date_format:H:i',
            'scheduled_check_out' => 'nullable|date_format:H:i|after:scheduled_check_in',
            'monthly_leave_days_allowed' => 'nullable|integer|min:0|max:30',
            'annual_entitlement' => 'required|integer|min:0',
            'overtime_hourly_rate' => 'nullable|numeric|min:0',
            'deduction_hourly_rate' => 'nullable|numeric|min:0',
            'default_paid_off_amount' => 'nullable|numeric|min:0',
            'deduct_if_underworked' => 'nullable|boolean',
            'overtime_paid' => 'nullable|boolean',
            'accrue_leaves' => 'nullable|boolean',
            'weekly_off_days' => 'nullable|array',
            'weekly_off_days.*' => 'nullable|string',
            'photo' => $isUpdate ? 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048' : 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'cv_files.*' => 'nullable|mimes:pdf,doc,docx|max:5120', // كل ملف حتى 5MB
        ];

        // إزالة photo من القواعد في حالة التحديث دون تغيير الصورة
        if ($isUpdate && !$this->hasFile('photo')) {
            unset($rules['photo']);
        }

        return $rules;
    }

    /**
     * رسائل الخطأ المخصصة لكل قاعدة تحقق.
     *
     * @return array
     */
    public function messages()
    {
        return [
            // رقم الموظف
            'employee_id.unique' => 'رقم الموظف مستخدم مسبقًا.',
            'employee_id.max' => 'لا يجوز أن يتجاوز رقم الموظف :max حرف.',

            // الاسم
            'name.required' => 'الاسم الكامل مطلوب.',
            'name.max' => 'لا يجوز أن يتجاوز الاسم :max حرفًا.',

            // البريد الإلكتروني
            'email.email' => 'يجب أن يكون البريد الإلكتروني عنوان بريد إلكتروني صالحًا.',
            'email.unique' => 'البريد الإلكتروني مستخدم مسبقًا.',

            // الهاتف
            'phone.max' => 'لا يجوز أن يتجاوز رقم الهاتف :max رقمًا.',

            // الوظيفة والقسم
            'position_id.exists' => 'المنصب المحدد غير موجود.',
            'department_id.exists' => 'القسم المحدد غير موجود.',

            // تاريخ التوظيف
            'hire_date.date' => 'تاريخ التوظيف غير صحيح.',

            // الراتب والحوافز/البدلات
            'salary.numeric' => 'الراتب الأساسي يجب أن يكون رقمًا.',
            'salary.min' => 'لا يمكن أن يكون الراتب الأساسي أقل من صفر.',
            'incentives.numeric' => 'قيمة البدلات/الحوافز يجب أن تكون رقمًا.', // 🔄 تم التعديل
            'incentives.min' => 'لا يمكن أن تكون قيمة البدلات/الحوافز سالبة.', // 🔄 تم التعديل

            // الحالة
            'status.required' => 'حالة الموظف مطلوبة.',
            'status.in' => 'الحالة المحددة غير صالحة.',

            // تاريخ الميلاد
            'birth_date.before_or_equal' => 'لا يمكن أن يكون تاريخ الميلاد في المستقبل.',
            'birth_date.date' => 'تاريخ الميلاد غير صحيح.',

            // العمر
            'age.integer' => 'العمر يجب أن يكون رقمًا صحيحًا.',
            'age.min' => 'الحد الأدنى للعمر هو :min سنة.',
            'age.max' => 'الحد الأقصى للعمر هو :max سنة.',

            // المؤهل والرقم الوطني
            'qualification.max' => 'المؤهل العلمي لا يجوز أن يتجاوز :max حرفًا.',
            'national_id.unique' => 'الرقم الوطني مستخدم مسبقًا.',
            'national_id.max' => 'الرقم الوطني لا يجوز أن يتجاوز :max حرفًا.',

            // العنوان
            'address.max' => 'العنوان لا يجوز أن يتجاوز :max حرفًا.',

            // أوقات الحضور والانصراف
            'scheduled_check_in.date_format' => 'يجب أن يكون وقت الحضور بصيغة ساعة:دقيقة (مثل: 09:00).',
            'scheduled_check_out.date_format' => 'يجب أن يكون وقت الانصراف بصيغة ساعة:دقيقة (مثل: 17:00).',
            'scheduled_check_out.after' => 'يجب أن يكون وقت الانصراف بعد وقت الحضور.',

            // أيام الإجازة الشهرية
            'monthly_leave_days_allowed.integer' => 'عدد أيام الإجازة يجب أن يكون رقماً صحيحاً.',
            'monthly_leave_days_allowed.min' => 'عدد أيام الإجازة لا يمكن أن يكون أقل من 0.',
            'monthly_leave_days_allowed.max' => 'عدد أيام الإجازة لا يمكن أن يتجاوز 30 يومًا.',

            // الرصيد السنوي
            'annual_entitlement.required' => 'الرصيد السنوي مطلوب.',
            'annual_entitlement.integer' => 'الرصيد السنوي يجب أن يكون رقماً صحيحاً.',
            'annual_entitlement.min' => 'الرصيد السنوي لا يمكن أن يكون أقل من 0.',

            // الصورة
            'photo.required' => 'صورة الموظف مطلوبة عند الإنشاء.',
            'photo.image' => 'يجب أن يكون الملف المرفق صورة.',
            'photo.mimes' => 'يجب أن تكون الصورة بصيغة: jpeg, png, jpg، أو gif.',
            'photo.max' => 'لا يجوز أن يتجاوز حجم الصورة 2 ميجابايت.',

            // السيرة الذاتية
            'cv_files.*.mimes' => 'يجب أن تكون ملفات السيرة الذاتية بصيغة PDF أو DOC أو DOCX.',
            'cv_files.*.max' => 'لا يجوز أن يتجاوز حجم كل ملف سيرة ذاتية 5 ميجابايت.',
            'deduct_if_underworked.boolean' => 'قيمة خيار الخصم غير صحيحة.',
            'overtime_paid.boolean' => 'قيمة خيار احتساب الأوفر تايم غير صحيحة.',
        ];
    }
}