<?php

namespace App\Http\Middleware;

use App\Models\Tenant; // نموذج المستأجر
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Str; // لاستخدام Str::contains

class HandleTenant
{
    public function handle(Request $request, Closure $next)
    {
        // ✅ 1. جلب الـ host من الـ URL
        $host = $request->getHost(); // مثلاً: shark_company.localhost, shark_company.127.0.0.1:8000, shark-company.myapp.com

        // ✅ 2. تحديد subdomain بذكاء
        $subdomain = null;

        // ✅ التحقق مما إذا كان الـ host يحتوي على localhost أو 127.0.0.1 (بيئة التطوير)
        if (Str::contains($host, 'localhost') || Str::contains($host, '127.0.0.1')) {
            // في حالة localhost أو 127.0.0.1، نستخدم الجزء الأول قبل النقطة (مثل: shark_company.localhost)
            $parts = explode('.', $host);
            // نحتاج على الأقل جزأين (subdomain.localhost) أو (subdomain.127.0.0.1) لاستخراج subdomain
            if (count($parts) >= 2) {
                $subdomain = $parts[0];
            }
            // إذا كان المضيف هو فقط localhost أو 127.0.0.1 (بدون subdomain)، $subdomain سيبقى null
        } else {
            // ✅ للـ domains الحقيقية (الاستضافة - Production) (مثل: shark-company.myapp.com)
            // نفترض تنسيق subdomain.domain.com (أو subdomain.domain.tld.tld2)
            $parts = explode('.', $host);
            // نحتاج على الأقل 3 أجزاء (subdomain.domain.com) لتحديد أن الجزء الأول هو subdomain
            if (count($parts) >= 3) {
                $subdomain = $parts[0];
            }
            // ملاحظة: إذا كان لديك تنسيق مختلف (مثلاً subdomain.domain.tld.tld2)، قد تحتاج لتعديل الشرط
        }

        // ✅ 3. التحقق مما إذا كان subdomain موجودًا وصالحًا
        // نتجاهل subdomains الشائعة مثل admin, www
        $ignoredSubdomains = ['admin', 'www', 'localhost', '127', '0', '0', '1', 'test', 'dev']; // أضف أي subdomains ترغب في تجاهلها

        if ($subdomain && !in_array($subdomain, $ignoredSubdomains)) {
            // ✅ البحث عن المستأجر في قاعدة البيانات
            $tenant = Tenant::where('subdomain', $subdomain)->first();

            if (!$tenant || $tenant->status !== 'active') {
                // إذا لم توجد الشركة أو كانت غير نشطة، أعد خطأ
                abort(404, 'الشركة غير موجودة أو معطلة.');
            }

            // ✅ حفظ معرف المستأجر في الجلسة
            session(['current_tenant_id' => $tenant->id]);
            session(['current_tenant_subdomain' => $tenant->subdomain]); // (اختياري) لاستخدامه في الترقيم أو التحقق
        } else {
            // ✅ إذا لم يتم تحديد subdomain (مثلاً عند الدخول إلى localhost:8000 مباشرة أو myapp.com)
            // نمسح معلومات tenant_id من الجلسة لضمان عدم استخدام بيانات شركة أخرى
            session()->forget('current_tenant_id');
            session()->forget('current_tenant_subdomain');
        }

        return $next($request);
    }
}