<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Role; // استيراد نموذج Role
use App\Models\Activity; // ✅ استيراد نموذج النشاط
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB; // استيراد DB لاستخدام transaction

class UserController extends Controller
{
    /**
     * عرض قائمة بجميع المستخدمين.
     */
    public function index(): View
    {
        $this->authorize('viewAny', User::class);

        // جلب البيانات للإحصائيات
        $totalUsers = User::count();
        $activeUsers = User::where('status', 'active')->count();
        $inactiveUsers = User::where('status', 'inactive')->count();

        // التحقق من السوبر أدمن (مثلاً المستخدم ذو المعرف 1)
        $adminUsers = User::where('id', 1)->count(); // يمكنك تغيير الشرط حسب نظامك

        // جلب المستخدمين مع الأدوار و ترقيم الصفحات
        $users = User::with(['employee', 'roles'])->latest()->paginate(15);

        // جلب الأدوار و الموظفين لاستخدامهم في النماذج
        $roles = Role::all(); // استرجاع جميع الأدوار
        $employees = \App\Models\Employee::orderBy('name')->get();

        return view('users.index', compact(
            'users',
            'roles', // تمرير الأدوار إلى الـ View
            'employees',
            'totalUsers',
            'activeUsers',
            'inactiveUsers',
            'adminUsers'
        ));
    }

    /**
     * عرض تفاصيل مستخدم محدد.
     */
    public function show(User $user): View
    {
        $this->authorize('view', $user);

        $user->load(['employee', 'roles']); // تحميل الأدوار أيضًا

        return view('users.show', compact('user'));
    }

    /**
     * عرض نموذج إنشاء مستخدم جديد.
     */
    public function create(): View
    {
        $this->authorize('create', User::class);

        $employees = \App\Models\Employee::orderBy('name')->get();
        $roles = Role::all(); // تمرير الأدوار إلى الـ View

        return view('users.create', compact('employees', 'roles'));
    }

    /**
     * تخزين مستخدم جديد في قاعدة البيانات.
     */
    public function store(Request $request): RedirectResponse
    {
        $this->authorize('create', User::class);

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'employee_id' => 'nullable|exists:employees,id',
            'status' => 'required|in:active,inactive',
            'role' => 'required|exists:roles,name', // التحقق من أن الدور المدخل موجود في جدول roles
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ], [
            'name.required' => 'الاسم مطلوب',
            'email.required' => 'البريد الإلكتروني مطلوب',
            'email.email' => 'صيغة البريد الإلكتروني غير صحيحة',
            'email.unique' => 'البريد الإلكتروني مستخدم مسبقًا',
            'password.required' => 'كلمة المرور مطلوبة',
            'password.min' => 'كلمة المرور يجب أن تكون على الأقل 8 أحرف',
            'password.confirmed' => 'تأكيد كلمة المرور غير متطابق',
            'employee_id.exists' => 'الموظف المحدد غير موجود',
            'role.required' => 'الدور مطلوب',
            'role.exists' => 'الدور المحدد غير موجود',
            'status.required' => 'الحالة مطلوبة',
            'photo.image' => 'الملف يجب أن يكون صورة',
            'photo.mimes' => 'صيغة الصورة يجب أن تكون jpeg, png, jpg, أو gif',
            'photo.max' => 'حجم الصورة لا يمكن أن يتجاوز 2 ميجابايت',
        ]);

        DB::beginTransaction(); // لضمان سلامة البيانات

        try {
            $data = $request->only(['name', 'email', 'employee_id', 'status']);
            $data['password'] = Hash::make($request->password);

            // رفع الصورة إذا وُجدت
            if ($request->hasFile('photo')) {
                $data['photo'] = $request->file('photo')->store('users', 'public');
            }

            $user = User::create($data);

            // ربط المستخدم بالدور المحدد
            $role = Role::where('name', $request->role)->first();
            if ($role) {
                $user->roles()->attach($role->id); // استخدام attach لربط العلاقة
            }

            // ✅ تسجيل نشاط إنشاء المستخدم
            Activity::create([
                'user_id' => auth()->id(), // المستخدم الحالي قام بالنشاط
                'description' => 'created_user', // وصف النشاط
                'subject_type' => User::class, // نوع الكيان المتأثر
                'subject_id' => $user->id, // معرف الكيان المتأثر
                'properties' => [
                    'user_name' => $user->name,
                    'user_email' => $user->email,
                    'role_assigned' => $role->name ?? 'N/A',
                ],
            ]);

            DB::commit();
            return redirect()->route('users.index')->with('success', 'تم إنشاء المستخدم وربطه بالدور بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors(['error' => 'حدث خطأ أثناء إنشاء المستخدم: ' . $e->getMessage()]);
        }
    }

    /**
     * عرض نموذج تعديل المستخدم.
     */
    public function edit(User $user): View
    {
        $this->authorize('update', $user);

        $employees = \App\Models\Employee::orderBy('name')->get();
        $roles = Role::all(); // تمرير الأدوار إلى الـ View

        return view('users.edit', compact('user', 'employees', 'roles'));
    }

    /**
     * تحديث مستخدم.
     */
    public function update(Request $request, User $user): RedirectResponse
    {
        $this->authorize('update', $user);

        // لا يمكن تعديل السوبر أدمن (المستخدم ذو المعرف 1)
        if ($user->id == 1) {
            return redirect()->route('users.index')->with('error', 'لا يمكن تعديل حساب السوبر أدمن.');
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . $user->id,
            'password' => 'nullable|string|min:8|confirmed',
            'employee_id' => 'nullable|exists:employees,id',
            'status' => 'required|in:active,inactive',
            'role' => 'nullable|exists:roles,name', // الدور الجديد (اختياري)
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ], [
            'name.required' => 'الاسم مطلوب',
            'email.required' => 'البريد الإلكتروني مطلوب',
            'email.email' => 'صيغة البريد الإلكتروني غير صحيحة',
            'email.unique' => 'البريد الإلكتروني مستخدم مسبقًا',
            'password.min' => 'كلمة المرور يجب أن تكون على الأقل 8 أحرف',
            'password.confirmed' => 'تأكيد كلمة المرور غير متطابق',
            'employee_id.exists' => 'الموظف المحدد غير موجود',
            'role.exists' => 'الدور المحدد غير موجود',
            'status.required' => 'الحالة مطلوبة',
            'photo.image' => 'الملف يجب أن يكون صورة',
            'photo.mimes' => 'صيغة الصورة يجب أن تكون jpeg, png, jpg, أو gif',
            'photo.max' => 'حجم الصورة لا يمكن أن يتجاوز 2 ميجابايت',
        ]);

        DB::beginTransaction();

        try {
            $data = $request->only(['name', 'email', 'employee_id', 'status']);

            // تحديث كلمة المرور فقط إذا تم إدخالها
            if ($request->filled('password')) {
                $data['password'] = Hash::make($request->password);
            }

            // إدارة الصورة
            if ($request->hasFile('photo')) {
                // حذف الصورة القديمة إذا وُجدت
                if ($user->photo && Storage::disk('public')->exists($user->photo)) {
                    Storage::disk('public')->delete($user->photo);
                }
                // رفع الصورة الجديدة
                $data['photo'] = $request->file('photo')->store('users', 'public');
            }

            $user->update($data);

            // ✅ تخزين الحالة القديمة للدور قبل التحديث (لإظهارها في النشاط)
            $oldRoleNames = $user->roles->pluck('name')->join(', ');

            // تحديث الدور
            if ($request->filled('role')) {
                $newRole = Role::where('name', $request->role)->first();
                if ($newRole) {
                    // تفريغ الأدوار السابقة وربط الدور الجديد فقط
                    $user->roles()->detach(); // تفريغ العلاقة الحالية
                    $user->roles()->attach($newRole->id); // ربط الدور الجديد
                }
            }

            // ✅ تسجيل نشاط تحديث المستخدم
            Activity::create([
                'user_id' => auth()->id(), // المستخدم الحالي قام بالنشاط
                'description' => 'updated_user', // وصف النشاط
                'subject_type' => User::class, // نوع الكيان المتأثر
                'subject_id' => $user->id, // معرف الكيان المتأثر
                'properties' => [
                    'user_name' => $user->name,
                    'user_email' => $user->email,
                    'old_role' => $oldRoleNames ?: 'N/A',
                    'new_role' => $newRole->name ?? 'N/A',
                    'fields_updated' => implode(', ', array_keys($request->only(['name', 'email', 'employee_id', 'status', 'photo']))),
                ],
            ]);

            DB::commit();
            return redirect()->route('users.index')->with('success', 'تم تحديث المستخدم بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors(['error' => 'حدث خطأ أثناء تحديث المستخدم: ' . $e->getMessage()]);
        }
    }

    /**
     * حذف مستخدم.
     */
    public function destroy(User $user): RedirectResponse
    {
        $this->authorize('delete', $user);

        // لا يمكن حذف السوبر أدمن
        if ($user->id == 1) {
            return redirect()->route('users.index')->with('error', 'لا يمكن حذف حساب السوبر أدمن.');
        }

        DB::beginTransaction();

        try {
            // ✅ تخزين معلومات المستخدم قبل الحذف لاستخدامها في النشاط
            $userName = $user->name;
            $userEmail = $user->email;
            $userRoleNames = $user->roles->pluck('name')->join(', ');

            // ✅ تسجيل نشاط حذف المستخدم (قبل الحذف الفعلي)
            Activity::create([
                'user_id' => auth()->id(), // المستخدم الحالي قام بالنشاط
                'description' => 'deleted_user', // وصف النشاط
                'subject_type' => User::class, // نوع الكيان المتأثر (تم حذفه)
                'subject_id' => $user->id, // معرف الكيان المتأثر (تم حذفه)
                'properties' => [
                    'user_name' => $userName,
                    'user_email' => $userEmail,
                    'roles' => $userRoleNames ?: 'N/A',
                ],
            ]);

            // حذف الصورة من التخزين
            if ($user->photo && Storage::disk('public')->exists($user->photo)) {
                Storage::disk('public')->delete($user->photo);
            }

            $user->delete(); // هذا سيؤدي إلى حذف السجلات المرتبطة في user_roles بسبب onDelete('cascade')

            DB::commit();
            return redirect()->route('users.index')->with('success', 'تم حذف المستخدم بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors(['error' => 'حدث خطأ أثناء حذف المستخدم: ' . $e->getMessage()]);
        }
    }
}