<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\TrainingEnrollment;
use App\Models\TrainingProgram;
use App\Models\Employee;

class TrainingEnrollmentController extends Controller
{
    public function index()
    {
        $this->authorize('viewAny', TrainingEnrollment::class);
        
        $query = TrainingEnrollment::with(['employee', 'program']);

        // تصفية البيانات
        if (request('search')) {
            $search = request('search');
            $query->whereHas('employee', function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('employee_id', 'like', "%{$search}%");
            })->orWhereHas('program', function($q) use ($search) {
                $q->where('title', 'like', "%{$search}%");
            });
        }

        if (request('employee_id')) {
            $query->where('employee_id', request('employee_id'));
        }

        if (request('program_id')) {
            $query->where('program_id', request('program_id'));
        }

        if (request('status')) {
            $query->where('status', request('status'));
        }

        $enrollments = $query->latest()->paginate(15);
        
        // إضافة قائمة الموظفين والبرامج للاستخدام في النموذج
        $employees = Employee::orderBy('name')->get();
        $programs = TrainingProgram::orderBy('title')->get();

        return view('training.enrollments.index', compact('enrollments', 'employees', 'programs'));
    }

    public function create()
    {
        $this->authorize('create', TrainingEnrollment::class);
        $employees = Employee::orderBy('name')->get();
        $programs = TrainingProgram::where('status', 'active')->orderBy('title')->get();
        return view('training.enrollments.create', compact('employees', 'programs'));
    }

    public function store(Request $request)
    {
        $this->authorize('create', TrainingEnrollment::class);
        
        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'program_id' => 'required|exists:training_programs,id',
            'status' => 'required|in:enrolled,in_progress,completed,cancelled',
            'progress_percentage' => 'nullable|integer|min:0|max:100',
            'score' => 'nullable|integer|min:0|max:100',
            'notes' => 'nullable|string|max:500'
        ], [
            'employee_id.required' => 'الموظف مطلوب',
            'employee_id.exists' => 'الموظف المحدد غير موجود',
            'program_id.required' => 'البرنامج التدريبي مطلوب',
            'program_id.exists' => 'البرنامج التدريبي المحدد غير موجود',
            'status.required' => 'الحالة مطلوبة',
            'status.in' => 'الحالة المحددة غير صحيحة',
            'progress_percentage.integer' => 'نسبة التقدم يجب أن تكون رقماً صحيحاً',
            'progress_percentage.min' => 'نسبة التقدم لا يمكن أن تكون أقل من 0',
            'progress_percentage.max' => 'نسبة التقدم لا يمكن أن تكون أكثر من 100',
            'score.integer' => 'النتيجة يجب أن تكون رقماً صحيحاً',
            'score.min' => 'النتيجة لا يمكن أن تكون أقل من 0',
            'score.max' => 'النتيجة لا يمكن أن تكون أكثر من 100',
            'notes.max' => 'الملاحظات لا يمكن أن تتجاوز 500 حرف'
        ]);

        // التحقق من عدم تكرار التسجيل
        $existing = TrainingEnrollment::where('employee_id', $request->employee_id)
                                    ->where('program_id', $request->program_id)
                                    ->first();
        
        if ($existing) {
            return redirect()->back()
                           ->with('error', 'هذا الموظف مسجل مسبقاً في هذا البرنامج التدريبي')
                           ->withInput();
        }

        TrainingEnrollment::create([
            'employee_id' => $request->employee_id,
            'program_id' => $request->program_id,
            'status' => $request->status,
            'progress_percentage' => $request->progress_percentage ?? 0,
            'score' => $request->score,
            'notes' => $request->notes
        ]);

        return redirect()->route('training.enrollments.index')
                        ->with('success', 'تم تسجيل الموظف في البرنامج التدريبي بنجاح');
    }

    public function show(TrainingEnrollment $enrollment)
    {
        $this->authorize('view', $enrollment);
        $enrollment->load(['employee', 'program', 'program.trainer']);
        return view('training.enrollments.show', compact('enrollment'));
    }

    public function edit(TrainingEnrollment $enrollment)
    {
        $this->authorize('update', $enrollment);
        $employees = Employee::orderBy('name')->get();
        $programs = TrainingProgram::orderBy('title')->get();
        $enrollment->load(['employee', 'program']);
        return view('training.enrollments.edit', compact('enrollment', 'employees', 'programs'));
    }

    public function update(Request $request, TrainingEnrollment $enrollment)
    {
        $this->authorize('update', $enrollment);
        
        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'program_id' => 'required|exists:training_programs,id',
            'status' => 'required|in:enrolled,in_progress,completed,cancelled',
            'progress_percentage' => 'nullable|integer|min:0|max:100',
            'score' => 'nullable|integer|min:0|max:100',
            'notes' => 'nullable|string|max:500'
        ], [
            'employee_id.required' => 'الموظف مطلوب',
            'employee_id.exists' => 'الموظف المحدد غير موجود',
            'program_id.required' => 'البرنامج التدريبي مطلوب',
            'program_id.exists' => 'البرنامج التدريبي المحدد غير موجود',
            'status.required' => 'الحالة مطلوبة',
            'status.in' => 'الحالة المحددة غير صحيحة',
            'progress_percentage.integer' => 'نسبة التقدم يجب أن تكون رقماً صحيحاً',
            'progress_percentage.min' => 'نسبة التقدم لا يمكن أن تكون أقل من 0',
            'progress_percentage.max' => 'نسبة التقدم لا يمكن أن تكون أكثر من 100',
            'score.integer' => 'النتيجة يجب أن تكون رقماً صحيحاً',
            'score.min' => 'النتيجة لا يمكن أن تكون أقل من 0',
            'score.max' => 'النتيجة لا يمكن أن تكون أكثر من 100',
            'notes.max' => 'الملاحظات لا يمكن أن تتجاوز 500 حرف'
        ]);

        $enrollment->update([
            'employee_id' => $request->employee_id,
            'program_id' => $request->program_id,
            'status' => $request->status,
            'progress_percentage' => $request->progress_percentage ?? 0,
            'score' => $request->score,
            'notes' => $request->notes
        ]);

        return redirect()->route('training.enrollments.index')
                        ->with('success', 'تم تحديث التسجيل في البرنامج التدريبي بنجاح');
    }

    public function destroy(TrainingEnrollment $enrollment)
    {
        $this->authorize('delete', $enrollment);
        $enrollment->delete();
        return redirect()->route('training.enrollments.index')
                        ->with('success', 'تم حذف التسجيل في البرنامج التدريبي بنجاح');
    }

    public function approve(TrainingEnrollment $enrollment)
    {
        $this->authorize('approve', $enrollment);
        
        $enrollment->update([
            'status' => 'approved',
            'approved_by' => auth()->id(),
            'approved_at' => now()
        ]);

        return redirect()->route('training.enrollments.index')
                        ->with('success', 'تم الموافقة على التسجيل في البرنامج التدريبي');
    }

    public function reject(Request $request, TrainingEnrollment $enrollment)
    {
        $this->authorize('approve', $enrollment);
        
        $request->validate([
            'rejection_reason' => 'required|string|max:500'
        ], [
            'rejection_reason.required' => 'سبب الرفض مطلوب',
            'rejection_reason.max' => 'سبب الرفض لا يمكن أن يتجاوز 500 حرف'
        ]);

        $enrollment->update([
            'status' => 'rejected',
            'rejection_reason' => $request->rejection_reason,
            'rejected_by' => auth()->id(),
            'rejected_at' => now()
        ]);

        return redirect()->route('training.enrollments.index')
                        ->with('success', 'تم رفض التسجيل في البرنامج التدريبي');
    }

    public function complete(TrainingEnrollment $enrollment)
    {
        $this->authorize('complete', $enrollment);
        
        $enrollment->update([
            'status' => 'completed',
            'completed_at' => now()
        ]);

        return redirect()->route('training.enrollments.index')
                        ->with('success', 'تم إكمال البرنامج التدريبي للموظف');
    }
}