<?php

namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\User;
use App\Services\SuperAdminAuditLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;

class SuperAdminController extends Controller
{
    // ==================== لوحة التحكم ====================
    public function dashboard()
    {
        $companiesCount = Company::count();
        $usersCount = User::count();
        $companies = Company::all();
        
        return view('super-admin.dashboard', compact('companiesCount', 'usersCount', 'companies'));
    }

    // ==================== إدارة الشركات ====================
    public function companiesIndex()
    {
        $companies = Company::paginate(15);
        return view('super-admin.companies.index', compact('companies'));
    }

    public function companiesCreate()
    {
        return view('super-admin.companies.create');
    }

    public function companiesStore(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|unique:companies',
            'email' => 'required|email|unique:companies',
            'phone' => 'nullable|string',
            'address' => 'nullable|string',
            'city' => 'nullable|string',
            'country' => 'nullable|string',
            'tax_id' => 'nullable|string|unique:companies',
            'status' => 'required|in:active,inactive,suspended',
            // حقول الفوترة الجديدة
            'subscription_fee' => 'required|numeric|min:0',
            'billing_cycle' => 'required|in:monthly,yearly',
            'subscription_start_date' => 'required|date',
            'subscription_end_date' => 'required|date|after_or_equal:subscription_start_date',
            'max_employees' => 'nullable|integer|min:1',
        ]);

        // حساب تاريخ الدفع التالي
        $nextPaymentDate = $validated['subscription_end_date'];
        if ($validated['billing_cycle'] === 'monthly') {
            $nextPaymentDate = \Carbon\Carbon::parse($validated['subscription_end_date'])->addMonth();
        } else {
            $nextPaymentDate = \Carbon\Carbon::parse($validated['subscription_end_date'])->addYear();
        }

        $companyData = array_merge($validated, [
            'payment_status' => 'paid',
            'last_payment_date' => now(),
            'next_payment_date' => $nextPaymentDate,
            'outstanding_amount' => 0,
            'grace_period_days' => 7,
            'can_add_employees' => true,
            'can_process_payroll' => true,
            'can_access_reports' => true,
            'current_employee_count' => 0,
            'total_logins' => 0,
        ]);

        $company = Company::create($companyData);
        SuperAdminAuditLog::logCompanyCreated($company);

        return redirect()->route('super-admin.companies.index')
            ->with('success', 'تم إنشاء الشركة بنجاح');
    }

    public function companiesEdit(Company $company)
    {
        return view('super-admin.companies.edit', compact('company'));
    }

    public function companiesUpdate(Request $request, Company $company)
    {
        $validated = $request->validate([
            'name' => 'required|string|unique:companies,name,' . $company->id,
            'email' => 'required|email|unique:companies,email,' . $company->id,
            'phone' => 'nullable|string',
            'address' => 'nullable|string',
            'city' => 'nullable|string',
            'country' => 'nullable|string',
            'tax_id' => 'nullable|string|unique:companies,tax_id,' . $company->id,
            'status' => 'required|in:active,inactive,suspended',
        ]);

        $changes = $company->getChanges();
        $company->update($validated);
        SuperAdminAuditLog::logCompanyUpdated($company, $changes);

        return redirect()->route('super-admin.companies.index')
            ->with('success', 'تم تحديث الشركة بنجاح');
    }

    public function companiesDestroy(Company $company)
    {
        SuperAdminAuditLog::logCompanyDeleted($company);
        $company->delete();
        return redirect()->route('super-admin.companies.index')
            ->with('success', 'تم حذف الشركة بنجاح');
    }

    // ==================== إدارة مستخدمي الشركة ====================
    public function usersIndex(Company $company)
    {
        $users = $company->users()->paginate(15);
        return view('super-admin.users.index', compact('company', 'users'));
    }

    public function usersCreate(Company $company)
    {
        return view('super-admin.users.create', compact('company'));
    }

    public function usersStore(Request $request, Company $company)
    {
        $validated = $request->validate([
            'name' => 'required|string',
            'email' => 'required|email|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'role' => 'required|in:admin,manager,employee',
            'status' => 'required|in:active,inactive,pending',
        ]);

        $validated['password'] = Hash::make($validated['password']);
        $validated['company_id'] = $company->id;

        $user = User::create($validated);
        SuperAdminAuditLog::logUserCreated($user, $company);

        return redirect()->route('super-admin.users.index', $company)
            ->with('success', 'تم إنشاء المستخدم بنجاح');
    }

    public function usersEdit(Company $company, User $user)
    {
        if ($user->company_id !== $company->id) {
            abort(403);
        }
        return view('super-admin.users.edit', compact('company', 'user'));
    }

    public function usersUpdate(Request $request, Company $company, User $user)
    {
        if ($user->company_id !== $company->id) {
            abort(403);
        }

        $validated = $request->validate([
            'name' => 'required|string',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'password' => 'nullable|string|min:8|confirmed',
            'role' => 'required|in:admin,manager,employee',
            'status' => 'required|in:active,inactive,pending',
        ]);

        $changes = [];
        if ($validated['password']) {
            $validated['password'] = Hash::make($validated['password']);
            $changes['password'] = 'تم تحديث كلمة المرور';
        } else {
            unset($validated['password']);
        }

        foreach ($validated as $key => $value) {
            if ($user->$key !== $value) {
                $changes[$key] = ['from' => $user->$key, 'to' => $value];
            }
        }

        $user->update($validated);
        if (!empty($changes)) {
            SuperAdminAuditLog::logUserUpdated($user, $company, $changes);
        }

        return redirect()->route('super-admin.users.index', $company)
            ->with('success', 'تم تحديث المستخدم بنجاح');
    }

    public function usersDestroy(Company $company, User $user)
    {
        if ($user->company_id !== $company->id) {
            abort(403);
        }

        SuperAdminAuditLog::logUserDeleted($user, $company);
        $user->delete();
        return redirect()->route('super-admin.users.index', $company)
            ->with('success', 'تم حذف المستخدم بنجاح');
    }

    // ==================== محاكاة دخول (Impersonation) ====================
    public function impersonate(Company $company, User $user)
    {
        if ($user->company_id !== $company->id) {
            abort(403);
        }

        SuperAdminAuditLog::logImpersonation($user, $company);
        session(['impersonated_user_id' => $user->id, 'original_user_id' => auth()->id()]);
        auth()->loginUsingId($user->id);

        return redirect()->route('dashboard')
            ->with('success', 'تم تسجيل الدخول كـ ' . $user->name);
    }

    public function stopImpersonation()
    {
        if (session('original_user_id')) {
            $originalUserId = session('original_user_id');
            $impersonatedUser = User::find(session('impersonated_user_id'));
            
            if ($impersonatedUser) {
                SuperAdminAuditLog::logStopImpersonation($impersonatedUser);
            }
            
            session()->forget(['impersonated_user_id', 'original_user_id']);
            auth()->loginUsingId($originalUserId);

            return redirect()->route('super-admin.dashboard')
                ->with('success', 'تم إنهاء محاكاة الدخول');
        }

        return redirect()->back();
    }

    // ==================== التحكم في الشركات ====================
    public function suspendCompany(Company $company, Request $request)
    {
        $request->validate([
            'reason' => 'required|string|min:5'
        ]);

        $company->suspend($request->reason, auth()->id());
        
        // إيقاف جميع مستخدمي الشركة
        $company->users()->update([
            'can_login' => false,
            'suspended_at' => now(),
            'suspension_reason' => 'تم إيقاف الشركة: ' . $request->reason,
            'suspended_by' => auth()->id()
        ]);

        SuperAdminAuditLog::log('company_suspended', [
            'company_id' => $company->id,
            'company_name' => $company->name,
            'reason' => $request->reason
        ]);

        return redirect()->back()->with('success', 'تم إيقاف الشركة وجميع مستخدميها بنجاح');
    }

    public function unsuspendCompany(Company $company)
    {
        $company->unsuspend();
        
        // إعادة تفعيل جميع مستخدمي الشركة
        $company->users()->update([
            'can_login' => true,
            'suspended_at' => null,
            'suspension_reason' => null,
            'suspended_by' => null
        ]);

        SuperAdminAuditLog::log('company_unsuspended', [
            'company_id' => $company->id,
            'company_name' => $company->name
        ]);

        return redirect()->back()->with('success', 'تم إعادة تفعيل الشركة ومستخدميها بنجاح');
    }

    public function updateCompanyPayment(Company $company, Request $request)
    {
        $request->validate([
            'payment_status' => 'required|in:paid,pending,overdue,cancelled',
            'payment_amount' => 'required|numeric|min:0',
            'payment_notes' => 'nullable|string'
        ]);

        $oldStatus = $company->payment_status;
        $company->payment_status = $request->payment_status;
        $company->payment_notes = $request->payment_notes;
        
        if ($request->payment_status === 'paid' && $request->payment_amount > 0) {
            $company->last_payment_date = now();
            $company->outstanding_amount = max(0, $company->outstanding_amount - $request->payment_amount);
            
            // تحديث تاريخ الدفع التالي
            if ($company->billing_cycle === 'monthly') {
                $company->next_payment_date = now()->addMonth();
            } else {
                $company->next_payment_date = now()->addYear();
            }
        }

        $company->save();

        SuperAdminAuditLog::log('payment_updated', [
            'company_id' => $company->id,
            'company_name' => $company->name,
            'old_status' => $oldStatus,
            'new_status' => $request->payment_status,
            'amount' => $request->payment_amount
        ]);

        return redirect()->back()->with('success', 'تم تحديث حالة الدفع بنجاح');
    }

    public function toggleCompanyFeatures(Company $company, Request $request)
    {
        $features = ['can_add_employees', 'can_process_payroll', 'can_access_reports'];
        
        foreach ($features as $feature) {
            if ($request->has($feature)) {
                $company->$feature = $request->boolean($feature);
            }
        }

        $company->save();

        SuperAdminAuditLog::log('features_toggled', [
            'company_id' => $company->id,
            'company_name' => $company->name,
            'features' => $request->only($features)
        ]);

        return redirect()->back()->with('success', 'تم تحديث صلاحيات الشركة بنجاح');
    }

    // ==================== التحكم في المستخدمين ====================
    public function suspendUser(Company $company, User $user, Request $request)
    {
        $request->validate([
            'reason' => 'required|string|min:5'
        ]);

        $user->suspend($request->reason, auth()->id());

        SuperAdminAuditLog::log('user_suspended', [
            'user_id' => $user->id,
            'user_name' => $user->name,
            'company_id' => $company->id,
            'company_name' => $company->name,
            'reason' => $request->reason
        ]);

        return redirect()->back()->with('success', 'تم إيقاف المستخدم بنجاح');
    }

    public function unsuspendUser(Company $company, User $user)
    {
        $user->unsuspend();

        SuperAdminAuditLog::log('user_unsuspended', [
            'user_id' => $user->id,
            'user_name' => $user->name,
            'company_id' => $company->id,
            'company_name' => $company->name
        ]);

        return redirect()->back()->with('success', 'تم إعادة تفعيل المستخدم بنجاح');
    }

    public function deactivateUser(Company $company, User $user, Request $request)
    {
        $request->validate([
            'reason' => 'required|string|min:5'
        ]);

        $user->deactivate($request->reason, auth()->id());

        SuperAdminAuditLog::log('user_deactivated', [
            'user_id' => $user->id,
            'user_name' => $user->name,
            'company_id' => $company->id,
            'company_name' => $company->name,
            'reason' => $request->reason
        ]);

        return redirect()->back()->with('success', 'تم إلغاء تفعيل المستخدم بنجاح');
    }

    public function activateUser(Company $company, User $user)
    {
        $user->activate();

        SuperAdminAuditLog::log('user_activated', [
            'user_id' => $user->id,
            'user_name' => $user->name,
            'company_id' => $company->id,
            'company_name' => $company->name
        ]);

        return redirect()->back()->with('success', 'تم تفعيل المستخدم بنجاح');
    }

    public function resetUserPassword(Company $company, User $user, Request $request)
    {
        $request->validate([
            'new_password' => 'required|string|min:8|confirmed'
        ]);

        $user->password = Hash::make($request->new_password);
        $user->login_attempts = 0;
        $user->locked_until = null;
        $user->save();

        SuperAdminAuditLog::log('password_reset', [
            'user_id' => $user->id,
            'user_name' => $user->name,
            'company_id' => $company->id,
            'company_name' => $company->name
        ]);

        return redirect()->back()->with('success', 'تم إعادة تعيين كلمة المرور بنجاح');
    }
}

