<?php

namespace App\Http\Controllers;

use App\Models\SalaryComponent;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use App\Models\Activity; // إذا كنت ترغب في تسجيل الأنشطة

class SalaryComponentController extends Controller
{
    /**
     * عرض قائمة بجميع مكونات الراتب.
     */
    public function index()
    {
        // ✅ التحقق من الصلاحية
        $this->authorize('viewAny', SalaryComponent::class);

        // ✅ جلب جميع المكونات
        $salaryComponents = SalaryComponent::orderBy('name')->paginate(15); // أو get() لجلب الكل دفعة واحدة

        return view('salary-components.index', compact('salaryComponents'));
    }

    /**
     * عرض نموذج إنشاء مكون راتب جديد.
     */
    public function create()
    {
        // ✅ التحقق من الصلاحية
        $this->authorize('create', SalaryComponent::class);

        return view('salary-components.create');
    }

    /**
     * حفظ مكون راتب جديد في قاعدة البيانات.
     */
    public function store(Request $request)
    {
        // ✅ التحقق من الصلاحية
        $this->authorize('create', SalaryComponent::class);

        // ✅ التحقق من صحة البيانات
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:salary_components,name',
            'type' => 'nullable|string|max:100', // مثلاً: allowance, bonus, deduction (إذا كنت تستخدم نوع)
            'description' => 'nullable|string|max:1000',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $data = $validator->validated();

        DB::beginTransaction();
        try {
            // ✅ إنشاء المكون
            $component = SalaryComponent::create([
                'name' => $data['name'],
                'type' => $data['type'] ?? null, // اجعل الحقل فارغًا إذا لم يُرسل
                'description' => $data['description'] ?? null,
            ]);

            // ✅ تسجيل النشاط (اختياري)
            Activity::create([
                'user_id' => auth()->id(),
                'action' => 'created_salary_component',
                'model_type' => SalaryComponent::class,
                'model_id' => $component->id,
                'description' => 'تم إنشاء مكون راتب جديد: ' . $component->name,
            ]);

            DB::commit();

            return redirect()->route('salary-components.index')->with('success', 'تم إنشاء مكون الراتب "' . $component->name . '" بنجاح.');
        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('SalaryComponent store error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'حدث خطأ أثناء إنشاء مكون الراتب.')->withInput();
        }
    }

    /**
     * عرض تفاصيل مكون راتب معين.
     */
    public function show(SalaryComponent $salaryComponent) // ✅ استخدام Route Model Binding
    {
        // ✅ التحقق من الصلاحية
        $this->authorize('view', $salaryComponent);

        // ✅ تحميل العلاقات إذا لزم الأمر (مثلاً: من يستخدم هذا المكون)
        // $salaryComponent->load('employees'); // مثال

        return view('salary-components.show', compact('salaryComponent'));
    }

    /**
     * عرض نموذج تعديل مكون راتب موجود.
     */
    public function edit(SalaryComponent $salaryComponent) // ✅ استخدام Route Model Binding
    {
        // ✅ التحقق من الصلاحية
        $this->authorize('update', $salaryComponent);

        return view('salary-components.edit', compact('salaryComponent'));
    }

    /**
     * تحديث مكون راتب موجود في قاعدة البيانات.
     */
    public function update(Request $request, SalaryComponent $salaryComponent) // ✅ استخدام Route Model Binding
    {
        // ✅ التحقق من الصلاحية
        $this->authorize('update', $salaryComponent);

        // ✅ التحقق من صحة البيانات (مع تجاهل اسم المكون الحالي عند التحقق من التكرار)
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:salary_components,name,' . $salaryComponent->id, // تجاهل ID الحالي
            'type' => 'nullable|string|max:100',
            'description' => 'nullable|string|max:1000',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $data = $validator->validated();

        DB::beginTransaction();
        try {
            // ✅ تحديث المكون
            $salaryComponent->update([
                'name' => $data['name'],
                'type' => $data['type'] ?? null,
                'description' => $data['description'] ?? null,
            ]);

            // ✅ تسجيل النشاط (اختياري)
            Activity::create([
                'user_id' => auth()->id(),
                'action' => 'updated_salary_component',
                'model_type' => SalaryComponent::class,
                'model_id' => $salaryComponent->id,
                'description' => 'تم تعديل مكون الراتب: ' . $salaryComponent->name,
            ]);

            DB::commit();

            return redirect()->route('salary-components.index')->with('success', 'تم تحديث مكون الراتب "' . $salaryComponent->name . '" بنجاح.');
        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('SalaryComponent update error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث مكون الراتب.')->withInput();
        }
    }

    /**
     * حذف مكون راتب من قاعدة البيانات.
     */
    public function destroy(SalaryComponent $salaryComponent) // ✅ استخدام Route Model Binding
    {
        // ✅ التحقق من الصلاحية
        $this->authorize('delete', $salaryComponent);

        // ⚠️ تحذير: إذا كان المكون مرتبطًا برواتب أو موظفين، قد تحتاج ل handling خاص (on delete restrict/cascade)
        // مثال: if ($salaryComponent->payrolls()->exists()) { return redirect()->back()->with('error', 'لا يمكن حذف المكون لأنه مستخدم في رواتب.'); }

        DB::beginTransaction();
        try {
            $name = $salaryComponent->name; // لحفظ الاسم قبل الحذف

            // ✅ حذف المكون
            $salaryComponent->delete();

            // ✅ تسجيل النشاط (اختياري)
            Activity::create([
                'user_id' => auth()->id(),
                'action' => 'deleted_salary_component',
                'model_type' => SalaryComponent::class,
                'model_id' => $salaryComponent->id, // أو اتركه null إذا لم يكن موجودًا بعد الحذف
                'description' => 'تم حذف مكون الراتب: ' . $name,
            ]);

            DB::commit();

            return redirect()->route('salary-components.index')->with('success', 'تم حذف مكون الراتب "' . $name . '" بنجاح.');
        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('SalaryComponent destroy error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'حدث خطأ أثناء حذف مكون الراتب.');
        }
    }
}