<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Payroll;
use App\Models\Employee;
use App\Models\Deduction;
use App\Models\Loan;
use App\Models\Leave;
use App\Models\Activity;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use App\Models\Setting;
use App\Models\SalaryComponent;
use App\Models\LoanInstallment; // ✅ استيراد النموذج المطلوب لدالة printDetails

class PayrollController extends Controller
{
    private function getWeekendDays(): array
    {
        return [5, 6]; // الجمعة والسبت
    }

    /**
     * Resolve an employee's compensation snapshot for a given date.
     * Returns an array with keys: base, incentives, overtime_rate, monthly_leave_entitlement, components (assoc name=>value)
     */
    private function resolveCompensationForDate(Employee $employee, \Carbon\Carbon $date): array
    {
        $history = $employee->getCompensationForDate($date);
        $components = [];
        if ($history) {
            $base = (float) ($history->base_salary ?? 0);
            $incentives = (float) ($history->incentives ?? 0);
            $overtime = (float) ($history->overtime_rate ?? ($employee->overtime_hourly_rate ?? 0));
            $monthlyLeave = $history->monthly_leave_entitlement ?? $employee->monthly_leave_days_allowed ?? 0;
            $annualLeave = $history->annual_leave_entitlement ?? $employee->annual_entitlement ?? 0;
            // history->components is expected to be an assoc array name=>value
            if (is_array($history->components)) {
                $components = $history->components;
            }
        } else {
            $base = (float) ($employee->salary ?? 0);
            $incentives = (float) ($employee->incentives ?? 0);
            $overtime = (float) ($employee->overtime_hourly_rate ?? 0);
            $monthlyLeave = $employee->monthly_leave_days_allowed ?? 0;
            $annualLeave = $employee->annual_entitlement ?? 0;
        }

        // If history did not provide components, fall back to relational and named component fields
        if (empty($components)) {
            // relational components (pivot values)
            try {
                $relComponents = $employee->salaryComponents()->get()->mapWithKeys(function ($c) {
                    $val = $c->pivot->value ?? 0;
                    return [$c->name => (float) $val];
                })->toArray();
            } catch (\Throwable $e) {
                $relComponents = [];
            }
            $components = $relComponents;
            // also include named component_1..7
            for ($i = 1; $i <= 7; $i++) {
                $field = "component_$i";
                $nameField = "component_name_$i";
                $name = $employee->{$nameField} ?? null;
                $value = (float) ($employee->{$field} ?? 0);
                if (!empty($name) && $value != 0) {
                    $components[$name] = $value;
                }
            }
        }

        $componentsSum = array_sum(array_map(fn($v) => (float) $v, $components));

        return [
            'base' => $base,
            'incentives' => $incentives,
            'overtime_rate' => $overtime,
            'monthly_leave_entitlement' => $monthlyLeave,
            'annual_leave_entitlement' => $annualLeave,
            'components' => $components,
            'components_sum' => $componentsSum,
            'gross' => $base + $incentives + $componentsSum,
        ];
    }

    public function index()
    {
        // ✅ التحقق من الصلاحية العامة لعرض قائمة الرواتب
        $this->authorize('viewAny', Payroll::class);

        // ✅ جلب رمز العملة من الإعدادات
        $currency_name = get_currency_code();

        // ✅ التحقق مما إذا كان المستخدم الحالي من دور 'employee'
        $authenticatedUser = auth()->user();
        $isEmployee = $authenticatedUser->hasRole('employee');

        // ✅ تحديد قائمة الموظفين لعرض معلوماتهم
        $query = Employee::where('status', 'active')
            ->with(['salaryComponents' => fn($q) => $q->withPivot('value')])
            ->orderBy('name');

        if ($isEmployee && $authenticatedUser->employee) {
            // إذا كان المستخدم موظفًا، قيّد البحث على موظفه فقط
            $query->where('id', $authenticatedUser->employee->id);
        }

        $employees = $query->get();

        $currentMonth = now()->month;
        $currentYear = now()->year;

        $dateForCurrent = Carbon::create($currentYear, $currentMonth, 1)->startOfMonth();
        foreach ($employees as $employee) {
            $comp = $this->resolveCompensationForDate($employee, $dateForCurrent);

            $employee->computed_basic_salary = $comp['base'];
            $employee->computed_incentives = $comp['incentives'];

            $employee->computed_active_deductions = $this->getActiveDeductionsForPayroll($employee, $currentYear, $currentMonth);
            $employee->computed_loan_installments = $this->getTotalLoanInstallmentsForPayroll($employee, $currentYear, $currentMonth);

            // ✅ استخدام الدالة الجديدة للحصول على خصم الإجازات (شهري + سنوي معاً)
            $leaveDeductionDetails = $employee->calculateLeaveDeductionDetailsNew($currentYear, $currentMonth);
            $leaveDeduction = 0;
            if ($leaveDeductionDetails && count($leaveDeductionDetails) > 0) {
                // أخذ آخر صف (الشهر الحالي) والمجموع الكلي للخصم
                $leaveDeduction = $leaveDeductionDetails->last()['leave_deduction_amount'] ?? 0;
            } else {
                // احتياطي: استخدام الدالة القديمة إذا فشلت الجديدة
                $leaveDeduction = $employee->calculateLeaveDeductionForMonth($currentYear, $currentMonth);
            }
            $employee->computed_leave_deduction = $leaveDeduction;

            $employee->computed_total_salary = $comp['gross'];
            $employee->computed_net_salary = $comp['gross']
                - $employee->computed_active_deductions
                - $employee->computed_loan_installments
                - $employee->computed_leave_deduction;

            $pendingPayroll = $employee->payrolls()
                ->where('month', $currentMonth)
                ->where('year', $currentYear)
                ->where('status', 'pending')
                ->first();

            $employee->has_pending_payroll_for_current_month = $pendingPayroll !== null;
            $employee->pending_payroll_for_current_month = $pendingPayroll;
        }

        // --- حساب الإحصائيات العامة (سيتم عرضها فقط للادمن/HR) ---
        // ✅ حساب عدد جميع الرواتب المدفوعة في النظام (للكارت الأخضر)
        $paidPayrolls = Payroll::where('status', 'paid')->count();

        // ✅ حساب الرواتب المدفوعة *لهذا الشهر فقط* (لجدول "الرواتب المدفوعة لهذا الشهر")
        $paidThisMonthQuery = Payroll::with('employee')
            ->where('month', $currentMonth)
            ->where('year', $currentYear)
            ->where('status', 'paid');

        // ✅ تطبيق الفلتر على $paidThisMonth أيضًا إذا كان المستخدم موظفًا
        if ($isEmployee && $authenticatedUser->employee) {
            $paidThisMonthQuery->where('employee_id', $authenticatedUser->employee->id);
        }

        $paidThisMonth = $paidThisMonthQuery->get();

        $totalPayrolls = Payroll::count();
        $pendingPayrolls = Payroll::where('status', 'pending')->count();
        // ✅ لحساب إجمالي صافي الرواتب المدفوعة فقط (للكارت الأزرق)
        $totalAmountQuery = Payroll::where('paid_at', '!=', null);

        // ✅ تطبيق الفلتر على $totalAmount أيضًا إذا كان المستخدم موظفًا
        if ($isEmployee && $authenticatedUser->employee) {
             $totalAmountQuery->where('employee_id', $authenticatedUser->employee->id);
        }

        $totalAmount = $totalAmountQuery->sum('net_salary');

        // --- تمرير المتغيرات إلى العرض ---
        return view('payroll.index', compact(
            'employees', // ✅ تحتوي فقط على موظف واحد إذا كان المستخدم موظفًا
            'paidThisMonth', // ✅ تحتوي فقط على رواتب الموظف إذا كان المستخدم موظفًا
            'totalPayrolls', // ✅ قد يكون مفيدًا فقط للادمن
            'paidPayrolls', // ✅ قد يكون مفيدًا فقط للادمن
            'pendingPayrolls', // ✅ قد يكون مفيدًا فقط للادمن
            'totalAmount', // ✅ يحتوي على مجموع رواتب الموظف فقط إذا كان المستخدم موظفًا
            'currency_name'
        ));
    }

    public function create(Request $request)
    {
        // authorize generally, but allow an authenticated employee to open the create
        // form for their own payroll even if they don't have broader "create" permission.
        try {
            $this->authorize('create', Payroll::class);
        } catch (\Illuminate\Auth\Access\AuthorizationException $e) {
            // if employee_id is provided and belongs to the current user, allow
            $employeeIdCheck = $request->query('employee_id');
            $user = auth()->user();
            if (!($employeeIdCheck && $user && $user->employee && $user->employee->id == (int) $employeeIdCheck)) {
                throw $e; // rethrow if not a self-access case
            }
            // else: allow to continue (employee viewing their own create form)
        }

        $employeeId = $request->query('employee_id');
        if ($employeeId) {
            $employee = Employee::with([
                'salaryComponents' => fn($q) => $q->withPivot('value'),
            ])->findOrFail($employeeId);

            $salaryComponents = SalaryComponent::all();
            $currency_name = get_currency_code();

            // ✅ حساب القيم المطلوبة للشهر الحالي (أو الشهر المحدد في URL إذا كان موجودًا)
            $selectedYear = $request->query('year', now()->year);
            $selectedMonth = $request->query('month', now()->month);

            $activeDeductions = $this->getActiveDeductionsForPayroll($employee, $selectedYear, $selectedMonth);
            $loanInstallments = $this->getTotalLoanInstallmentsForPayroll($employee, $selectedYear, $selectedMonth);
            
            // ✅ استخدام الدالة الجديدة للحصول على خصم الإجازات (شهري + سنوي معاً)
            $leaveDeductionDetails = $employee->calculateLeaveDeductionDetailsNew($selectedYear, $selectedMonth);
            $leaveDeduction = 0;
            if ($leaveDeductionDetails && count($leaveDeductionDetails) > 0) {
                // ابحث عن البيانات للشهر والسنة المطلوبين بالضبط
                $currentMonthDetails = $leaveDeductionDetails->firstWhere(function($item) use ($selectedYear, $selectedMonth) {
                    return $item['year'] == $selectedYear && $item['month'] == $selectedMonth;
                });
                if ($currentMonthDetails) {
                    $leaveDeduction = $currentMonthDetails['leave_deduction_amount'] ?? 0;
                } else {
                    // احتياطي: استخدام آخر صف إذا لم نجد الشهر المطلوب
                    $leaveDeduction = $leaveDeductionDetails->last()['leave_deduction_amount'] ?? 0;
                }
            } else {
                // احتياطي: استخدام الدالة القديمة إذا فشلت الجديدة
                $leaveDeduction = $employee->calculateLeaveDeductionForMonth($selectedYear, $selectedMonth);
            }

            // ✅ حساب الرصيد المتاح/المتبقي/الأيام الزائدة
            $balanceBeforeDeduction = $employee->getAvailableBalanceAttribute();
            $usedThisMonth = $employee->getUsedLeaveDaysInMonth($selectedYear, $selectedMonth);
            $allowed = $employee->monthly_leave_days_allowed ?? 0;
            $excessDays = max(0, $usedThisMonth - $allowed);
            $balanceAfterDeduction = $balanceBeforeDeduction - $excessDays;

            // حساب اوفرتايم للعرض على صفحة الإنشاء: جمع ساعات الاوفر تايم من الحضور خلال الشهر المحدد
            $startMonth = Carbon::create($selectedYear, $selectedMonth, 1)->startOfMonth();
            $endMonth = (clone $startMonth)->endOfMonth();

            // snapshot of compensation for the payroll month (used by several calculations)
            $compSnapshot = $this->resolveCompensationForDate($employee, $startMonth);

            $attendanceOvertimeHours = $employee->attendances()
                ->whereBetween('date', [$startMonth, $endMonth])
                ->get()
                ->map(fn($a) => $a->getOvertimeHours())
                ->sum();

            // count attendances on weekly-off/holidays that were marked paid
            $paidOffDays = $employee->attendances()
                ->whereBetween('date', [$startMonth, $endMonth])
                ->where('paid_for_off', true)
                ->count();

            $paidOffAmount = 0;
            if ($paidOffDays > 0) {
                // prioritize employee's default_paid_off_amount, then fallback to calculated daily rate
                $dailyRate = 0;
                if (!empty($employee->default_paid_off_amount) && $employee->default_paid_off_amount > 0) {
                    $dailyRate = $employee->default_paid_off_amount;
                } elseif (!empty($compSnapshot['gross']) && $compSnapshot['gross'] > 0) {
                    $dailyRate = ($compSnapshot['gross'] ?? 0) / 30;
                }
                $paidOffAmount = round($paidOffDays * $dailyRate, 2);
            }

            $overtimeAmount = 0;
            $rate = $compSnapshot['overtime_rate'] ?? 0;
            // respect employee's overtime_paid flag
            if (($employee->overtime_paid ?? true) && $attendanceOvertimeHours > 0 && $rate > 0) {
                $overtimeAmount = round($attendanceOvertimeHours * $rate, 2);
            }

                // --- Paid off-days (attendance on weekly-off/holiday marked paid) ---
                $paidOffDays = $employee->attendances()
                    ->whereBetween('date', [$startMonth, $endMonth])
                    ->where('paid_for_off', true)
                    ->count();

                $paidOffAmount = 0;
                if ($paidOffDays > 0) {
                    // prioritize employee's default_paid_off_amount, then fallback to calculated daily rate
                    $dailyRate = 0;
                    if (!empty($employee->default_paid_off_amount) && $employee->default_paid_off_amount > 0) {
                        $dailyRate = $employee->default_paid_off_amount;
                    } elseif (!empty($compSnapshot['gross']) && $compSnapshot['gross'] > 0) {
                        $dailyRate = ($compSnapshot['gross'] ?? 0) / 30;
                    }
                    $paidOffAmount = round($paidOffDays * $dailyRate, 2);
                }

            // حساب نقصان الساعات (underworked) وعرض القيمة في واجهة الإنشاء
            $attendanceUnderworkedHours = $employee->attendances()
                ->whereBetween('date', [$startMonth, $endMonth])
                ->get()
                ->map(fn($a) => method_exists($a, 'getUnderworkedHours') ? $a->getUnderworkedHours() : 0)
                ->sum();

            $underworkedAmount = 0;
            if (($employee->deduct_if_underworked ?? false) && $attendanceUnderworkedHours > 0) {
                $underworkedAmount = round($employee->attendances()
                    ->whereBetween('date', [$startMonth, $endMonth])
                    ->get()
                    ->map(fn($a) => method_exists($a, 'getUnderworkedValue') ? $a->getUnderworkedValue() : 0)
                    ->sum(), 2);
            }

            // حساب مجموع ساعات النقص (underworked) وقيمتها النقدية إذا كان الموظف يخضع للخصم
            $attendanceUnderworkedHours = $employee->attendances()
                ->whereBetween('date', [$startMonth, $endMonth])
                ->get()
                ->map(fn($a) => method_exists($a, 'getUnderworkedHours') ? $a->getUnderworkedHours() : 0)
                ->sum();

            $underworkedAmount = 0;
            if (($employee->deduct_if_underworked ?? false) && $attendanceUnderworkedHours > 0) {
                $underworkedAmount = round($employee->attendances()
                    ->whereBetween('date', [$startMonth, $endMonth])
                    ->get()
                    ->map(fn($a) => method_exists($a, 'getUnderworkedValue') ? $a->getUnderworkedValue() : 0)
                    ->sum(), 2);
            }

            // جلب قائمة الأقساط المعلقة لنفس الشهر — لعرضها في الواجهة مع checkboxes
            $start = Carbon::create($selectedYear, $selectedMonth, 1)->startOfMonth();
            $end = (clone $start)->endOfMonth();
            $pendingInstallments = $employee->loans()
                ->where('status', 'active')
                ->with(['installments' => function ($q) use ($start, $end) {
                    $q->whereBetween('due_date', [$start, $end])->where('status', 'pending');
                }])
                ->get()
                ->pluck('installments')
                ->flatten();

            // ✅ جلب قيم الراتب الأساسي والحوافز والمكونات من سجل التعويض للشهر المحدد
            $compSnapshot = $this->resolveCompensationForDate($employee, $startMonth);
            $defaultBasicSalary = $compSnapshot['base'];
            $defaultIncentives = $compSnapshot['incentives'];
            $defaultComponents = $compSnapshot['components']; // مصفوفة associative name => value

            return view('payroll.create-form', compact(
                'employee',
                'salaryComponents',
                'currency_name',
                'selectedYear', // ✅ تم تغيير الاسم من currentYear إلى selectedYear
                'selectedMonth', // ✅ تم تغيير الاسم من currentMonth إلى selectedMonth
                // ✅ القيم المحسوبة
                'activeDeductions',
                'loanInstallments',
                'leaveDeduction',
                'balanceBeforeDeduction',
                'balanceAfterDeduction',
                'excessDays'
                , 'attendanceOvertimeHours', 'overtimeAmount', 'attendanceUnderworkedHours', 'underworkedAmount'
                , 'pendingInstallments'
                // ✅ قيم الراتب الأساسي والحوافز والمكونات من سجل التعويض
                , 'defaultBasicSalary', 'defaultIncentives', 'defaultComponents'
                , 'paidOffDays', 'paidOffAmount'
            ));
        }

        return $this->index();
    }

       /**
     * ✅ دالة جديدة: عرض جميع رواتب الموظف المسجل دخوله حاليًا.
     */
    public function showEmployeePayrolls()
    {
        // ✅ التحقق من أن المستخدم مسجل الدخول وله علاقة بـ Employee
        $authenticatedUser = auth()->user();
        if (!$authenticatedUser || !$authenticatedUser->employee) {
            abort(403, 'Access Denied. User does not have an associated employee profile.');
        }

        // ✅ التحقق من الصلاحية باستخدام policy على نموذج Payroll
        // نستخدم $authenticatedUser->employee->payrolls() لضمان أنه يعرض فقط رواتب موظفه
        $this->authorize('viewAny', Payroll::class); // هذه.policy يجب أن تسمح لـ employee

        // ✅ جلب رواتب الموظف المرتبط بالمستخدم الحالي فقط
        $employee = $authenticatedUser->employee; // نجلب نموذج Employee المرتبط
        $payrolls = $employee->payrolls()
            ->with('payrollSalaryComponents')
            ->orderByDesc('year')
            ->orderByDesc('month')
            ->paginate(10); // عرض 10 رواتب في كل صفحة

        $currency_name = get_currency_code();

        // ✅ تمرير $employee (الذي هو المستخدم نفسه) إلى العرض
        return view('payroll.employee-payrolls', compact('employee', 'payrolls', 'currency_name'));
    }

    private function getActiveDeductionsForPayroll(Employee $employee, int $year, int $month): float
    {
        $start = Carbon::create($year, $month, 1)->startOfMonth();
        $end = (clone $start)->endOfMonth();
        return $employee->deductions()
            ->where('status', 'applied')
            ->where(function ($q) use ($start, $end) {
                // one-time / single deductions: frequency null/once OR legacy is_monthly=false
                $q->where(function ($one) use ($start, $end) {
                    $one->where(function($qq){ $qq->whereNull('frequency')->orWhere('frequency', 'once')->orWhere('frequency','one_time'); })
                        ->whereBetween('deduction_date', [$start, $end]);
                });

                // monthly recurring deductions
                $q->orWhere(function ($monthly) use ($start) {
                    $monthly->where(function($qq){ $qq->where('frequency','monthly')->orWhere('is_monthly', true); })
                        ->where('deduction_date', '<=', $start)
                        ->where(function ($qq) use ($start) {
                            $qq->whereNull('end_date')->orWhere('end_date', '>=', $start);
                        });
                });

                // yearly recurring deductions: apply if the deduction's month matches the payroll month
                $q->orWhere(function ($yearly) use ($start, $end) {
                    $yearly->where('frequency', 'yearly')
                        ->whereRaw('MONTH(deduction_date) = ?', [$start->month])
                        ->where('deduction_date', '<=', $end)
                        ->where(function ($qq) use ($start) {
                            $qq->whereNull('end_date')->orWhere('end_date', '>=', $start);
                        });
                });
            })
            ->sum('amount');
    }

    private function getTotalLoanInstallmentsForPayroll(Employee $employee, int $year, int $month): float
    {
        $start = Carbon::create($year, $month, 1)->startOfMonth();
        $end = (clone $start)->endOfMonth();

        return $employee->loans()
            ->where('status', 'active')
            ->with(['installments' => function ($q) use ($start, $end) {
                $q->whereBetween('due_date', [$start, $end])->where('status', 'pending');
            }])
            ->get()
            ->pluck('installments')
            ->flatten()
            ->sum('amount');
    }

    public function generateMonthly(Request $request)
    {
        $this->authorize('generateMonthly', Payroll::class);

        // ✅ التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'month' => 'required|integer|min:1|max:12',
            'year' => 'required|integer|min:2000|max:2100',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $month = $request->month;
        $year = $request->year;

        DB::beginTransaction();
        try {
            // ✅ جلب جميع الموظفين النشطين
            $employees = Employee::where('status', 'active')
                ->with(['salaryComponents' => fn($q) => $q->withPivot('value')])
                ->get();

            // ✅ عدد الموظفين النشطين
            $totalEmployees = $employees->count();
            $createdCount = 0; // عدد الرواتب التي تم إنشاؤها

            foreach ($employees as $employee) {
                // ✅ التحقق مما إذا كان هناك راتب معلق أو مدفوع لهذا الشهر والسنة
                $existingPayroll = Payroll::where('employee_id', $employee->id)
                    ->where('month', $month)
                    ->where('year', $year)
                    ->first();

                // ✅ إذا كان هناك راتب موجود، نتجاهله ونستمر مع باقي الموظفين
                if ($existingPayroll) {
                    continue; // 👈 هذا هو التعديل المهم!
                }

                // ✅ حساب القيم
                $startMonth = Carbon::create($year, $month, 1)->startOfMonth();
                $compSnapshot = $this->resolveCompensationForDate($employee, $startMonth);
                $totalSalary = $compSnapshot['gross'];
                $activeDeductions = $this->getActiveDeductionsForPayroll($employee, $year, $month);
                $loanInstallments = $this->getTotalLoanInstallmentsForPayroll($employee, $year, $month);
                
                // ✅ استخدام الدالة الجديدة للحصول على خصم الإجازات (شهري + سنوي معاً)
                $leaveDeductionDetails = $employee->calculateLeaveDeductionDetailsNew($year, $month);
                $leaveDeduction = 0;
                if ($leaveDeductionDetails && count($leaveDeductionDetails) > 0) {
                    // ابحث عن البيانات للشهر والسنة المطلوبين بالضبط
                    $currentMonthDetails = $leaveDeductionDetails->firstWhere(function($item) use ($year, $month) {
                        return $item['year'] == $year && $item['month'] == $month;
                    });
                    if ($currentMonthDetails) {
                        $leaveDeduction = $currentMonthDetails['leave_deduction_amount'] ?? 0;
                    } else {
                        // احتياطي: استخدام آخر صف إذا لم نجد الشهر المطلوب
                        $leaveDeduction = $leaveDeductionDetails->last()['leave_deduction_amount'] ?? 0;
                    }
                } else {
                    // احتياطي: استخدام الدالة القديمة إذا فشلت الجديدة
                    $leaveDeduction = $employee->calculateLeaveDeductionForMonth($year, $month);
                }

                // حساب اوفرتايم: جمع ساعات الاوفر تايم من الحضور خلال الشهر
                $endMonth = (clone $startMonth)->endOfMonth();
                $attendanceOvertimeHours = $employee->attendances()
                    ->whereBetween('date', [$startMonth, $endMonth])
                    ->get()
                    ->map(fn($a) => $a->getOvertimeHours())
                    ->sum();

                // حساب نقصان الساعات: جمع ساعات النقصان من الحضور خلال الشهر
                $attendanceUnderworkedHours = $employee->attendances()
                    ->whereBetween('date', [$startMonth, $endMonth])
                    ->get()
                    ->map(fn($a) => $a->getUnderworkedHours())
                    ->sum();

                $overtimeAmount = 0;
                // respect overtime_paid flag; use snapshot rate
                if (($employee->overtime_paid ?? true) && $attendanceOvertimeHours > 0 && ($compSnapshot['overtime_rate'] ?? 0) > 0) {
                    $overtimeAmount = round($attendanceOvertimeHours * $compSnapshot['overtime_rate'], 2);
                }

                // حساب قيمة خصم نقصان الساعات
                $underworkedAmount = 0;
                if ($attendanceUnderworkedHours > 0) {
                    // Use employee's deduction_hourly_rate if set, otherwise calculate automatically
                    $hourlyRate = $employee->deduction_hourly_rate ?? 0;
                    if ($hourlyRate == 0 && ($compSnapshot['gross'] ?? 0) > 0) {
                        $hourlyRate = round((($compSnapshot['gross'] ?? 0) / 30) / ($employee->scheduled_hours ?? 8), 4);
                    }
                    $underworkedAmount = round($attendanceUnderworkedHours * $hourlyRate, 2);
                }

                // Count attendances on weekly-off/holidays that were marked paid (paid-for-off)
                $paidOffDays = $employee->attendances()
                    ->whereBetween('date', [$startMonth, $endMonth])
                    ->where('paid_for_off', true)
                    ->count();

                $paidOffAmount = 0;
                if ($paidOffDays > 0) {
                    // prioritize employee's default_paid_off_amount, then fallback to calculated daily rate
                    $dailyRate = 0;
                    if (!empty($employee->default_paid_off_amount) && $employee->default_paid_off_amount > 0) {
                        $dailyRate = $employee->default_paid_off_amount;
                    } elseif (!empty($compSnapshot['gross']) && $compSnapshot['gross'] > 0) {
                        $dailyRate = ($compSnapshot['gross'] ?? 0) / 30;
                    }
                    $paidOffAmount = round($paidOffDays * $dailyRate, 2);
                }

                // ✅ حساب القيم قبل الحفظ (من الـ snapshot)
                $grossSalary = $compSnapshot['gross'];
                $netSalary = $grossSalary - $activeDeductions - $loanInstallments - $leaveDeduction;
                $netSalary = max(0, $netSalary); // تأكد أن لا يقل عن 0

                // ✅ إنشاء الراتب - يستخدم قيم سجل التعويض للشهر المحدد
                // ملاحظة مهمة: الرواتب القديمة (التي تم إنشاؤها سابقاً) تحتفظ بقيمها الأصلية
                // ولا تتأثر بسجلات التعويض الجديدة - كل راتب يحتفظ بقيمته وقت إنشائه
                $payroll = Payroll::create([
                    'employee_id' => $employee->id,
                    'month' => $month,
                    'year' => $year,
                    'basic_salary' => $compSnapshot['base'], // من سجل التعويض للشهر المحدد
                    'incentives' => $compSnapshot['incentives'] ?? 0, // من سجل التعويض للشهر المحدد
                    'active_deductions' => $activeDeductions,
                    'loan_installments' => $loanInstallments,
                    'leave_deduction' => $leaveDeduction,
                    'total_salary' => $grossSalary,
                    'net_salary' => $netSalary,
                    'status' => 'pending',
                ]);

                // حفظ مكون الاوفر تايم إن وُجد
                if ($overtimeAmount > 0) {
                    // احفظ اوفرتايم مع تفاصيل الحساب (ساعات × سعر)
                    $meta = [
                        'type' => 'overtime',
                        'hours' => $attendanceOvertimeHours,
                        'hourly_rate' => $compSnapshot['overtime_rate'] ?? 0,
                        'formula' => "hours × hourly_rate",
                        'calculation' => round($attendanceOvertimeHours * ($compSnapshot['overtime_rate'] ?? 0), 2),
                    ];

                    $payroll->payrollSalaryComponents()->create([
                        'salary_component_id' => null,
                        'name' => 'Overtime',
                        'value' => $overtimeAmount,
                        'meta' => $meta,
                    ]);
                }

                // حفظ خصم نقصان الساعات (إذا وُجد)
                if ($underworkedAmount > 0) {
                    $uwComponent = SalaryComponent::firstOrCreate(
                        ['name' => 'Underworked Deduction (خصم نقصان ساعات)'],
                        ['type' => 'deduction', 'description' => 'Deduction for hours worked less than scheduled']
                    );

                    // Use employee's deduction_hourly_rate if set, otherwise calculate automatically
                    $hourlyRate = $employee->deduction_hourly_rate ?? 0;
                    if ($hourlyRate == 0 && ($compSnapshot['gross'] ?? 0) > 0) {
                        $hourlyRate = round((($compSnapshot['gross'] ?? 0) / 30) / ($employee->scheduled_hours ?? 8), 4);
                    }
                    
                    $meta = [
                        'type' => 'underworked',
                        'hours' => $attendanceUnderworkedHours,
                        'hourly_rate' => $hourlyRate,
                        'formula' => 'underworked_hours × hourly_rate',
                        'calculation' => round($underworkedAmount, 2),
                    ];

                    $payroll->payrollSalaryComponents()->create([
                        'salary_component_id' => $uwComponent->id,
                        'name' => $uwComponent->name,
                        'value' => -1 * $underworkedAmount,
                        'meta' => $meta,
                    ]);
                }

                // ✅ حفظ مكونات الراتب: إذا كانت هناك مكونات محفوظة في الـ snapshot فَضعها، وإلا استخدم العلاقة الحالية
                if (!empty($compSnapshot['components'])) {
                    foreach ($compSnapshot['components'] as $name => $value) {
                        $payroll->payrollSalaryComponents()->create([
                            'salary_component_id' => null,
                            'name' => $name,
                            'value' => (float) $value,
                        ]);
                    }
                } else {
                    foreach ($employee->salaryComponents as $comp) {
                        $payroll->payrollSalaryComponents()->create([
                            'salary_component_id' => $comp->id,
                            'name' => $comp->name,
                            'value' => $comp->pivot->value ?? 0,
                        ]);
                    }
                }

                // Add Paid Off-day Compensation component for generated payrolls if applicable
                if (!empty($paidOffAmount) && is_numeric($paidOffAmount) && (float)$paidOffAmount > 0) {
                    // Check if this component already exists to prevent duplicates
                    $existingComponent = $payroll->payrollSalaryComponents()
                        ->where('name', 'Paid Off-day Compensation')
                        ->first();
                    
                    if (!$existingComponent) {
                        $payroll->payrollSalaryComponents()->create([
                            'salary_component_id' => null,
                            'name' => 'Paid Off-day Compensation',
                            'value' => (float) $paidOffAmount,
                            'meta' => ['type' => 'paid_off_days', 'days' => $paidOffDays ?? null],
                        ]);
                    } else {
                        // Update existing component with new value if needed
                        if ($existingComponent->value != (float) $paidOffAmount) {
                            $existingComponent->value = (float) $paidOffAmount;
                            $existingComponent->meta = ['type' => 'paid_off_days', 'days' => $paidOffDays ?? null];
                            $existingComponent->save();
                        }
                    }
                }

                // معالجة أيام الإجازة بدون راتب المعتمدة في نفس الفترة (تُخصم من الراتب)
                $start = Carbon::create($year, $month, 1)->startOfMonth();
                $end = (clone $start)->endOfMonth();
                // include legacy unpaid leaves (type 'unpaid') and mixed single-record leaves
                $unpaidLeaves = Leave::where('employee_id', $employee->id)
                    ->where('status', 'approved')
                    ->whereBetween('approved_at', [$start, $end])
                    ->where(function($q) {
                        $q->whereHas('leaveTypeModel', function($qq){ $qq->where('code', 'unpaid'); })
                          ->orWhere('unpaid_deducted_days', '>', 0);
                    })
                    ->where(function($q){
                        $q->whereNull('is_unpaid_deduction_applied')->orWhere('is_unpaid_deduction_applied', false);
                    })
                    ->get();

                // sum both existing deducted_days (legacy unpaid leaves) and unpaid_deducted_days (mixed leaves)
                $unpaidDays = $unpaidLeaves->sum(function($l){
                    return (int)($l->deducted_days ?? 0) + (int)($l->unpaid_deducted_days ?? 0);
                });
                if ($unpaidDays > 0) {
                    $dailyRate = ($compSnapshot['gross'] ?? $employee->getTotalSalaryWithComponentsAttribute()) / 30;
                    $unpaidDeductionAmount = round($unpaidDays * $dailyRate, 2);

                    // نستخدم قيمة سالبة لتكون خصمًا في المكونات
                    $payroll->payrollSalaryComponents()->create([
                        'salary_component_id' => null,
                        'name' => 'خصم إجازة بدون راتب',
                        'value' => -1 * $unpaidDeductionAmount,
                    ]);

                    // علّم الإجازات بأنها مُطبّقة للخصم
                    foreach ($unpaidLeaves as $ul) {
                        $ul->is_unpaid_deduction_applied = true;
                        $ul->deduction_date = now()->toDateString();
                        $ul->save();
                    }
                }

                // ✅ إعادة الحساب بعد حفظ المكونات
                $payroll->calculateTotalSalary();
                $payroll->calculateNetSalary();
                $payroll->save();

                // ✅ تسجيل النشاط
                Activity::create([
                    'user_id' => auth()->id(),
                    'action' => 'created_payroll',
                    'model_type' => Payroll::class,
                    'model_id' => $payroll->id,
                    'description' => 'created_payroll',
                    'properties' => [
                        'employee_id' => $employee->id,
                        'employee' => $employee->name,
                        'month' => $month,
                        'year' => $year,
                        'net_salary' => $payroll->net_salary,
                    ],
                ]);

                // ✅ إرسال إشعار لجميع المديرين والمسؤولين
                $managers = \App\Models\User::whereHas('roles', function ($query) {
                    $query->whereIn('name', ['admin', 'manager', 'hr']);
                })->get();

                foreach ($managers as $manager) {
                    $manager->notify(new \App\Notifications\PayrollCreatedNotification($payroll));
                }

                $createdCount++;
            }

            DB::commit();

            // ✅ إرسال رسالة نجاح
            return redirect()->route('payroll.index')->with('success', "تم إنشاء رواتب لـ {$createdCount} موظف من أصل {$totalEmployees} موظف نشط بنجاح.");

        } catch (\Throwable $e) {
            DB::rollBack();
            // ✅ تسجيل الخطأ في سجلات النظام
            Log::error('generateMonthly error: ' . $e->getMessage());
            // ✅ إرسال رسالة خطأ للمستخدم
            return redirect()->back()->with('error', 'حدث خطأ أثناء إنشاء الرواتب الشهرية. يرجى مراجعة سجلات النظام.');
        }
    }

    // ✅ الدالة المعدلة والمضمونة — تُرجع الحسابات بدقة
    public function getCalculations(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|exists:employees,id',
            'month' => 'required|integer|min:1|max:12',
            'year' => 'required|integer|min:2000|max:2100',
            'installment_ids' => 'nullable|array',
            'installment_ids.*' => 'integer|exists:loan_installments,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'بيانات غير صالحة.'
            ], 422);
        }

        try {
            $employee = Employee::findOrFail($request->employee_id);
            $month = (int) $request->month;
            $year = (int) $request->year;

            // ✅ جلب تفاصيل الشهر المطلوب من الدالة الصحيحة
            $details = $employee->calculateLeaveDeductionDetailsNew($year, $month);
            $current = $details->where('month', $month)->where('year', $year)->first();

            if (!$current) {
                // fallback
                $balanceBefore = 0.0;
                $balanceAfter = 0.0;
                $used = 0.0;
                $excess = 0.0;
                $deduction = 0.0;
            } else {
                $balanceBefore = $current['balance_before_deduction'];
                $balanceAfter = $current['balance_after_deduction'];
                $used = $current['used_this_month'];
                $excess = $current['excess_days'];
                $deduction = $current['leave_deduction_amount'];
            }

            $activeDeductions = $this->getActiveDeductionsForPayroll($employee, $year, $month);
            $loanInstallments = $this->getTotalLoanInstallmentsForPayroll($employee, $year, $month);

            // إذا أرسل المستخدم أقساط محددة ليُضمّنها في الحساب (مثلاً دفع أقساط إضافية)
            $extraInstallments = 0;
            if ($request->filled('installment_ids') && is_array($request->installment_ids)) {
                $ids = array_filter($request->installment_ids, fn($v) => is_numeric($v));
                if (!empty($ids)) {
                    $extraInstallments = (float) \App\Models\LoanInstallment::whereIn('id', $ids)
                        ->where('status', 'pending')
                        ->sum('amount');
                }
            }

            $loanInstallments += $extraInstallments;

            $start = Carbon::create($year, $month, 1)->startOfMonth();
            $compSnapshot = $this->resolveCompensationForDate($employee, $start);
            $totalSalary = $compSnapshot['gross'];
            $dailyRate = $totalSalary > 0 ? $totalSalary / 30 : 0;
            // حساب نقصان الساعات لهذا الموظف في الشهر (للعرض الحي في صفحة الإنشاء)
            $end = (clone $start)->endOfMonth();
            $attendanceUnderworkedHours = $employee->attendances()
                ->whereBetween('date', [$start, $end])
                ->get()
                ->map(fn($a) => method_exists($a, 'getUnderworkedHours') ? $a->getUnderworkedHours() : 0)
                ->sum();

            $underworkedAmount = 0;
            if (($employee->deduct_if_underworked ?? false) && $attendanceUnderworkedHours > 0) {
                $underworkedAmount = round($employee->attendances()
                    ->whereBetween('date', [$start, $end])
                    ->get()
                    ->map(fn($a) => method_exists($a, 'getUnderworkedValue') ? $a->getUnderworkedValue() : 0)
                    ->sum(), 2);
            }
            
            // ✅ إرجاع قيم الراتب الأساسي والحوافز والمكونات من سجل التعويض
            return response()->json([
                'success' => true,
                'calculations' => [
                    'active_deductions' => round($activeDeductions, 2),
                    'loan_installments' => round($loanInstallments, 2),
                    'leave_deduction' => round($deduction, 2),
                    'balance_before_deduction' => (float) $balanceBefore, // ✅ 10.00 يوم
                    'balance_after_deduction' => (float) $balanceAfter,   // ✅ 1.00 يوم
                    'used_days_in_month' => (float) $used,
                    'excess_days' => (float) $excess,                     // ✅ 9.00 يوم فقط
                    'daily_rate' => round($dailyRate, 2),
                    'attendance_underworked_hours' => $attendanceUnderworkedHours,
                    'attendance_underworked_deduction' => round($underworkedAmount, 2),
                    // ✅ إضافة قيم الراتب الأساسي والحوافز والمكونات من سجل التعويض
                    'basic_salary' => round($compSnapshot['base'], 2),
                    'incentives' => round($compSnapshot['incentives'], 2),
                    'components' => $compSnapshot['components'], // مصفوفة associative name => value
                ],
            ]);
        } catch (\Exception $e) {
            Log::error('getCalculations failed for emp=' . ($request->employee_id ?? 'null') . ', month=' . ($request->month ?? 'null') . ', year=' . ($request->year ?? 'null') . ': ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'خطأ داخلي: ' . substr($e->getMessage(), 0, 150)
            ], 500);
        }
    }

    public function store(Request $request)
    {
        // authorize generally, but allow an authenticated employee to create their own payroll
        try {
            $this->authorize('create', Payroll::class);
        } catch (\Illuminate\Auth\Access\AuthorizationException $e) {
            $employeeIdCheck = $request->input('employee_id') ?? $request->query('employee_id');
            $user = auth()->user();
            if (!($employeeIdCheck && $user && $user->employee && $user->employee->id == (int) $employeeIdCheck)) {
                throw $e;
            }
            // else: allow creating payroll for own employee record
        }

        // Log incoming request for debugging when create doesn't complete
        try {
            Log::info('Payroll.store called', ['user_id' => auth()->id(), 'input' => $request->all()]);
        } catch (\Throwable $e) {
            // ignore logging failures
        }

        // Temporary debug hook: if caller passes debug=1 (post or query)
        // return a JSON dump so we can confirm the request reached the server
        // without proceeding to create a payroll. Remove this in production.
        if (($request->input('debug') ?? $request->query('debug')) == '1') {
            return response()->json([
                'received' => true,
                'user_id' => auth()->id(),
                'user_employee_id' => optional(auth()->user()?->employee)->id,
                'input' => $request->all(),
            ], 200);
        }
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|exists:employees,id',
            'month' => 'required|integer|min:1|max:12',
            'year' => 'required|integer|min:2000|max:2100',
            'basic_salary' => 'required|numeric|min:0',
            'incentives' => 'nullable|numeric|min:0',
            'paid_off_amount' => 'nullable|numeric|min:0',
            'salary_component_names' => 'array|max:7',
            'salary_component_names.*' => 'string|max:255|nullable',
            'salary_component_values' => 'array|max:7',
            'salary_component_values.*' => 'nullable|numeric|min:0',
            'salary_component_ids' => 'array|max:7',
            'salary_component_ids.*' => 'nullable|integer|exists:salary_components,id',
            'installment_ids' => 'nullable|array',
            'installment_ids.*' => 'integer|exists:loan_installments,id',
        ]);

        if ($validator->fails()) {
            // Log validation errors for debugging
            try {
                Log::warning('Payroll.store validation failed', ['user_id' => auth()->id(), 'errors' => $validator->errors()->toArray()]);
            } catch (\Throwable $e) {
                // ignore
            }
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $data = $validator->validated();

        if (Payroll::where('employee_id', $data['employee_id'])
            ->where('month', $data['month'])
            ->where('year', $data['year'])
            ->exists()) {
            return redirect()->back()->with('error', 'لقد تم إنشاء راتب لهذا الموظف في هذا الشهر والسنة بالفعل.');
        }

        DB::beginTransaction();
        try {
            $employee = Employee::with(['salaryComponents' => fn($q) => $q->withPivot('value')])
                ->findOrFail($data['employee_id']);

            // ✅ جلب القيم المطلوبة من قاعدة البيانات عند الحفظ
            $activeDeductions = $this->getActiveDeductionsForPayroll($employee, $data['year'], $data['month']);
            $loanInstallments = $this->getTotalLoanInstallmentsForPayroll($employee, $data['year'], $data['month']);

            // إذا تم اختيار أقساط معينة لضمها (دفع إضافي من الراتب)، اجمع قيمها
            $extraInstallments = 0;
            $selectedInstallmentIds = [];
            if (!empty($data['installment_ids']) && is_array($data['installment_ids'])) {
                $selectedInstallmentIds = array_filter($data['installment_ids'], fn($v) => is_numeric($v));
                if (!empty($selectedInstallmentIds)) {
                    $extraInstallments = (float) LoanInstallment::whereIn('id', $selectedInstallmentIds)
                        ->where('status', 'pending')
                        ->sum('amount');
                }
            }
            $loanInstallments += $extraInstallments;
            
            // ✅ استخدام الدالة الجديدة للحصول على خصم الإجازات (شهري + سنوي معاً)
            $leaveDeductionDetails = $employee->calculateLeaveDeductionDetailsNew($data['year'], $data['month']);
            $leaveDeduction = 0;
            if ($leaveDeductionDetails && count($leaveDeductionDetails) > 0) {
                // ابحث عن البيانات للشهر والسنة المطلوبين بالضبط
                $currentMonthDetails = $leaveDeductionDetails->firstWhere(function($item) use ($data) {
                    return $item['year'] == $data['year'] && $item['month'] == $data['month'];
                });
                if ($currentMonthDetails) {
                    $leaveDeduction = $currentMonthDetails['leave_deduction_amount'] ?? 0;
                } else {
                    // احتياطي: استخدام آخر صف إذا لم نجد الشهر المطلوب
                    $leaveDeduction = $leaveDeductionDetails->last()['leave_deduction_amount'] ?? 0;
                }
            } else {
                // احتياطي: استخدام الدالة القديمة إذا فشلت الجديدة
                $leaveDeduction = $employee->calculateLeaveDeductionForMonth($data['year'], $data['month']);
            }

            // حساب اوفرتايم للحفظ اليدوي (sum from attendances in the payroll month)
            $start = Carbon::create($data['year'], $data['month'], 1)->startOfMonth();
            $end = (clone $start)->endOfMonth();
            $attendanceOvertimeHours = $employee->attendances()
                ->whereBetween('date', [$start, $end])
                ->get()
                ->map(fn($a) => $a->getOvertimeHours())
                ->sum();

            $compSnapshot = $this->resolveCompensationForDate($employee, $start);
            $overtimeAmount = 0;
            if (($employee->overtime_paid ?? true) && $attendanceOvertimeHours > 0 && ($compSnapshot['overtime_rate'] ?? 0) > 0) {
                $overtimeAmount = round($attendanceOvertimeHours * $compSnapshot['overtime_rate'], 2);
            }

            // ✅ حساب القيم قبل الحفظ لتجنب خطأ "Field doesn't have a default value"
            $grossSalary = $data['basic_salary'] + ($data['incentives'] ?? 0);
            $netSalary = $grossSalary - $activeDeductions - $loanInstallments - $leaveDeduction;
            $netSalary = max(0, $netSalary); // تأكد أن لا يقل عن 0

            $payroll = Payroll::create([
                'employee_id' => $employee->id,
                'month' => $data['month'],
                'year' => $data['year'],
                'basic_salary' => $data['basic_salary'],
                'incentives' => $data['incentives'] ?? 0,
                'active_deductions' => $activeDeductions,
                'loan_installments' => $loanInstallments,
                'leave_deduction' => $leaveDeduction,
                'total_salary' => $grossSalary, // ✅ تم تمرير القيمة
                'net_salary' => $netSalary,      // ✅ تم تمرير القيمة
                'status' => 'pending',
            ]);

            // حفظ معرفات الأقساط المختارة في حقل meta (JSON) ليتم استخدامها لاحقًا عند الدفع
            if (!empty($selectedInstallmentIds)) {
                $meta = $payroll->meta ?? [];
                if (!is_array($meta)) {
                    $meta = is_string($meta) ? json_decode($meta, true) ?? [] : [];
                }
                $meta['installment_ids'] = array_values($selectedInstallmentIds);
                $payroll->meta = $meta;
                $payroll->save();
            }

            // حفظ اوفرتايم كمكون إن وجد
            if ($overtimeAmount > 0) {
                // احفظ اوفرتايم مع تفاصيل الحساب (ساعات × سعر)
                $meta = [
                    'type' => 'overtime',
                    'hours' => $attendanceOvertimeHours,
                    'hourly_rate' => $compSnapshot['overtime_rate'] ?? 0,
                    'formula' => "hours × hourly_rate",
                    'calculation' => round($attendanceOvertimeHours * ($compSnapshot['overtime_rate'] ?? ($employee->overtime_hourly_rate ?? 0)), 2),
                ];

                $payroll->payrollSalaryComponents()->create([
                    'salary_component_id' => null,
                    'name' => 'Overtime',
                    'value' => $overtimeAmount,
                    'meta' => $meta,
                ]);
            }

            // حفظ خصم نقصان الساعات (إذا وُجد) عند الحفظ اليدوي
            // تم حساب $start/$end أعلاه
            $attendanceUnderworkedHours = $employee->attendances()
                ->whereBetween('date', [$start, $end])
                ->get()
                ->map(fn($a) => method_exists($a, 'getUnderworkedHours') ? $a->getUnderworkedHours() : 0)
                ->sum();

            $underworkedAmount = 0;
            if (($employee->deduct_if_underworked ?? false) && $attendanceUnderworkedHours > 0) {
                $underworkedAmount = round($employee->attendances()
                    ->whereBetween('date', [$start, $end])
                    ->get()
                    ->map(fn($a) => method_exists($a, 'getUnderworkedValue') ? $a->getUnderworkedValue() : 0)
                    ->sum(), 2);
            }

            if ($underworkedAmount > 0) {
                $uwComponent = SalaryComponent::firstOrCreate(
                    ['name' => 'Underworked Deduction (خصم نقصان ساعات)'],
                    ['type' => 'deduction', 'description' => 'Deduction for hours worked less than scheduled']
                );

                // Use employee's deduction_hourly_rate if set, otherwise calculate automatically
                $hourlyRate = $employee->deduction_hourly_rate ?? 0;
                if ($hourlyRate == 0 && ($compSnapshot['gross'] ?? 0) > 0) {
                    $hourlyRate = round((($compSnapshot['gross'] ?? 0) / 30) / ($employee->scheduled_hours ?? 8), 4);
                }
                
                $meta = [
                    'type' => 'underworked',
                    'hours' => $attendanceUnderworkedHours,
                    'hourly_rate' => $hourlyRate,
                    'formula' => 'underworked_hours × hourly_rate',
                    'calculation' => round($underworkedAmount, 2),
                ];

                $payroll->payrollSalaryComponents()->create([
                    'salary_component_id' => $uwComponent->id,
                    'name' => $uwComponent->name,
                    'value' => -1 * $underworkedAmount,
                    'meta' => $meta,
                ]);
            }

            if (isset($data['salary_component_names']) && isset($data['salary_component_values'])) {
                foreach ($data['salary_component_names'] as $index => $name) {
                    if (empty($name) || !isset($data['salary_component_values'][$index])) continue;
                    $componentId = $data['salary_component_ids'][$index] ?? null;
                    $payroll->payrollSalaryComponents()->create([
                        'salary_component_id' => $componentId,
                        'name' => $name,
                        'value' => $data['salary_component_values'][$index],
                    ]);
                }
            }

                // إضافة مكوّن قيمة أيام الحضور المدفوعة إن وُجد
            if (!empty($data['paid_off_amount']) && is_numeric($data['paid_off_amount']) && (float)$data['paid_off_amount'] > 0) {
                // Check if this component already exists to prevent duplicates
                $existingComponent = $payroll->payrollSalaryComponents()
                    ->where('name', 'Paid Off-day Compensation')
                    ->first();
                
                if (!$existingComponent) {
                    $payroll->payrollSalaryComponents()->create([
                        'salary_component_id' => null,
                        'name' => 'Paid Off-day Compensation',
                        'value' => (float) $data['paid_off_amount'],
                        'meta' => ['type' => 'paid_off_days', 'days' => $paidOffDays ?? null],
                    ]);
                } else {
                    // Update existing component with new value if needed
                    if ($existingComponent->value != (float) $data['paid_off_amount']) {
                        $existingComponent->value = (float) $data['paid_off_amount'];
                        $existingComponent->meta = ['type' => 'paid_off_days', 'days' => $paidOffDays ?? null];
                        $existingComponent->save();
                    }
                }
            }

            // معالجة أيام الإجازة بدون راتب المعتمدة في نفس الفترة (تُخصم من الراتب)
            $start = Carbon::create($data['year'], $data['month'], 1)->startOfMonth();
            $end = (clone $start)->endOfMonth();
            // include legacy unpaid leaves (type 'unpaid') and mixed single-record leaves
            $unpaidLeaves = Leave::where('employee_id', $employee->id)
                ->where('status', 'approved')
                ->whereBetween('approved_at', [$start, $end])
                ->where(function($q) {
                    $q->whereHas('leaveTypeModel', function($qq){ $qq->where('code', 'unpaid'); })
                      ->orWhere('unpaid_deducted_days', '>', 0);
                })
                ->where(function($q){
                    $q->whereNull('is_unpaid_deduction_applied')->orWhere('is_unpaid_deduction_applied', false);
                })
                ->get();

            // sum both existing deducted_days (legacy unpaid leaves) and unpaid_deducted_days (mixed leaves)
            $unpaidDays = $unpaidLeaves->sum(function($l){
                return (int)($l->deducted_days ?? 0) + (int)($l->unpaid_deducted_days ?? 0);
            });
            if ($unpaidDays > 0) {
                    $dailyRate = ($compSnapshot['gross'] ?? $employee->getTotalSalaryWithComponentsAttribute()) / 30;
                $unpaidDeductionAmount = round($unpaidDays * $dailyRate, 2);

                $payroll->payrollSalaryComponents()->create([
                    'salary_component_id' => null,
                    'name' => 'خصم إجازة بدون راتب',
                    'value' => -1 * $unpaidDeductionAmount,
                ]);

                foreach ($unpaidLeaves as $ul) {
                    $ul->is_unpaid_deduction_applied = true;
                    $ul->deduction_date = now()->toDateString();
                    $ul->save();
                }
            }

            // ✅ إعادة الحساب بعد حفظ المكونات (لإضافة مكونات الراتب إلى total_salary)
            $payroll->calculateTotalSalary(); // تُضيف مكونات الراتب (التي تم حفظها مع هذا الراتب) إلى total_salary
            $payroll->calculateNetSalary();   // تُعيد حساب net_salary بناءً على total_salary الجديد
            $payroll->save();                 // حفظ القيم المُحدثة

            Activity::create([
                'user_id' => auth()->id(),
                'action' => 'created_payroll',
                'model_type' => Payroll::class,
                'model_id' => $payroll->id,
                'description' => 'created_payroll',
                'properties' => [
                    'employee_id' => $employee->id,
                    'employee' => $employee->name,
                    'month' => $data['month'],
                    'year' => $data['year'],
                    'net_salary' => $payroll->net_salary,
                ],
            ]);

            DB::commit();
            return redirect()->route('payroll.index')->with('success', 'تم إنشاء راتب الموظف بنجاح.');
        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('Payroll store error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'حدث خطأ أثناء إنشاء الراتب.')->withInput();
        }
    }

     public function show(Payroll $payroll)
        {
            $this->authorize('view', $payroll);
            $payroll->load(['employee', 'payrollSalaryComponents']);

            // ✅ حساب القيم الأساسية
            $usedDays = $payroll->employee->getUsedLeaveDaysInMonth($payroll->year, $payroll->month);
            $allowed = $payroll->employee->monthly_leave_days_allowed ?? 0;
            $excess = max(0, $usedDays - $allowed);
            // Prefer the payroll's total_salary (snapshot) for daily rate and leave deduction
            $startMonth = Carbon::create($payroll->year, $payroll->month, 1)->startOfMonth();
            $compSnapshot = $this->resolveCompensationForDate($payroll->employee, $startMonth);
            $dailyRate = $payroll->total_salary > 0 ? ($payroll->total_salary / 30) : (($compSnapshot['gross'] ?? $payroll->employee->getTotalSalaryWithComponentsAttribute()) / 30);
            $leaveDeduction = $excess * $dailyRate;

            // ✅ جلب الاستقطاعات النشطة لهذا الموظف في نفس شهر وسنة هذا الراتب
            $actualDeductions = $payroll->employee->deductions()
                ->where('status', 'applied')
                ->where(function ($q) use ($payroll) {
                    $start = Carbon::create($payroll->year, $payroll->month, 1)->startOfMonth();
                    $end = (clone $start)->endOfMonth();

                    // one-time
                    $q->where(function ($one) use ($start, $end) {
                        $one->where(function($qq){ $qq->whereNull('frequency')->orWhere('frequency','once')->orWhere('frequency','one_time'); })
                            ->whereBetween('deduction_date', [$start, $end]);
                    });

                    // monthly
                    $q->orWhere(function ($monthly) use ($start) {
                        $monthly->where(function($qq){ $qq->where('frequency','monthly')->orWhere('is_monthly', true); })
                            ->where('deduction_date', '<=', $start)
                            ->where(function ($qq) use ($start) {
                                $qq->whereNull('end_date')->orWhere('end_date', '>=', $start);
                            });
                    });

                    // yearly
                    $q->orWhere(function ($yearly) use ($start, $end) {
                        $yearly->where('frequency', 'yearly')
                            ->whereRaw('MONTH(deduction_date) = ?', [$start->month])
                            ->where('deduction_date', '<=', $end)
                            ->where(function ($qq) use ($start) {
                                $qq->whereNull('end_date')->orWhere('end_date', '>=', $start);
                            });
                    });
                })
                ->get()
                ->map(function ($d) {
                    // ✅ تحويل نوع الاستقطاع إلى اللغة العربية
                    $d->type_arabic = [
                        'tax' => 'ضريبة',
                        'insurance' => 'تأمين',
                        'loan' => 'قرض',
                        'fine' => 'غرامة',
                        'other' => 'أخرى',
                        'leave' => 'إجازة',
                        'health_insurance' => 'تأمين صحي',
                        'social_insurance' => 'تأمين اجتماعي',
                        'penalties' => 'جزاءات',
                        'deductions' => 'خصومات',
                        'loans' => 'سلف',
                        'subscriptions' => 'اشتراكات',
                        'other_deductions' => 'استقطاعات أخرى',
                        'cash' => 'نقدي',
                        'in_kind' => 'عيني',
                        'deduction' => 'خصم',
                        'penalty' => 'جزاء',
                        'subscription' => 'اشتراك',
                        'absence' => 'غياب',
                    ][$d->type] ?? $d->type;

                    return $d;
                });

            // ✅ جلب أقساط القروض لهذا الموظف في نفس شهر وسنة هذا الراتب
            $loanInstallments = $payroll->employee->loans()
                ->where('status', 'active')
                ->with(['installments' => function ($q) use ($payroll) {
                    $start = Carbon::create($payroll->year, $payroll->month, 1)->startOfMonth();
                    $end = (clone $start)->endOfMonth();

                    $q->whereBetween('due_date', [$start, $end])
                      ->where('status', 'pending');
                }])
                ->get()
                ->pluck('installments')
                ->flatten()
                ->map(function ($inst) {
                    // ✅ تحويل نوع القرض إلى اللغة العربية
                    $inst->loan->loan_type_arabic = [
                        'personal' => 'شخصي',
                        'car' => 'سيارة',
                        'home' => 'سكن',
                        'education' => 'تعليم',
                        'medical' => 'طبي',
                        'emergency' => 'طوارئ',
                        'business' => 'تجاري',
                        'other' => 'أخرى',
                    ][$inst->loan->loan_type] ?? $inst->loan->loan_type;

                    return $inst;
                });

            // ✅ جلب الرواتب السابقة (10 رواتب أخيرة) لنفس الموظف مع تحميل مكونات الراتب
            $previousPayrolls = $payroll->employee->payrolls()
                ->where('id', '!=', $payroll->id) // استبعاد الراتب الحالي
                ->with('payrollSalaryComponents')
                ->orderByDesc('year')
                ->orderByDesc('month')
                ->limit(10)
                ->get();

            // ✅ جلب معلومات الإجازات (استخدم نفس الدالة التي في create-form)
            $leaveDetails = $payroll->employee->calculateLeaveDeductionDetailsNew($payroll->year, $payroll->month);
            $currentMonthDetails = $leaveDetails->firstWhere(function ($detail) use ($payroll) {
                return $detail['month'] == $payroll->month && $detail['year'] == $payroll->year;
            });

            $cumulativeLeaveInfo = [
                'current_month_used' => $currentMonthDetails['used_this_month'] ?? 0,
                'current_month_excess' => $currentMonthDetails['excess_days'] ?? 0,
            ];

            $currency_name = get_currency_code();

            // ✅ حساب اوفرتايم الشهري (ساعات وقيمة) للعرض في صفحة العرض
            $startMonth = Carbon::create($payroll->year, $payroll->month, 1)->startOfMonth();
            $endMonth = (clone $startMonth)->endOfMonth();

            // جلب تفاصيل الحضور خلال الشهر
            $attendances = $payroll->employee->attendances()
                ->whereBetween('date', [$startMonth, $endMonth])
                ->get();

            // حساب معدل الأوفر تايم بالساعة بناءً على snapshot التعويضات
            $compSnapshot = $this->resolveCompensationForDate($payroll->employee, $startMonth);
            $rate = $compSnapshot['overtime_rate'] ?? $payroll->employee->overtime_hourly_rate ?? 0;

            // Use employee's deduction_hourly_rate for underworked calculations
            $deductionRate = $payroll->employee->deduction_hourly_rate ?? 0;
            if ($deductionRate == 0 && ($compSnapshot['gross'] ?? 0) > 0) {
                $deductionRate = round((($compSnapshot['gross'] ?? 0) / 30) / ($payroll->employee->scheduled_hours ?? 8), 4);
            }

            // تفاصيل الأيام: نضيف ساعات النقص، قيمة الخصم، وساعات الأوفر تايم وقيمتها لكل يوم
            $attendanceDetails = $attendances->map(function ($attendance) use ($rate, $deductionRate) {
                $overtimeHours = method_exists($attendance, 'getOvertimeHours') ? $attendance->getOvertimeHours() : 0;
                $underworkedHours = method_exists($attendance, 'getUnderworkedHours') ? $attendance->getUnderworkedHours() : 0;
                $underworkedValue = ($underworkedHours > 0 && $deductionRate > 0)
                    ? round($underworkedHours * $deductionRate, 2)
                    : 0;
                $overtimeValue = ($overtimeHours > 0 && $rate > 0)
                    ? round($overtimeHours * $rate, 2)
                    : 0;

                $status = 'حضور عادي';
                if ($attendance->is_absent) {
                    $status = 'غياب';
                } elseif ($attendance->is_late || $attendance->is_early_leave) {
                    $status = 'تأخير أو انصراف مبكر';
                } elseif ($overtimeHours > 0) {
                    $status = 'أوفر تايم';
                }

                return [
                    'date' => $attendance->date,
                    'status' => $status,
                    'is_absent' => $attendance->is_absent,
                    'is_late' => $attendance->is_late,
                    'is_early_leave' => $attendance->is_early_leave,
                    'overtime_hours' => $overtimeHours,
                    'overtime_value' => $overtimeValue,
                    'underworked_hours' => $underworkedHours,
                    'underworked_value' => $underworkedValue,
                    'check_in' => $attendance->check_in,
                    'check_out' => $attendance->check_out,
                ];
            })->sortBy('date');

            // إجمالي ساعات الأوفر تايم وقيمتها خلال الشهر
            $attendanceOvertimeHours = $attendanceDetails->sum('overtime_hours');
            $overtimeAmount = $attendanceDetails->sum('overtime_value');

            // --- أيام الحضور في الإجازات / العطلات المدفوعة (Paid Off-days) لنفس شهر الراتب ---
            $paidOffAttendances = $payroll->employee->attendances()
                ->whereBetween('date', [$startMonth, $endMonth])
                ->where('paid_for_off', true)
                ->get();
            
            $paidOffDays = $paidOffAttendances->count();

            // --- جلب الإجازات المستخدمة خلال شهر الراتب ---
            $usedLeaves = $payroll->employee->leaves()
                ->whereBetween('start_date', [$startMonth, $endMonth])
                ->orWhereBetween('end_date', [$startMonth, $endMonth])
                ->where(function($q) {
                    $q->where('status', 'approved')
                      ->orWhere('status', 'modified')
                      ->orWhere('approval_modified', 1);
                })
                ->with('leaveTypeModel')
                ->get();

            $paidOffAmount = 0;
            if ($paidOffDays > 0) {
                // prioritize employee's default_paid_off_amount, then fallback to calculated daily rate
                $dailyRatePaidOff = 0;
                if (!empty($payroll->employee->default_paid_off_amount) && $payroll->employee->default_paid_off_amount > 0) {
                    $dailyRatePaidOff = $payroll->employee->default_paid_off_amount;
                } elseif (!empty($compSnapshot['gross']) && $compSnapshot['gross'] > 0) {
                    $dailyRatePaidOff = ($compSnapshot['gross'] ?? 0) / 30;
                }
                $paidOffAmount = round($paidOffDays * $dailyRatePaidOff, 2);
            }

            // ✅ تعريف مصفوفات الترجمة
            $deductionTypes = [
                'tax' => 'ضريبة',
                'insurance' => 'تأمين',
                'loan' => 'قرض',
                'fine' => 'غرامة',
                'other' => 'أخرى',
                'leave' => 'إجازة',
                'health_insurance' => 'تأمين صحي',
                'social_insurance' => 'تأمين اجتماعي',
                'penalties' => 'جزاءات',
                'deductions' => 'خصومات',
                'loans' => 'سلف',
                'subscriptions' => 'اشتراكات',
                'other_deductions' => 'استقطاعات أخرى',
                'cash' => 'نقدي',
                'in_kind' => 'عيني',
                'deduction' => 'خصم',
                'penalty' => 'جزاء',
                'subscription' => 'اشتراك',
                'absence' => 'غياب',
            ];

            $loanTypes = [
                'personal' => 'شخصي',
                'car' => 'سيارة',
                'home' => 'سكن',
                'education' => 'تعليم',
                'medical' => 'طبي',
                'emergency' => 'طوارئ',
                'business' => 'تجاري',
                'other' => 'أخرى',
            ];

            return view('payroll.show', compact(
                'payroll',
                'usedDays',
                'allowed',
                'excess',
                'dailyRate',
                'leaveDeduction',
                'actualDeductions',    // ✅ تم تمريرها
                'loanInstallments',   // ✅ تم تمريرها
                'previousPayrolls',   // ✅ تم تمريرها
                'cumulativeLeaveInfo', // ✅ تم تمريرها
                'currency_name',      // ✅ تم تمريرها
                'deductionTypes',     // ✅ تم تمريرها
                'loanTypes',          // ✅ تم تمريرها
                'attendanceOvertimeHours', 'overtimeAmount',
                'attendanceDetails',   // ✅ تفاصيل الحضور والغياب والأوفر تايم
                'paidOffDays', 'paidOffAmount', 'paidOffAttendances', 'deductionRate', // ✅ أيام الحضور المدفوعة وقيمتها
                'usedLeaves' // ✅ الإجازات المستخدمة
            ));
        }
        public function edit(Payroll $payroll)
        {
            $this->authorize('update', $payroll);
            if ($payroll->status !== 'pending') {
                return redirect()->route('payroll.index')->with('error', 'لا يمكن تعديل راتب ليس بحالة "قيد الدفع".');
            }

            $payroll->load(['employee', 'payrollSalaryComponents']);
            // جلب الأقساط المعلقة لهذا الراتب (لنفس الشهر والسنة)
            $start = Carbon::create($payroll->year, $payroll->month, 1)->startOfMonth();
            $end = (clone $start)->endOfMonth();
            $pendingInstallments = $payroll->employee->loans()
                ->where('status', 'active')
                ->with(['installments' => function ($q) use ($start, $end) {
                    $q->whereBetween('due_date', [$start, $end])->where('status', 'pending');
                }])
                ->get()
                ->pluck('installments')
                ->flatten();
            $salaryComponents = SalaryComponent::all();
            $currency_name = get_currency_code();

            $usedDays = $payroll->employee->getUsedLeaveDaysInMonth($payroll->year, $payroll->month);
            $allowed = $payroll->employee->monthly_leave_days_allowed ?? 0;
            $excess = max(0, $usedDays - $allowed);
            $compSnapshot = $this->resolveCompensationForDate($payroll->employee, $startMonth);
            $dailyRate = $payroll->total_salary > 0 ? ($payroll->total_salary / 30) : (($compSnapshot['gross'] ?? $payroll->employee->getTotalSalaryWithComponentsAttribute()) / 30);
            
            // ✅ استخدام القيمة المحفوظة أولاً، أو حسابها من الدالة الجديدة
            $leaveDeduction = $payroll->leave_deduction ?? 0;
            if ($leaveDeduction == 0) {
                // إذا لم تكن هناك قيمة محفوظة، احسبها من الدالة الجديدة
                $leaveDeductionDetails = $payroll->employee->calculateLeaveDeductionDetailsNew($payroll->year, $payroll->month);
                if ($leaveDeductionDetails && count($leaveDeductionDetails) > 0) {
                    // البحث عن بيانات الشهر المحدد فقط (ليس آخر شهر في السلسلة)
                    $currentMonthDetails = $leaveDeductionDetails->firstWhere(function($item) use ($payroll) {
                        return $item['year'] == $payroll->year && $item['month'] == $payroll->month;
                    });
                    $leaveDeduction = $currentMonthDetails['leave_deduction_amount'] ?? 0;
                } else {
                    // احتياطي: استخدام الحساب اليدوي إذا فشلت الدالة
                    $leaveDeduction = $excess * $dailyRate;
                }
            }

            // ✅ حساب اوفرتايم الشهري (ساعات وقيمة) للعرض في صفحة التعديل
            $startMonth = Carbon::create($payroll->year, $payroll->month, 1)->startOfMonth();
            $endMonth = (clone $startMonth)->endOfMonth();
            $attendanceOvertimeHours = $payroll->employee->attendances()
                ->whereBetween('date', [$startMonth, $endMonth])
                ->get()
                ->map(fn($a) => $a->getOvertimeHours())
                ->sum();
            $overtimeAmount = 0;
            $compSnapshot = $this->resolveCompensationForDate($payroll->employee, $startMonth);
            $rate = $compSnapshot['overtime_rate'] ?? $payroll->employee->overtime_hourly_rate ?? 0;
            if ($attendanceOvertimeHours > 0 && $rate > 0) {
                $overtimeAmount = round($attendanceOvertimeHours * $rate, 2);
            }

            return view('payroll.edit', compact(
                'payroll', 'salaryComponents', 'currency_name', 'usedDays', 'allowed', 'excess', 'dailyRate', 'leaveDeduction'
                , 'attendanceOvertimeHours', 'overtimeAmount'
            ));
        }

        public function update(Request $request, Payroll $payroll)
        {
            $this->authorize('update', $payroll);
            if ($payroll->status !== 'pending') {
                return redirect()->route('payroll.index')->with('error', 'لا يمكن تعديل راتب ليس بحالة "قيد الدفع".');
            }

            $validator = Validator::make($request->all(), [
                'basic_salary' => 'required|numeric|min:0',
                'incentives' => 'nullable|numeric|min:0',
                'salary_component_names' => 'array|max:7',
                'salary_component_names.*' => 'string|max:255|nullable',
                'salary_component_values' => 'array|max:7',
                'salary_component_values.*' => 'required_with:salary_component_names.*|numeric|min:0',
                'salary_component_ids' => 'array|max:7',
                'salary_component_ids.*' => 'nullable|integer|exists:salary_components,id',
                'paid_off_amount' => 'nullable|numeric|min:0',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }

            DB::beginTransaction();
            try {
                $data = $validator->validated();

                // ✅ جلب القيم المطلوبة من قاعدة البيانات عند التحديث
                $activeDeductions = $this->getActiveDeductionsForPayroll($payroll->employee, $payroll->year, $payroll->month);
                // use compensation snapshot for the payroll month
                $snapshotDate = Carbon::create($payroll->year, $payroll->month, 1)->startOfMonth();
                $payrollCompSnapshot = $this->resolveCompensationForDate($payroll->employee, $snapshotDate);
                $loanInstallments = $this->getTotalLoanInstallmentsForPayroll($payroll->employee, $payroll->year, $payroll->month);
                $leaveDeduction = $payroll->employee->calculateLeaveDeductionForMonth($payroll->year, $payroll->month);

                // ✅ عند تحديث راتب موجود، نحتفظ بالقيم التي أدخلها المستخدم من النموذج
                // هذه القيم هي القيم المحفوظة في الراتب (من $payroll->basic_salary و $payroll->incentives)
                // ولا تتأثر بسجلات التعويض الجديدة - الرواتب القديمة تحتفظ بقيمها الأصلية
                $payroll->update([
                    'basic_salary' => $data['basic_salary'], // القيمة من النموذج (القيمة المحفوظة في الراتب)
                    'incentives' => $data['incentives'] ?? 0, // القيمة من النموذج (القيمة المحفوظة في الراتب)
                    'active_deductions' => $activeDeductions,
                    'loan_installments' => $loanInstallments,
                    'leave_deduction' => $leaveDeduction,
                ]);

                $payroll->payrollSalaryComponents()->delete();
                if (isset($data['salary_component_names']) && isset($data['salary_component_values'])) {
                    foreach ($data['salary_component_names'] as $index => $name) {
                        if (empty($name) || !isset($data['salary_component_values'][$index])) continue;
                        $componentId = $data['salary_component_ids'][$index] ?? null;
                        $payroll->payrollSalaryComponents()->create([
                            'salary_component_id' => $componentId,
                            'name' => $name,
                            'value' => $data['salary_component_values'][$index],
                        ]);
                    }
                }

                // إضافة مكوّن قيمة أيام الحضور المدفوعة إن وُجد في الطلب
                $postedPaidOff = $data['paid_off_amount'] ?? null;
                if (!empty($postedPaidOff) && is_numeric($postedPaidOff) && (float)$postedPaidOff > 0) {
                    // Check if this component already exists to prevent duplicates
                    $existingComponent = $payroll->payrollSalaryComponents()
                        ->where('name', 'Paid Off-day Compensation')
                        ->first();
                    
                    if (!$existingComponent) {
                        $payroll->payrollSalaryComponents()->create([
                            'salary_component_id' => null,
                            'name' => 'Paid Off-day Compensation',
                            'value' => (float) $postedPaidOff,
                            'meta' => ['type' => 'paid_off_days'],
                        ]);
                    } else {
                        // Update existing component with new value if needed
                        if ($existingComponent->value != (float) $postedPaidOff) {
                            $existingComponent->value = (float) $postedPaidOff;
                            $existingComponent->meta = ['type' => 'paid_off_days'];
                            $existingComponent->save();
                        }
                    }
                }

                // إعادة حساب وإضافة خصم نقصان الساعات عند التحديث التلقائي للمكونات
                $start = Carbon::create($payroll->year, $payroll->month, 1)->startOfMonth();
                $end = (clone $start)->endOfMonth();
                $attendanceUnderworkedHours = $payroll->employee->attendances()
                    ->whereBetween('date', [$start, $end])
                    ->get()
                    ->map(fn($a) => method_exists($a, 'getUnderworkedHours') ? $a->getUnderworkedHours() : 0)
                    ->sum();

                $underworkedAmount = 0;
                if (($payroll->employee->deduct_if_underworked ?? false) && $attendanceUnderworkedHours > 0) {
                    $underworkedAmount = round($payroll->employee->attendances()
                        ->whereBetween('date', [$start, $end])
                        ->get()
                        ->map(fn($a) => method_exists($a, 'getUnderworkedValue') ? $a->getUnderworkedValue() : 0)
                        ->sum(), 2);
                }

                if ($underworkedAmount > 0) {
                    $uwComponent = SalaryComponent::firstOrCreate(
                        ['name' => 'Underworked Deduction (خصم نقصان ساعات)'],
                        ['type' => 'deduction', 'description' => 'Deduction for hours worked less than scheduled']
                    );

                    // Use employee's deduction_hourly_rate if set, otherwise calculate automatically
                    $hourlyRate = $payroll->employee->deduction_hourly_rate ?? 0;
                    if ($hourlyRate == 0 && ($payrollCompSnapshot['gross'] ?? 0) > 0) {
                        $hourlyRate = round((($payrollCompSnapshot['gross'] ?? 0) / 30) / ($payroll->employee->scheduled_hours ?? 8), 4);
                    }
                    
                    $meta = [
                        'type' => 'underworked',
                        'hours' => $attendanceUnderworkedHours,
                        'hourly_rate' => $hourlyRate,
                        'formula' => 'underworked_hours × hourly_rate',
                        'calculation' => round($underworkedAmount, 2),
                    ];

                    $payroll->payrollSalaryComponents()->create([
                        'salary_component_id' => $uwComponent->id,
                        'name' => $uwComponent->name,
                        'value' => -1 * $underworkedAmount,
                        'meta' => $meta,
                    ]);
                }

                // ✅ إعادة الحساب بعد حفظ المكونات (لإضافة مكونات الراتب إلى total_salary)
                $payroll->calculateTotalSalary();
                $payroll->calculateNetSalary();
                $payroll->save();

                Activity::create([
                    'user_id' => auth()->id(),
                    'action' => 'updated_payroll',
                    'model_type' => Payroll::class,
                    'model_id' => $payroll->id,
                    'description' => 'updated_payroll',
                    'properties' => [
                        'employee_id' => $payroll->employee->id,
                        'employee' => $payroll->employee->name,
                        'month' => $payroll->month,
                        'year' => $payroll->year,
                        'net_salary' => $payroll->net_salary,
                    ],
                ]);

                DB::commit();
                return redirect()->route('payroll.index')->with('success', 'تم تحديث راتب الموظف بنجاح.');
            } catch (\Throwable $e) {
                DB::rollBack();
                Log::error('Payroll update error: ' . $e->getMessage());
                return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث الراتب.')->withInput();
            }
        }

        public function destroy(Payroll $payroll)
        {
            $this->authorize('delete', $payroll);
            if ($payroll->status !== 'pending') {
                return redirect()->route('payroll.index')->with('error', 'لا يمكن حذف راتب ليس بحالة "قيد الدفع".');
            }
            $payroll->delete();
            return redirect()->route('payroll.index')->with('success', 'تم حذف الراتب بنجاح.');
        }

        public function pay(Payroll $payroll)
        {
            $this->authorize('pay', $payroll);
            if ($payroll->status !== 'pending') {
                return redirect()->route('payroll.index')->with('error', 'لا يمكن دفع راتب ليس بحالة "قيد الدفع".');
            }

            DB::beginTransaction();
            try {
                    // ✅ تأكد من إعادة حساب الإجمالي والصافي استنادًا إلى مكونات الراتب المحفوظة
                    $payroll->load(['employee', 'payrollSalaryComponents']);
                    $payroll->calculateTotalSalary();
                    $payroll->calculateNetSalary();

                    // حفظ القيم المحسوبة ثم تهئية حالة الدفع
                    $payroll->update([
                        'total_salary' => $payroll->total_salary,
                        'net_salary' => $payroll->net_salary,
                        'status' => 'paid',
                        'paid_at' => now(),
                    ]);

                    // إرسال إشعار الدفع للموظف
                    if ($payroll->employee->user) {
                        $payroll->employee->user->notify(new \App\Notifications\PayrollPaidNotification($payroll));
                    }

                Activity::create([
                    'user_id' => auth()->id(),
                    'action' => 'paid_payroll',
                    'model_type' => Payroll::class,
                    'model_id' => $payroll->id,
                    'description' => 'paid_payroll',
                    'properties' => [
                        'employee_id' => $payroll->employee->id,
                        'employee' => $payroll->employee->name,
                        'month' => $payroll->month,
                        'year' => $payroll->year,
                        'net_salary' => $payroll->net_salary,
                    ],
                ]);

                // عند دفع الراتب: تحقق من أقساط القروض المستحقة لهذا الشهر وعلمها كمدفوعة
                try {
                    $start = Carbon::create($payroll->year, $payroll->month, 1)->startOfMonth();
                    $end = (clone $start)->endOfMonth();

                    $installments = LoanInstallment::whereHas('loan', function ($q) use ($payroll) {
                        $q->where('employee_id', $payroll->employee_id);
                    })
                    ->whereBetween('due_date', [$start, $end])
                    ->where('status', 'pending')
                    ->with('loan')
                    ->get();

                    foreach ($installments as $inst) {
                        try {
                            $inst->markAsPaid('payroll', 'مدفوع عبر صرف الراتب #' . $payroll->id);

                            $loan = $inst->loan;
                            if ($loan) {
                                $paidSum = $loan->installments()->where('status', 'paid')->sum('amount');
                                $loan->paid_amount = $paidSum;
                                $loan->remaining_amount = max(0, $loan->total_amount - $paidSum);
                                $loan->save();
                            }
                        } catch (\Throwable $e) {
                            Log::error('Failed to mark loan installment as paid for payroll #' . $payroll->id . ': ' . $e->getMessage());
                        }
                    }
                    // --------------------------------------------------
                    // معالجة أقساط إضافية محددة يدوياً (installment_ids)
                    // يمكن إرسالها ضمن الطلب عند الدفع أو حفظها مسبقًا في حقل notes عند الإنشاء
                    $requestedExtraIds = $request->input('installment_ids', null);
                    if (empty($requestedExtraIds) && !empty($payroll->meta)) {
                        try {
                            $meta = is_array($payroll->meta) ? $payroll->meta : (json_decode($payroll->meta, true) ?: []);
                            if (!empty($meta['installment_ids']) && is_array($meta['installment_ids'])) {
                                $requestedExtraIds = $meta['installment_ids'];
                            }
                        } catch (\Throwable $e) {
                            // تجاهل أخطاء JSON
                        }
                    }

                    if (!empty($requestedExtraIds) && is_array($requestedExtraIds)) {
                        $extraIds = array_filter($requestedExtraIds, fn($v) => is_numeric($v));
                        if (!empty($extraIds)) {
                            $extraInstallments = LoanInstallment::whereIn('id', $extraIds)
                                ->where('status', 'pending')
                                ->whereHas('loan', fn($q) => $q->where('employee_id', $payroll->employee_id))
                                ->with('loan')
                                ->get();

                            foreach ($extraInstallments as $einst) {
                                try {
                                    $einst->markAsPaid('payroll', 'مدفوع عبر صرف الراتب #' . $payroll->id . ' (دفعة إضافية)');
                                    $loan = $einst->loan;
                                    if ($loan) {
                                        $paidSum = $loan->installments()->where('status', 'paid')->sum('amount');
                                        $loan->paid_amount = $paidSum;
                                        $loan->remaining_amount = max(0, $loan->total_amount - $paidSum);
                                        $loan->save();
                                    }
                                } catch (\Throwable $ee) {
                                    Log::error('Failed to mark extra loan installment as paid for payroll #' . $payroll->id . ': ' . $ee->getMessage());
                                }
                            }
                        }
                    }
                    // --------------------------------------------------
                } catch (\Throwable $e) {
                    Log::error('Error while processing loan installments on payroll pay: ' . $e->getMessage());
                }

                DB::commit();
                return redirect()->route('payroll.index')->with('success', 'تم دفع راتب الموظف بنجاح.');
            } catch (\Throwable $e) {
                DB::rollBack();
                Log::error('Payroll pay error: ' . $e->getMessage());
                return redirect()->route('payroll.index')->with('error', 'حدث خطأ أثناء دفع الراتب.');
            }
        }

    public function printDetails(Payroll $payroll)
    {
        $this->authorize('view', $payroll);

        // ✅ تحميل العلاقات المهمة
        $payroll->load(['employee', 'payrollSalaryComponents']);

        // ✅ حساب معلومات الإجازات
        $leaveDetails = $payroll->employee->calculateLeaveDeductionDetailsNew($payroll->year, $payroll->month);
        $currentMonthDetails = $leaveDetails->firstWhere(function ($detail) use ($payroll) {
            return $detail['month'] == $payroll->month && $detail['year'] == $payroll->year;
        });
        $cumulativeLeaveInfo = [
            'current_month_used' => $currentMonthDetails['used_this_month'] ?? 0,
            'current_month_excess' => $currentMonthDetails['excess_days'] ?? 0,
        ];
        $dailyRate = $currentMonthDetails['daily_rate'] ?? 0;

        // ✅ حساب مكونات الراتب الإضافية (التي تظهر في قسم "إجمالي الدخل")
        $payrollComponents = $payroll->payrollSalaryComponents;
        $totalSalary = $payroll->basic_salary + $payroll->incentives;
        foreach ($payrollComponents as $component) {
            $totalSalary += $component->value;
        }

        // ✅ جلب الاستقطاعات النشطة لهذا الموظف في نفس شهر وسنة هذا الراتب
        $actualDeductions = $this->getActiveDeductionsForPayroll($payroll->employee, $payroll->year, $payroll->month);

        // ✅ جلب أقساط القروض المدفوعة في نفس شهر وسنة هذا الراتب
// ✅ التصحيح: نستخدم العلاقة مع جدول loans أولاً، لأن loan_installments لا يحتوي على employee_id
$loanInstallments = LoanInstallment::whereHas('loan', function ($query) use ($payroll) {
        $query->where('employee_id', $payroll->employee->id);
    })
    ->whereYear('due_date', $payroll->year)
    ->whereMonth('due_date', $payroll->month)
    ->where('status', 'paid') // ✅ افتراض أن المدفوع فقط هو الذي يخصم
    ->with('loan') // ✅ تحميل معلومات القرض
    ->get();

        // ✅ حساب إجمالي الخصومات
        $calculatedTotalDeductions = $payroll->active_deductions + $payroll->loan_installments + $payroll->leave_deduction;

        // ✅ حساب الراتب الصافي
        $netSalary = $payroll->net_salary; // ✅ استخدام القيمة المحفوظة في قاعدة البيانات

        // ✅ جلب رمز العملة من الإعدادات
        $currency_name = get_currency_code();

        // ✅ تعريف مصفوفات الترجمة
        $deductionTypes = [
            'tax' => 'ضريبة',
            'insurance' => 'تأمين',
            'loan' => 'قرض',
            'fine' => 'غرامة',
            'absence' => 'غياب',
            'penalty' => 'جزاء',
            'deduction' => 'خصم',
            'subscription' => 'اشتراك',
            'other' => 'أخرى',
            'other_deductions' => 'استقطاعات أخرى', // ✅ إضافة القيمة الجديدة
        ];

        // ✅ تمرير جميع المتغيرات إلى العرض
        return view('payroll.print-payroll-details', compact(
            'payroll',
            'payrollComponents',
            'totalSalary',
            'cumulativeLeaveInfo',
            'dailyRate',
            'actualDeductions',
            'loanInstallments',
            'calculatedTotalDeductions',
            'currency_name',
            'deductionTypes',
            'netSalary' // ✅ تمرير الراتب الصافي
        ));
    }

}