<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Activity; // ✅ استيراد نموذج النشاط
use Illuminate\Support\Facades\Auth;
use Illuminate\Notifications\DatabaseNotification;
use App\Models\Notification; // ✅ استيراد نموذج الإشعار المخصص (App\Models\Notification)

class NotificationController extends Controller
{
    /**
     * عرض قائمة الإشعارات للمستخدم الحالي.
     */
    public function index(Request $request)
    {
        $user = auth()->user();
        $isEmployee = $user->hasRole('employee');

        // ✅ تحديد استعلام الإشعارات بناءً على دور المستخدم
        if ($isEmployee) {
            // 🟢 للموظف: جلب إشعاراته فقط
            $query = $user->notifications();
        } else {
            // 🔵 للإدارة: جلب جميع الإشعارات في النظام
            $query = Notification::latest(); // أو Notification::orderByDesc('created_at')
        }

        // ⚠️ تطبيق الفلترات الإضافية (مثل البحث، الحالة) فقط إذا لم يكن المستخدم موظفًا عاديًا
        if (!$isEmployee) {
            if ($search = $request->input('search')) {
                $query->where('data', 'like', "%{$search}%");
            }

            if ($type = $request->input('type')) {
                $query->where('type', 'like', "%{$type}%");
            }

            if ($status = $request->input('status')) {
                $query->where('read_at', $status === 'read' ? '!= null' : null);
            }
        }

        // ✅ جلب البيانات مع التصفية وتطبيق الترقيم
        $notifications = $query->paginate(15);

        // ✅ حساب الإحصائيات
        // إذا كان المستخدم موظفًا، نحسب إشعاراته فقط
        // إذا كان من دور إداري، نحسب جميع الإشعارات
        if ($isEmployee) {
            $totalNotifications = $user->notifications()->count();
            $unreadNotifications = $user->unreadNotifications()->count();
        } else {
            $totalNotifications = Notification::count();
            $unreadNotifications = Notification::whereNull('read_at')->count();
        }
        $readNotifications = $totalNotifications - $unreadNotifications;

        // ✅ جلب الأنشطة الأخيرة (مثلاً آخر 5 أنشطة)
        // محدودة أيضًا حسب صلاحيات المستخدم
        $activitiesQuery = Activity::with(['user', 'subject']); // تحميل العلاقات إن وُجدت

        if ($isEmployee) {
            // إذا كان المستخدم موظفًا، يمكن تقييد الأنشطة لعرض أنشطته فقط
            // $activitiesQuery->where('user_id', $user->id);
            // أو فقط عرض أنشطة تخصه كـ "notifiable" (مثل طلباته)
            // $activitiesQuery->where('subject_type', get_class($user->employee))
            //                ->where('subject_id', $user->employee->id);
            // في هذا المثال، سنعرض الأنشطة العامة فقط للإدارة
            $recentActivities = collect(); // عرض قائمة فارغة للموظف
        } else {
            // للإدارة، عرض الأنشطة الأخيرة
            $recentActivities = $activitiesQuery->latest()->limit(5)->get();
        }

        return view('notifications.index', compact(
            'notifications',
            'totalNotifications',
            'unreadNotifications',
            'readNotifications',
            'recentActivities' // ✅ تمرير المتغير إلى العرض
        ));
    }

    /**
     * عرض صفحة إعدادات الإشعارات.
     */
    public function settings()
    {
        $user = auth()->user();
        // افتراض أن إعدادات الإشعارات مخزنة في علاقة أو عمود في نموذج المستخدم
        // $settings = $user->notificationSettings; // مثال
        $settings = []; // مؤقتاً

        return view('notifications.settings', compact('settings'));
    }

    /**
     * تحديث إعدادات الإشعارات.
     */
    public function updateSettings(Request $request)
    {
        $user = auth()->user();
        // $settings = $user->notificationSettings ?? new NotificationSetting();
        // $settings->fill($request->all());
        // $settings->user_id = auth()->id();
        // $settings->save();

        return redirect()->route('notifications.settings')->with('success', 'تم تحديث إعدادات الإشعارات بنجاح');
    }

    /**
     * تعليم إشعار واحد كمقروء.
     */
    public function markAsRead(DatabaseNotification $notification)
    {
        // ✅ التحقق من أن الإشعار يخص المستخدم الحالي
        // هذا التحقق ضروري لمنع المستخدم من تعديل إشعارات الآخرين
        if ($notification->notifiable_id !== Auth::id() || $notification->notifiable_type !== Auth::user()->getMorphClass()) {
            abort(403, 'لا يمكنك الوصول إلى هذا الإشعار.');
        }

        $notification->markAsRead();

        return response()->json(['success' => true]);
    }

    /**
     * تعليم جميع إشعارات المستخدم الحالي كمقروءة.
     */
    public function markAllAsRead()
    {
        $user = Auth::user();

        // ✅ تعليم جميع الإشعارات غير المقروءة فقط كمقروءة
        $user->unreadNotifications->markAsRead();

        // أو يمكنك استخدام:
        // $user->notifications()->whereNull('read_at')->update(['read_at' => now()]);

        return response()->json(['success' => true]);
    }

    /**
     * حذف إشعار واحد.
     */
    public function destroy(DatabaseNotification $notification)
    {
        // ✅ التحقق من أن الإشعار يخص المستخدم الحالي
        // هذا التحقق ضروري لمنع المستخدم من حذف إشعارات الآخرين
        if ($notification->notifiable_id !== Auth::id() || $notification->notifiable_type !== Auth::user()->getMorphClass()) {
            abort(403, 'لا يمكنك الوصول إلى هذا الإشعار.');
        }

        $notification->delete();

        return response()->json(['success' => true]);
        // أو إعادة التوجيه
        // return redirect()->route('notifications.index')->with('success', 'تم حذف الإشعار.');
    }

    /**
     * مسح جميع إشعارات المستخدم الحالي.
     */
    public function clearAll()
    {
        $user = Auth::user();

        // ✅ حذف جميع الإشعارات فقط المرتبطة بالمستخدم الحالي
        $user->notifications()->delete();

        return response()->json(['success' => true]);
        // أو إعادة التوجيه
        // return redirect()->route('notifications.index')->with('success', 'تم مسح جميع الإشعارات.');
    }
}