<?php

namespace App\Http\Controllers;

use App\Models\Loan;
use App\Models\Employee;
use App\Models\LoanInstallment;
use App\Notifications\LoanInstallmentPaidNotification;
use App\Notifications\LoanApprovedNotification;
use App\Notifications\NewLoanRequestNotification; // استيراد فئة الإشعار الجديدة
use App\Notifications\LoanInstallmentDue;
use App\Models\Activity;
use App\Models\User; // استيراد نموذج User
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class LoanController extends Controller
{
  /**
 * Display a listing of the resource.
 */
public function index()
{
    $this->authorize('viewAny', Loan::class);
    
    $authenticatedUser = auth()->user();

    // ✅ التحقق من وجود العلاقة مع Employee
    if ($authenticatedUser->hasRole('employee') && !$authenticatedUser->employee) {
        // ✅ إرجاع كائن Pagination فارغ بدل Collection لتجنب خطأ hasPages()
        $emptyPaginator = Loan::whereRaw('1 = 0')->paginate(15);
        return view('loans.index', [
            'loans' => $emptyPaginator,
            'employees' => collect(),
            'stats' => [
                'total' => 0,
                'active' => 0,
                'pending' => 0,
                'rejected' => 0,
                'paid' => 0,
                'cancelled' => 0,
            ],
            'error' => 'Your employee profile is not linked to your user account. Please contact HR.'
        ]);
    }

    if ($authenticatedUser->hasRole('employee')) {
        $query = Loan::with(['employee', 'employee.user'])->where('employee_id', $authenticatedUser->employee->id);
    } else {
        $query = Loan::with(['employee', 'employee.user']);
    }

    // تصفية البيانات (متوفرة فقط للإدارة)
    if (!$authenticatedUser->hasRole('employee')) {
        if (request('search')) {
            $search = request('search');
            $query->whereHas('employee', function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('employee_id', 'like', "%{$search}%");
            });
        }

        if (request('employee_id')) {
            $query->where('employee_id', request('employee_id'));
        }

        if (request('status')) {
            $query->where('status', request('status'));
        }

        if (request('type')) {
            $query->where('loan_type', request('type'));
        }
    }

    $loans = $query->latest()->paginate(15);
    
    // جلب الموظفين لقائمة الفلترة (متوفرة فقط للإدارة)
    if ($authenticatedUser->hasRole('employee')) {
        $employees = collect([$authenticatedUser->employee]);
    } else {
        $employees = Employee::orderBy('name')->get();
    }
    
    // إحصائيات
    if ($authenticatedUser->hasRole('employee')) {
        $stats = [
            'total' => $loans->total(),
            'active' => $query->where('status', 'active')->count(),
            'pending' => $query->where('status', 'pending')->count(),
            'rejected' => $query->where('status', 'rejected')->count(),
            'paid' => $query->where('status', 'paid')->count(),
            'cancelled' => $query->where('status', 'cancelled')->count(),
        ];
    } else {
        $stats = [
            'total' => Loan::count(),
            'active' => Loan::where('status', 'active')->count(),
            'pending' => Loan::where('status', 'pending')->count(),
            'rejected' => Loan::where('status', 'rejected')->count(),
            'paid' => Loan::where('status', 'paid')->count(),
            'cancelled' => Loan::where('status', 'cancelled')->count(),
        ];
    }
    
    return view('loans.index', compact('loans', 'employees', 'stats'));
}

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $this->authorize('create', Loan::class);
        
        if (auth()->user()->hasRole('employee')) {
            $employees = collect([auth()->user()->employee]);
        } else {
            $employees = Employee::orderBy('name')->get();
        }
        
        $currency_name = get_currency_code();
        
        return view('loans.create', compact('employees', 'currency_name'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $this->authorize('create', Loan::class);
        
        $currencyName = get_currency_code();
        $currencyCode = get_currency_code();
        
        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'loan_type' => 'required|in:personal,housing,car,emergency,education',
            'amount' => 'required|numeric|min:1|max:100000',
            'installment_count' => 'required|integer|min:1|max:120',
            'interest_rate' => 'nullable|numeric|min:0|max:100',
            'start_date' => 'required|date|after:today',
            'reason' => 'required|string|max:500',
            'notes' => 'nullable|string|max:500'
        ]);

        $authenticatedUser = auth()->user();

        // التأكد من أن الموظف يطلب القرض لنفسه فقط
        if ($authenticatedUser->hasRole('employee') && $authenticatedUser->employee->id !== (int)$request->employee_id) {
            abort(403, 'لا يمكنك إنشاء طلب قرض لموظف آخر.');
        }

        DB::transaction(function () use ($request, $currencyCode, $authenticatedUser) {
            $employee = Employee::findOrFail($request->employee_id);
            
            $loanTypeArabic = $this->getLoanTypeArabic($request->loan_type);
            $title = "قرض {$loanTypeArabic} - {$employee->employee_id} - {$request->amount} {$currencyCode}";

            $amount = $request->amount;
            $interestRate = $request->interest_rate ?? 0;
            $interestAmount = $amount * ($interestRate / 100);
            $totalAmount = $amount + $interestAmount;
            $installmentCount = $request->installment_count;
            $monthlyInstallment = $totalAmount / $installmentCount;
            
            $status = $authenticatedUser->hasRole(['admin', 'finance']) ? 'approved' : 'pending';
            $loanNumber = $this->generateLoanNumber();
            $startDate = Carbon::parse($request->start_date);
            $endDate = $startDate->copy()->addMonths($installmentCount - 1);

            $loan = Loan::create([
                'employee_id' => $request->employee_id,
                'loan_number' => $loanNumber,
                'title' => $title,
                'loan_type' => $request->loan_type,
                'amount' => $amount,
                'interest_rate' => $interestRate,
                'interest_amount' => $interestAmount,
                'total_amount' => $totalAmount,
                'installment_count' => $installmentCount,
                'installment_amount' => $monthlyInstallment,
                'start_date' => $request->start_date,
                'end_date' => $endDate,
                'reason' => $request->reason,
                'notes' => $request->notes,
                'status' => $status,
                'paid_amount' => 0,
                'remaining_amount' => $totalAmount
            ]);

            Activity::create([
                'user_id' => auth()->id(),
                'description' => 'created_loan',
                'subject_type' => Loan::class,
                'subject_id' => $loan->id,
                'properties' => [
                    'loan_amount' => $loan->amount,
                    'employee_name' => $loan->employee->name,
                    'loan_type' => $loan->loan_type,
                ],
            ]);

            // --- القسم المضاف ---
            // فقط إذا كان الحالة 'pending' (أي تم إنشاؤه من قبل موظف عادي)
            if ($loan->status === 'pending') {
                // جلب المستخدمين الذين لديهم الأدوار المطلوبة (Admin, HR, Super-Admin)
                // جلب المستخدمين الذين لديهم الأدوار المطلوبة باستخدام علاقة roles و spatie
$adminUsers = User::whereHas('roles', function ($query) {
    $query->whereIn('name', ['admin', 'hr', 'super-admin']);
})->get();

                // إرسال الإشعار إلى كل مستخدم إداري
                foreach ($adminUsers as $adminUser) {
                    $adminUser->notify(new NewLoanRequestNotification($loan));
                }
            }
            // --- نهاية القسم المضاف ---

            for ($i = 1; $i <= $installmentCount; $i++) {
                $dueDate = $startDate->copy()->addMonths($i - 1);
                $loan->installments()->create([
                    'installment_number' => $i,
                    'amount' => $monthlyInstallment,
                    'due_date' => $dueDate,
                    'status' => 'pending'
                ]);
            }

            if ($status === 'approved' && $employee->user) {
                $employee->user->notify(new LoanApprovedNotification($loan));
            }
        });

        return redirect()->route('loans.index')
                        ->with('success', 'تم تقديم طلب القرض بنجاح');
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $loan = Loan::with(['employee', 'employee.user', 'installments' => function($query) {
            $query->orderBy('installment_number');
        }])->findOrFail($id);
        
        $this->authorize('view', $loan);

        $authenticatedUser = auth()->user();

        // التأكد من أن الموظف يعرض تفاصيل قرضه الخاص فقط
        if ($authenticatedUser->hasRole('employee') && $authenticatedUser->employee->id !== $loan->employee_id) {
            abort(403, 'لا يمكنك عرض تفاصيل قرض لموظف آخر.');
        }

        return view('loans.show', compact('loan'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $loan = Loan::with(['employee', 'employee.user'])->findOrFail($id);
        $this->authorize('update', $loan);

        $authenticatedUser = auth()->user();

        // التأكد من أن الموظف يعرض تعديل لطلبه الخاص فقط
        if ($authenticatedUser->hasRole('employee') && $authenticatedUser->employee->id !== $loan->employee_id) {
            abort(403, 'لا يمكنك تعديل طلب قرض لموظف آخر.');
        }

        if (in_array($loan->status, ['active', 'paid'])) {
            return redirect()->route('loans.index')
                           ->with('error', 'لا يمكن تعديل القرض بعد تفعيله أو دفع أقساط منه');
        }

        if ($authenticatedUser->hasRole('employee')) {
            $employees = collect([$authenticatedUser->employee]);
        } else {
            $employees = Employee::orderBy('name')->get();
        }

        return view('loans.edit', compact('loan', 'employees'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Loan $loan)
    {
        $this->authorize('update', $loan);

        $authenticatedUser = auth()->user();

        // التأكد من أن الموظف يعدل طلبه الخاص فقط
        if ($authenticatedUser->hasRole('employee') && $authenticatedUser->employee->id !== $loan->employee_id) {
            abort(403, 'لا يمكنك تعديل طلب قرض لموظف آخر.');
        }

        if ($authenticatedUser->hasRole('employee') && $loan->employee_id !== (int)$request->employee_id) {
            abort(403, 'لا يمكنك تغيير الموظف المرتبط بالطلب.');
        }

        if (in_array($loan->status, ['active', 'paid'])) {
            return redirect()->route('loans.index')
                           ->with('error', 'لا يمكن تعديل القرض بعد تفعيله أو دفع أقساط منه');
        }

        $currencyName = get_currency_code();
        $currencyCode = get_currency_code();
        
        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'loan_type' => 'required|in:personal,housing,car,emergency,education',
            'amount' => 'required|numeric|min:1|max:100000',
            'installment_count' => 'required|integer|min:1|max:120',
            'interest_rate' => 'nullable|numeric|min:0|max:100',
            'start_date' => 'required|date|after:today',
            'reason' => 'required|string|max:500',
            'notes' => 'nullable|string|max:500'
        ]);

        DB::transaction(function () use ($request, $loan, $currencyCode) {
            $amount = $request->amount;
            $interestRate = $request->interest_rate ?? 0;
            $interestAmount = $amount * ($interestRate / 100);
            $totalAmount = $amount + $interestAmount;
            $installmentCount = $request->installment_count;
            $monthlyInstallment = $totalAmount / $installmentCount;
            
            $employee = Employee::findOrFail($loan->employee_id);
            $loanTypeArabic = $this->getLoanTypeArabic($request->loan_type);
            $title = "قرض {$loanTypeArabic} - {$employee->employee_id} - {$request->amount} {$currencyCode}";
            
            $startDate = Carbon::parse($request->start_date);
            $endDate = $startDate->copy()->addMonths($installmentCount - 1);

            $oldAmount = $loan->amount;
            $oldInstallmentCount = $loan->installment_count;
            $oldInterestRate = $loan->interest_rate;

            $loan->update([
                'employee_id' => $request->employee_id,
                'title' => $title,
                'loan_type' => $request->loan_type,
                'amount' => $amount,
                'interest_rate' => $interestRate,
                'interest_amount' => $interestAmount,
                'total_amount' => $totalAmount,
                'installment_count' => $installmentCount,
                'installment_amount' => $monthlyInstallment,
                'start_date' => $request->start_date,
                'end_date' => $endDate,
                'reason' => $request->reason,
                'notes' => $request->notes,
                'remaining_amount' => $totalAmount
            ]);

            Activity::create([
                'user_id' => auth()->id(),
                'description' => 'updated_loan',
                'subject_type' => Loan::class,
                'subject_id' => $loan->id,
                'properties' => [
                    'old_amount' => $oldAmount,
                    'new_amount' => $loan->amount,
                    'old_installment_count' => $oldInstallmentCount,
                    'new_installment_count' => $loan->installment_count,
                    'old_interest_rate' => $oldInterestRate,
                    'new_interest_rate' => $loan->interest_rate,
                    'employee_name' => $loan->employee->name,
                ],
            ]);

            $loan->installments()->delete();
            
            for ($i = 1; $i <= $installmentCount; $i++) {
                $dueDate = $startDate->copy()->addMonths($i - 1);
                $loan->installments()->create([
                    'installment_number' => $i,
                    'amount' => $monthlyInstallment,
                    'due_date' => $dueDate,
                    'status' => 'pending'
                ]);
            }
        });

        return redirect()->route('loans.index')
                        ->with('success', 'تم تحديث طلب القرض بنجاح');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Loan $loan)
    {
        $this->authorize('delete', $loan);

        $authenticatedUser = auth()->user();

        // التأكد من أن الموظف يحذف طلبه الخاص فقط
        if ($authenticatedUser->hasRole('employee') && $authenticatedUser->employee->id !== $loan->employee_id) {
            abort(403, 'لا يمكنك حذف طلب قرض لموظف آخر.');
        }

        if (in_array($loan->status, ['active', 'paid'])) {
            return redirect()->route('loans.index')
                           ->with('error', 'لا يمكن حذف القرض بعد تفعيله أو دفع أقساط منه');
        }

        if (!in_array($loan->status, ['pending'])) {
            return redirect()->route('loans.index')
                           ->with('error', 'لا يمكن حذف هذا الطلب.');
        }

        DB::transaction(function () use ($loan) {
            Activity::create([
                'user_id' => auth()->id(),
                'description' => 'deleted_loan',
                'subject_type' => Loan::class,
                'subject_id' => $loan->id,
                'properties' => [
                    'loan_amount' => $loan->amount,
                    'employee_name' => $loan->employee->name,
                    'loan_type' => $loan->loan_type,
                ],
            ]);

            $loan->installments()->delete();
            $loan->delete();
        });

        return redirect()->route('loans.index')
                        ->with('success', 'تم حذف طلب القرض بنجاح');
    }

    /**
     * Approve Loan.
     */
    public function approve(Loan $loan)
    {
        $this->authorize('approve', $loan);
        
        if ($loan->status !== 'pending') {
            return redirect()->route('loans.index')
                           ->with('error', 'لا يمكن الموافقة على قرض غير معلق');
        }

        $loan->update([
            'status' => 'approved',
            'approved_by' => auth()->id(),
            'approved_at' => now()
        ]);

        Activity::create([
            'user_id' => auth()->id(),
            'description' => 'approved_loan',
            'subject_type' => Loan::class,
            'subject_id' => $loan->id,
            'properties' => [
                'loan_amount' => $loan->amount,
                'employee_name' => $loan->employee->name,
                'approved_by' => auth()->user()->name,
            ],
        ]);

        if ($loan->employee && $loan->employee->user) {
            $loan->employee->user->notify(new LoanApprovedNotification($loan));
        }

        return redirect()->route('loans.index')
                        ->with('success', 'تمت الموافقة على طلب القرض');
    }

    /**
     * Reject Loan.
     */
    public function reject(Request $request, Loan $loan)
    {
        $this->authorize('approve', $loan);
        
        if ($loan->status !== 'pending') {
            return redirect()->route('loans.index')
                           ->with('error', 'لا يمكن رفض قرض غير معلق');
        }
        
        $request->validate([
            'rejection_reason' => 'required|string|max:500'
        ]);

        $loan->update([
            'status' => 'rejected',
            'rejection_reason' => $request->rejection_reason,
            'rejected_by' => auth()->id(),
            'rejected_at' => now()
        ]);

        Activity::create([
            'user_id' => auth()->id(),
            'description' => 'rejected_loan',
            'subject_type' => Loan::class,
            'subject_id' => $loan->id,
            'properties' => [
                'loan_amount' => $loan->amount,
                'employee_name' => $loan->employee->name,
                'rejection_reason' => $request->rejection_reason,
                'rejected_by' => auth()->user()->name,
            ],
        ]);

        // إرسال إشعار الرفض للموظف
        if ($loan->employee->user) {
            $loan->employee->user->notify(new \App\Notifications\LoanRejectedNotification($loan, $request->rejection_reason));
        }

        return redirect()->route('loans.index')
                        ->with('success', 'تم رفض طلب القرض');
    }

    /**
     * Activate Loan (make it active).
     */
    public function activate(Loan $loan)
    {
        $this->authorize('activate', $loan);
        
        if ($loan->status !== 'approved') {
            return redirect()->route('loans.index')
                           ->with('error', 'فقط القروض الموافق عليها يمكن تفعيلها');
        }

        $loan->update([
            'status' => 'active',
            'activated_by' => auth()->id(),
            'activated_at' => now()
        ]);

        Activity::create([
            'user_id' => auth()->id(),
            'description' => 'activated_loan',
            'subject_type' => Loan::class,
            'subject_id' => $loan->id,
            'properties' => [
                'loan_amount' => $loan->amount,
                'employee_name' => $loan->employee->name,
                'activated_by' => auth()->user()->name,
            ],
        ]);

        return redirect()->route('loans.index')
                        ->with('success', 'تم تفعيل القرض');
    }

    /**
     * Cancel Loan.
     */
    public function cancel(Loan $loan)
    {
        $this->authorize('cancel', $loan);
        
        $authenticatedUser = auth()->user();

        // التأكد من أن الموظف يلغي طلبه الخاص فقط
        if ($authenticatedUser->hasRole('employee') && $authenticatedUser->employee->id !== $loan->employee_id) {
            abort(403, 'لا يمكنك إلغاء طلب قرض لموظف آخر.');
        }

        if (!in_array($loan->status, ['approved', 'active'])) {
            return redirect()->route('loans.index')
                           ->with('error', 'فقط القروض الموافق عليها أو النشطة يمكن إلغاؤها');
        }

        $loan->update([
            'status' => 'cancelled',
            'cancelled_by' => auth()->id(),
            'cancelled_at' => now()
        ]);

        Activity::create([
            'user_id' => auth()->id(),
            'description' => 'cancelled_loan',
            'subject_type' => Loan::class,
            'subject_id' => $loan->id,
            'properties' => [
                'loan_amount' => $loan->amount,
                'employee_name' => $loan->employee->name,
                'cancelled_by' => auth()->user()->name,
            ],
        ]);

        return redirect()->route('loans.index')
                        ->with('success', 'تم إلغاء القرض');
    }

    /**
     * Pay Loan Installment.
     */
    public function payInstallment(Request $request, Loan $loan)
    {
        $this->authorize('pay', $loan);
        
        $authenticatedUser = auth()->user();

        // التأكد من أن الموظف يدفع قسطًا لقرضه الخاص فقط
        if ($authenticatedUser->hasRole('employee') && $authenticatedUser->employee->id !== $loan->employee_id) {
            abort(403, 'لا يمكنك دفع أقساط قرض لموظف آخر.');
        }

        if ($loan->status !== 'active') {
            return redirect()->route('loans.index')
                           ->with('error', 'لا يمكن دفع أقساط قرض غير نشط');
        }
        
        $currencyName = get_currency_code();
        
        $request->validate([
            'installment_id' => 'required|exists:loan_installments,id',
            'payment_date' => 'required|date',
            'payment_method' => 'required|in:cash,bank_transfer,check,deduction',
            'notes' => 'nullable|string|max:500'
        ]);

        DB::transaction(function () use ($request, $loan) {
            $installment = LoanInstallment::findOrFail($request->installment_id);
            
            if ($installment->loan_id !== $loan->id || $installment->status !== 'pending') {
                throw new \Illuminate\Http\Exceptions\HttpResponseException(
                    redirect()->back()->with('error', 'حدث خطأ أثناء معالجة الدفع. يرجى المحاولة مرة أخرى.')
                );
            }

            $oldStatus = $installment->status;
            $oldPaidDate = $installment->paid_date;

            $installment->update([
                'paid_date' => $request->payment_date,
                'payment_method' => $request->payment_method,
                'notes' => $request->notes,
                'status' => 'paid'
            ]);

            Activity::create([
                'user_id' => auth()->id(),
                'description' => 'paid_loan_installment',
                'subject_type' => LoanInstallment::class,
                'subject_id' => $installment->id,
                'properties' => [
                    'installment_number' => $installment->installment_number,
                    'installment_amount' => $installment->amount,
                    'loan_amount' => $loan->amount,
                    'employee_name' => $loan->employee->name,
                    'payment_method' => $installment->payment_method,
                    'old_status' => $oldStatus,
                    'new_status' => $installment->status,
                ],
            ]);
            
            $paidAmount = $loan->installments()->where('status', 'paid')->sum('amount');
            $remainingAmount = $loan->total_amount - $paidAmount;
            $status = $remainingAmount <= 0 ? 'paid' : 'active';
            
            $loan->update([
                'paid_amount' => $paidAmount,
                'remaining_amount' => $remainingAmount,
                'status' => $status
            ]);

            if ($loan->employee && $loan->employee->user) {
                $loan->employee->user->notify(new LoanInstallmentPaidNotification($installment, $loan));
            }
        });

        return redirect()->route('loans.show', $loan)
                        ->with('success', 'تم دفع القسط بنجاح');
    }

    /**
     * توليد رقم فريد للقرض
     */
    private function generateLoanNumber(): string
    {
        $prefix = 'LN';
        $date = now()->format('Ymd');
        $lastLoan = Loan::where('loan_number', 'like', $prefix . $date . '%')
            ->orderBy('id', 'desc')
            ->first();

        $number = $lastLoan ? (int) substr($lastLoan->loan_number, -4) + 1 : 1;
        return $prefix . $date . str_pad($number, 4, '0', STR_PAD_LEFT);
    }

    /**
     * تحويل نوع القرض إلى العربية
     */
    private function getLoanTypeArabic($type): string
    {
        return match($type) {
            'personal' => 'شخصي',
            'housing' => 'سكن',
            'car' => 'سيارة',
            'emergency' => 'طارئ',
            'education' => 'تعليمي',
            default => $type
        };
    }

    /**
     * عرض صفحة الطباعة لسجل قروض موظف محدد.
     */
    public function printEmployee($employeeId)
    {
        $employee = Employee::findOrFail($employeeId);
        $startDate = Carbon::now()->subMonths(6)->startOfMonth();
        $endDate = Carbon::now()->endOfMonth();

        $loans = $employee->loans()
            ->whereBetween('created_at', [$startDate, $endDate])
            ->orderBy('created_at', 'desc')
            ->get();

        return view('loans.print-employee', compact('employee', 'loans'));
    }
}