<?php

namespace App\Http\Controllers;

use App\Models\Leave;
use App\Models\Employee;
use App\Models\Activity;
use App\Notifications\LeaveRequestSubmittedNotification;
use Illuminate\Http\Request;
use App\Http\Requests\LeaveRequest;
use App\Services\LeaveBalanceService;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class LeaveController extends Controller
{
    /**
     * 🟦 عرض قائمة الإجازات.
     * إذا كان المستخدم موظفًا، يُعرض فقط طلباته.
     * خلاف ذلك، يُعرض جميع الطلبات (لـ admin, hr).
     */
    public function index()
    {
        $this->authorize('viewAny', Leave::class);

        $authenticatedUser = auth()->user();

        // ✅ التحقق من وجود العلاقة مع Employee
        if ($authenticatedUser->hasRole('employee') && !$authenticatedUser->employee) {
            // إذا كان المستخدم لديه دور 'employee' ولكن لا يوجد سجل موظف مرتبط،
            // يمكن عرض رسالة خطأ أو إعادة توجيه.
            return view('leaves.index', [
                'leaves' => collect(), // ✅ تأكد من تمرير Collection فارغة من نوع Paginated (أو استخدم paginate(0))
                'employees' => collect(),
                'employeesWithLeaveBalance' => collect(),
                'stats' => [
                    'total' => 0,
                    'approved' => 0,
                    'pending' => 0,
                    'rejected' => 0,
                    'cancelled' => 0,
                    'modified' => 0,
                ],
                'error' => 'Your employee profile is not linked to your user account. Please contact HR.'
            ]);
        }

        if ($authenticatedUser->hasRole('employee')) {
            // ✅ الآن نكون متأكدين أن $authenticatedUser->employee ليست null
            $query = Leave::with(['employee', 'employee.user'])->where('employee_id', $authenticatedUser->employee->id);
        } else {
            // جلب جميع الطلبات للإدارة (admin, hr)
            $query = Leave::with(['employee', 'employee.user']);
        }

        // 🔍 البحث
        if ($search = request('search')) {
            $query->whereHas('employee', function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('employee_id', 'like', "%{$search}%");
            });
        }

        // 🧩 الفلاتر (متوفرة فقط للإدارة)
        if (!$authenticatedUser->hasRole('employee')) {
            if ($employeeId = request('employee_id')) $query->where('employee_id', $employeeId);
            if ($status = request('status')) $query->where('status', $status);
            if ($type = request('type')) $query->whereHas('leaveTypeModel', function ($q) use ($type) { $q->where('code', $type); });
            if (request('month')) $query->whereMonth('start_date', request('month'));
            if (request('year')) $query->whereYear('start_date', request('year'));
        }

        $leaves = $query->latest()->paginate(15); // ✅ $leaves هو الآن Paginator

        // جلب الموظفين لقائمة الفلترة (متوفرة فقط للإدارة)
        if ($authenticatedUser->hasRole('employee')) {
            $employees = collect([$authenticatedUser->employee]); // فقط موظف المستخدم الحالي
        } else {
            $employees = Employee::orderBy('name')->get();
        }

        // 🧮 حساب الرصيد التراكمي الصحيح لكل موظف (من تاريخ التعيين فقط)
        // هذا الجزء مفيد للإدارة، للموظف يمكن حسابه بشكل مبسط في عرضه
        if ($authenticatedUser->hasRole('employee')) {
            // للموظف، لا نحتاج عرض جدول الموظفين
            $employeesWithLeaveBalance = collect([$authenticatedUser->employee]);
            $employeesWithLeaveBalance->map(function ($employee) {
                $hireDate = Carbon::parse($employee->hire_date);
                $now = Carbon::now();

                if ($hireDate->greaterThan($now)) {
                    $monthsWorked = 0;
                } else {
                    $monthsWorked = $hireDate->diffInMonths($now);
                    if ($now->day >= $hireDate->day) {
                        $monthsWorked += 1;
                    }
                }

                $monthsWorked = max(0, min($monthsWorked, 12));

                $usedDays = $employee->leaves()
                    ->whereIn('status', ['approved', 'modified'])
                    ->ofCodes(['annual', 'emergency'])
                    ->sum(DB::raw('COALESCE(deductible_days, days_count)'));

                $granted = round(($employee->monthly_leave_days_allowed ?? 0) * $monthsWorked, 2);
                $remaining = $granted - $usedDays;

                $employee->granted_leave_days = $granted;
                $employee->used_leave_days = $usedDays;
                $employee->remaining_leave_days = $remaining;

                return $employee;
            });
        } else {
            // للإدارة، نحتفظ بالحساب الكامل
            $employeesWithLeaveBalance = $employees->map(function ($employee) {
                $hireDate = Carbon::parse($employee->hire_date);
                $now = Carbon::now();

                if ($hireDate->greaterThan($now)) {
                    $monthsWorked = 0;
                } else {
                    $monthsWorked = $hireDate->diffInMonths($now);
                    if ($now->day >= $hireDate->day) {
                        $monthsWorked += 1;
                    }
                }

                $monthsWorked = max(0, min($monthsWorked, 12));

                $usedDays = $employee->leaves()
                    ->whereIn('status', ['approved', 'modified'])
                    ->ofCodes(['annual', 'emergency'])
                    ->sum(DB::raw('COALESCE(deductible_days, days_count)'));

                $granted = round(($employee->monthly_leave_days_allowed ?? 0) * $monthsWorked, 2);
                $remaining = $granted - $usedDays;

                $employee->granted_leave_days = $granted;
                $employee->used_leave_days = $usedDays;
                $employee->remaining_leave_days = $remaining;

                return $employee;
            });
        }

        // 📊 العدادات - ✅ لا تستخدم $leaves->get() هنا، بل استخدم $leaves->total() أو استخدم $query للحصول على العدادات
        if ($authenticatedUser->hasRole('employee')) {
            // العدادات الخاصة بالموظف (طلباته فقط) - ✅ استخدم $query على $leaves الأصلي
            $stats = [
                'total'     => $leaves->total(), // ✅ استخدم total() من Paginator
                'approved'  => $query->where('status', 'approved')->count(), // ✅ استخدم $query الأصلي
                'pending'   => $query->where('status', 'pending')->count(), // ✅ استخدم $query الأصلي
                'rejected'  => $query->where('status', 'rejected')->count(), // ✅ استخدم $query الأصلي
                'cancelled' => $query->where('status', 'cancelled')->count(), // ✅ استخدم $query الأصلي
                'modified'  => $query->where('status', 'modified')->count(), // ✅ استخدم $query الأصلي
            ];
        } else {
            // العدادات العامة (لكل الإجازات) - ✅ استخدم Leave:: مباشرة
            $stats = [
                'total'     => Leave::count(),
                'approved'  => Leave::where('status', 'approved')->count(),
                'pending'   => Leave::where('status', 'pending')->count(),
                'rejected'  => Leave::where('status', 'rejected')->count(),
                'cancelled' => Leave::where('status', 'cancelled')->count(),
                'modified'  => Leave::where('status', 'modified')->count(),
            ];
        }

        // جلب أنواع الإجازات لعرض فلتر الأنواع
        $leaveTypes = \App\Models\LeaveType::orderBy('name')->get();

        // expose stats as individual variables for the Blade view (backwards compatibility)
        $totalLeaves = $stats['total'] ?? 0;
        $approvedLeaves = $stats['approved'] ?? 0;
        $pendingLeaves = $stats['pending'] ?? 0;
        $rejectedLeaves = $stats['rejected'] ?? 0;
        $cancelledLeaves = $stats['cancelled'] ?? 0;
        $modifiedLeaves = $stats['modified'] ?? 0;

        return view('leaves.index', compact(
            'leaves', 'employees', 'employeesWithLeaveBalance', 'stats', 'leaveTypes',
            'totalLeaves', 'approvedLeaves', 'pendingLeaves', 'rejectedLeaves', 'cancelledLeaves', 'modifiedLeaves'
        ));
        // أو بشكل أكثر وضوحًا:
        // return view('leaves.index', [
        //     'leaves' => $leaves, // المتغير الأصلي من paginate()
        //     'employees' => $employees,
        //     'employeesWithLeaveBalance' => $employeesWithLeaveBalance,
        //     'stats' => $stats,
        // ]);
    }

    /**
     * 🟩 عرض نموذج إنشاء طلب إجازة جديد.
     * للموظف، يتم تحميل موظفه فقط.
     */
    public function create()
    {
        $this->authorize('create', Leave::class);

        $authenticatedUser = auth()->user();

        // تحميل الموظف الحالي مع جميع العلاقات المطلوبة
        if ($authenticatedUser->hasRole('admin') || $authenticatedUser->hasRole('hr') || $authenticatedUser->hasRole('finance')) {
            // للمديرين: عرض قائمة جميع الموظفين
            $employees = Employee::orderBy('name')->get();
        } else {
            // للموظفين العاديين: عرض موظفهم فقط
            $employee = $authenticatedUser->employee;
            if (!$employee) {
                abort(403, 'Your employee profile is not linked to your user account.');
            }
            $employees = collect([$employee]);
        }

        // جلب أنواع الإجازات من قاعدة البيانات لعرضها في النموذج
        $leaveTypes = \App\Models\LeaveType::orderBy('name')->get();

        return view('leaves.create', compact('employees', 'leaveTypes'));
    }

    /**
     * 🟨 حفظ طلب الإجازة.
     * يضمن أن الموظف لا يمكنه إنشاء طلب لشخص آخر.
     */
    public function store(LeaveRequest $request)
    {
        $this->authorize('create', Leave::class);
        // Validation moved to LeaveRequest (dynamic leave_type validation)

        $authenticatedUser = auth()->user();

        // التأكد من أن الموظف يطلب إجازة لنفسه فقط
        if ($authenticatedUser->hasRole('employee') && $authenticatedUser->employee->id !== (int)$request->employee_id) {
            abort(403, 'لا يمكنك إنشاء طلب إجازة لموظف آخر.');
        }

        // 🚫 منع التداخل في التواريخ
        $exists = Leave::where('employee_id', $request->employee_id)
            ->whereNotIn('status', ['rejected', 'cancelled'])
            ->where(function ($q) use ($request) {
                $q->whereBetween('start_date', [$request->start_date, $request->end_date])
                  ->orWhereBetween('end_date', [$request->start_date, $request->end_date])
                  ->orWhere(function ($q2) use ($request) {
                      $q2->where('start_date', '<=', $request->start_date)
                         ->where('end_date', '>=', $request->end_date);
                  });
            })->exists();

        if ($exists) {
            return back()->with('error', 'هناك طلب إجازة متداخل في نفس الفترة')->withInput();
        }

        $startDate = Carbon::parse($request->start_date)->startOfDay();
        $endDate   = Carbon::parse($request->end_date)->startOfDay();
        $daysCount = $startDate->diffInDays($endDate) + 1;

        $employee = Employee::findOrFail($request->employee_id);
        $hireDate = Carbon::parse($employee->hire_date);
        $now = Carbon::now();

        // ✅ حساب الأشهر الدقيقة (نفس منطق index)
        if ($hireDate->greaterThan($now)) {
            $monthsWorked = 0;
        } else {
            $monthsWorked = $hireDate->diffInMonths($now);
            if ($now->day >= $hireDate->day) {
                $monthsWorked += 1;
            }
        }

        $monthsWorked = max(0, min($monthsWorked, 12));
        $granted = round(($employee->monthly_leave_days_allowed ?? 0) * $monthsWorked, 2);

        $usedDays = $employee->leaves()
            ->whereIn('status', ['approved', 'modified'])
            ->whereIn('leave_type', ['annual', 'emergency'])
            ->sum(DB::raw('COALESCE(deductible_days, days_count)'));

        $remaining = $granted - $usedDays;

        // 💰 لو الإجازة تجاوزت الرصيد
        $actualLeaveType = $request->leave_type;
        if ($request->leave_type == 'annual' && $daysCount > $remaining && $remaining >= 0) {
            // We'll create a single leave record containing an internal breakdown
            // of paid/deductible vs unpaid deduction portions. The Leave model
            // computes excluded days and deductible_days on save; afterwards
            // we'll adjust the deductible part to consume remaining balance
            // and store the unpaid deductible portion in `unpaid_deducted_days`.

            $lt = \App\Models\LeaveType::where('code', 'annual')->first();
            $leave = Leave::create([
                'employee_id'     => $employee->id,
                'leave_type'      => 'annual',
                'leave_type_id'   => $lt?->id,
                'start_date'      => $request->start_date,
                'end_date'        => $request->end_date,
                'days_count'      => $daysCount,
                'reason'          => $request->reason,
                'contact_details' => $request->contact_details,
                'status'          => 'pending',
                // informational calendar split
                'paid_days'       => max(0, $remaining),
                'unpaid_days'     => max(0, $daysCount - $remaining),
            ]);

            // ensure excluded/deductible are computed in memory
            $leave->calculateExcludedDays();

            // total deductible days inside the requested span (after excluding offs/holidays)
            $totalDeductible = (int) ($leave->deductible_days ?? 0);

            // paid portion from balance consumes at most the remaining balance
            $paidDeductible = min((int)$remaining, $totalDeductible);
            $unpaidDeductible = max(0, $totalDeductible - $paidDeductible);

            // set the saved fields: deductible_days indicates days deducted from balance
            // unpaid_deducted_days indicates days to be deducted from salary by payroll
            $leave->deductible_days = $paidDeductible;
            $leave->unpaid_deducted_days = $unpaidDeductible;
            // keep deducted_days null until approval (LeaveBalanceService will set it on approve)
            $leave->save();

            return redirect()->route('leaves.index')
                ->with('success', 'تم إنشاء طلب إجازة واحد يحتوي على جزء مدفوع وجزء بدون راتب (مع تفصيل داخلي).');
        }

        // 🟢 الحالة العادية
        $lt = \App\Models\LeaveType::where('code', $actualLeaveType)->first();
        $leave = Leave::create([
            'employee_id'     => $employee->id,
            'leave_type'      => $actualLeaveType,
            'leave_type_id'   => $lt?->id,
            'start_date'      => $request->start_date,
            'end_date'        => $request->end_date,
            'days_count'      => $daysCount,
            'reason'          => $request->reason,
            'contact_details' => $request->contact_details,
            'status'          => 'pending',
        ]);
        if ($employee->user) {
            $employee->user->notify(new LeaveRequestSubmittedNotification($leave));
        }

        Activity::create([
            'user_id'      => auth()->id(),
            'description'  => 'created_leave_request',
            'subject_type' => Leave::class,
            'subject_id'   => $leave->id,
            'properties'   => [
                'employee_name' => $employee->name,
                'leave_type'    => $leave->leave_type,
                'leave_period'  => "{$leave->start_date} إلى {$leave->end_date}",
                'status'        => $leave->status,
            ],
        ]);

        // إذا أرسلت form مع خيار الموافقة التلقائية من قبل مدير/موارد بشرية/مالي
        if ($request->has('auto_approve') && $request->input('auto_approve')) {
            try {
                $service = app(LeaveBalanceService::class);
                // تحقق إذن الموافقة عبر policy إذا كان متاحًا
                if (auth()->user()->can('approve', $leave)) {
                    DB::beginTransaction();
                    $service->applyApproval($leave, auth()->id());
                    DB::commit();
                    return redirect()->route('leaves.index')->with('success', 'تم تقديم طلب الإجازة والموافقة عليه تلقائيًا');
                }
            } catch (\Exception $e) {
                DB::rollBack();
                // سجل الخطأ ثم عدّل الرسالة للمستخدم
                report($e);
                return redirect()->route('leaves.index')->with('warning', 'تم إنشاء الطلب، لكن حدث خطأ أثناء محاولة الموافقة التلقائية.');
            }
        }

        return redirect()->route('leaves.index')->with('success', 'تم تقديم طلب الإجازة بنجاح');
    }

    // ====================== EDIT ======================
    /**
     * عرض نموذج تعديل طلب إجازة.
     * يضمن أن الموظف لا يمكنه تعديل طلب لشخص آخر.
     */
    public function edit($id)
    {
        $leave = Leave::with('employee')->findOrFail($id);
        $this->authorize('update', $leave);

        $authenticatedUser = auth()->user();

        // التأكد من أن الموظف يعرض تعديل لطلبه الخاص فقط
        if ($authenticatedUser->hasRole('employee') && $authenticatedUser->employee->id !== $leave->employee_id) {
            abort(403, 'لا يمكنك تعديل طلب إجازة لموظف آخر.');
        }

        $employees = Employee::orderBy('name')->get();
        // جلب أنواع الإجازات من قاعدة البيانات
        $leaveTypes = \App\Models\LeaveType::orderBy('name')->get();

        $employee = $leave->employee;

        if (!$employee) {
            abort(404, 'الموظف المرتبط بطلب الإجازة غير موجود.');
        }

        $hireDate = Carbon::parse($employee->hire_date);
        $currentDate = Carbon::now();

        $lastMonthEnd = Carbon::createFromDate($currentDate->year, $currentDate->month, 1)->subDay();

        if ($hireDate->greaterThan($lastMonthEnd)) {
            $monthsWorked = 0;
        } else {
            $monthsWorked = $hireDate->diffInMonths($lastMonthEnd);
            if ($lastMonthEnd->day >= $hireDate->day) {
                $monthsWorked += 1;
            }
        }

        $monthsWorked = max(0, min($monthsWorked, 12));

        $granted = round(($employee->monthly_leave_days_allowed ?? 0) * $monthsWorked, 2);

        $usedDaysBeforeThisMonth = $employee->leaves()
            ->whereIn('status', ['approved', 'modified'])
            ->ofCodes(['annual', 'emergency'])
            ->where(function ($query) use ($lastMonthEnd) {
                $query->whereDate('start_date', '<=', $lastMonthEnd)
                      ->orWhereDate('end_date', '<=', $lastMonthEnd);
            })
            ->sum(DB::raw('COALESCE(deductible_days, days_count)'));

        $actualCumulativeBalance = $granted - $usedDaysBeforeThisMonth;

        $usedDaysThisMonth = $employee->leaves()
            ->whereIn('status', ['approved', 'modified'])
            ->ofCodes(['annual', 'emergency'])
            ->whereMonth('start_date', $currentDate->month)
            ->whereYear('start_date', $currentDate->year)
            ->sum(DB::raw('COALESCE(deductible_days, days_count)'));

        $newTotalUsed = $usedDaysBeforeThisMonth + $usedDaysThisMonth + ($leave->deductible_days ?? $leave->days_count);
        $balanceAfterLeave = $actualCumulativeBalance - $newTotalUsed;

        return view('leaves.edit', compact(
            'leave',
            'employees',
            'leaveTypes',
            'actualCumulativeBalance',
            'usedDaysThisMonth',
            'balanceAfterLeave'
        ));
    }

    // ====================== UPDATE ======================
    /**
     * تحديث طلب إجازة.
     * يضمن أن الموظف لا يمكنه تعديل طلب لشخص آخر.
     */
    public function update(LeaveRequest $request, Leave $leave)
    {
        // تأكد من أن $leave موجود وله معرّف صحيح
        if (!$leave || !$leave->id || $leave->id <= 0) {
            return redirect()->route('leaves.index')
                ->with('error', 'طلب الإجازة المطلوب غير موجود.');
        }

        $this->authorize('update', $leave);
        // Validation moved to LeaveRequest (dynamic leave_type validation)

        $authenticatedUser = auth()->user();

        // التأكد من أن الموظف يعدل طلبه الخاص فقط
        if ($authenticatedUser->hasRole('employee') && $authenticatedUser->employee->id !== $leave->employee_id) {
            abort(403, 'لا يمكنك تعديل طلب إجازة لموظف آخر.');
        }

        // التأكد من أن الموظف لا يغير صاحب الطلب إلى آخر
        if ($authenticatedUser->hasRole('employee') && $leave->employee_id !== (int)$request->employee_id) {
            abort(403, 'لا يمكنك تغيير الموظف المرتبط بالطلب.');
        }

        // 🚫 لا نتحقق من التداخل عند التعديل، لأنك تعدل طلب موجود بالفعل
        // إذا أردت منع التعديل إلى تواريخ متداخلة، استخدم منطق مختلف

        $startDate = Carbon::parse($request->start_date)->startOfDay();
        $endDate = Carbon::parse($request->end_date)->startOfDay();
        $daysCount = $startDate->diffInDays($endDate) + 1;

        $lt = \App\Models\LeaveType::where('code', $request->leave_type)->first();

        // ✅ تحضير بيانات التحديث
        $updateData = [
            'leave_type'      => $request->leave_type,
            'leave_type_id'   => $lt?->id,
            'start_date'      => $startDate->format('Y-m-d'),
            'end_date'        => $endDate->format('Y-m-d'),
            'days_count'      => $daysCount,
            'reason'          => $request->reason ?? null,
            'contact_details' => $request->contact_details ?? null,
        ];

        // أضف status فقط إذا كان موجوداً في الطلب
        if ($request->has('status')) {
            $updateData['status'] = $request->status;
        }

        // تسجيل بيانات التحديث للمساعدة في التشخيص
        \Log::debug('Leave update requested', ['leave_id' => $leave->id, 'payload' => $updateData, 'current_leave' => $leave->toArray()]);

        // إذا كانت الإجازة مُعتمدة بالفعل، فسنحتاج لإجراء تعديل خاص يعكس تغييرات الرصيد
        if ($leave->status === 'approved') {
            // اسمح فقط للأدوار الإدارية أو من يملك صلاحية الموافقة بتعديل إجازة معتمدة
            $authenticatedUser = auth()->user();
            if (! ($authenticatedUser->hasRole('admin') || $authenticatedUser->hasRole('hr') || $authenticatedUser->hasRole('finance') || $authenticatedUser->can('approve', $leave)) ) {
                return redirect()->back()->with('error', 'لا تملك الصلاحية لتعديل إجازة معتمدة. تواصل مع الموارد البشرية.');
            }

            try {
                $service = app(\App\Services\LeaveBalanceService::class);
                $service->adjustApproval($leave, $updateData, auth()->id());
                // capture optional note from request to include in activity log
                $approvalNote = $request->input('approval_modify_note');
                \Log::debug('Adjusted approved leave', ['leave_id' => $leave->id, 'leave_after' => $leave->toArray()]);
            } catch (\Throwable $e) {
                \Log::error("Adjust approved leave failed for id {$leave->id}: " . $e->getMessage());
                return redirect()->back()->withInput()->with('error', 'حدث خطأ أثناء تعديل الإجازة المعتمدة.');
            }
        } else {
            // استخدم Eloquent update مباشرة (يُحفّز Events بشكل صحيح)
            try {
                \Log::debug('Attempting Eloquent update for leave', ['leave_id' => $leave->id, 'updateData' => $updateData]);
                $updated = $leave->update($updateData);
                if (!$updated) {
                    \Log::warning("Leave update returned false for leave id: {$leave->id}");
                    return redirect()->back()->withInput()->with('error', 'لم يتم حفظ التغييرات، يرجى المحاولة مرة أخرى.');
                }
                \Log::debug('Leave updated successfully', ['leave_id' => $leave->id, 'leave_after' => $leave->toArray()]);
            } catch (\Throwable $e) {
                \Log::error("Leave update failed for id {$leave->id}: " . $e->getMessage());
                return redirect()->back()->withInput()->with('error', 'حدث خطأ أثناء حفظ التغييرات.');
            }
        }

        // الحصول على اسم الموظف بشكل آمن
        $employeeName = $leave->employee?->name ?? 'موظف غير معروف';

        Activity::create([
            'user_id'      => auth()->id(),
            'description'  => 'updated_leave',
            'subject_type' => Leave::class,
            'subject_id'   => $leave->id,
            'properties'   => [
                'details' => "تم تعديل طلب الإجازة #{$leave->id} لـ {$employeeName}",
                'approval_note' => $approvalNote ?? null,
            ],
        ]);

        return redirect()->route('leaves.index')
            ->with('success', "تم تعديل طلب الإجازة لـ {$employeeName} بنجاح.");
    }

    // ====================== APPROVE ======================
    public function approve(Leave $leave)
    {
        $this->authorize('approve', $leave);

        if ($leave->status !== 'pending') {
            return redirect()->route('leaves.index')->with('error', 'لا يمكن الموافقة على طلب غير معلق.');
        }

        // استخدم الخدمة لحساب التأثيرات وتطبيقها
        $service = new \App\Services\LeaveBalanceService();

        DB::beginTransaction();
        try {
            $service->applyApproval($leave, auth()->id());

            Activity::create([
                'user_id'      => auth()->id(),
                'description'  => 'approved_leave',
                'subject_type' => Leave::class,
                'subject_id'   => $leave->id,
                'properties'   => [
                    'details' => "تمت الموافقة على طلب الإجازة #{$leave->id} لـ {$leave->employee->name}",
                ],
            ]);

            if ($leave->employee->user) {
                $leave->employee->user->notify(new \App\Notifications\LeaveApprovedNotification($leave));
            }

            DB::commit();

            return redirect()->route('leaves.index')
                ->with('success', "تمت الموافقة على طلب الإجازة لـ {$leave->employee->name} بنجاح.");
        } catch (\Throwable $e) {
            DB::rollBack();
            \Log::error('Approve leave failed: ' . $e->getMessage());
            return redirect()->route('leaves.index')->with('error', 'حدث خطأ أثناء الموافقة على الطلب.');
        }
    }

    // ====================== REJECT ======================
    public function reject(Request $request, Leave $leave)
    {
        $this->authorize('reject', $leave);

        if ($leave->status !== 'pending') {
            return redirect()->route('leaves.index')->with('error', 'لا يمكن رفض طلب غير معلق.');
        }

        $request->validate([
            'rejection_reason' => 'nullable|string|max:500',
        ]);

        $leave->update([
            'status' => 'rejected',
            'rejection_reason' => $request->rejection_reason,
            'rejected_by' => auth()->id(),
            'rejected_at' => now(),
        ]);

        Activity::create([
            'user_id'      => auth()->id(),
            'description'  => 'rejected_leave',
            'subject_type' => Leave::class,
            'subject_id'   => $leave->id,
            'properties'   => [
                'details' => "تم رفض طلب الإجازة #{$leave->id} لـ {$leave->employee->name}",
                'rejection_reason' => $request->rejection_reason,
            ],
        ]);

        if ($leave->employee->user) {
            // يمكنك إنشاء إشعار مخصص مثل LeaveRejectedNotification
            $leave->employee->user->notify(new \App\Notifications\LeaveRejectedNotification($leave));
        }

        return redirect()->route('leaves.index')
            ->with('success', "تم رفض طلب الإجازة لـ {$leave->employee->name} بنجاح.");
    }

    // ====================== CANCEL ======================
    public function cancel(Leave $leave)
    {
        $this->authorize('cancel', $leave);

        $authenticatedUser = auth()->user();

        // التأكد من أن الموظف يلغي طلبه الخاص فقط
        if ($authenticatedUser->hasRole('employee') && $authenticatedUser->employee->id !== $leave->employee_id) {
            abort(403, 'لا يمكنك إلغاء طلب إجازة لموظف آخر.');
        }

        if (!in_array($leave->status, ['pending', 'approved'])) {
            return redirect()->route('leaves.index')->with('error', 'لا يمكن إلغاء هذا الطلب.');
        }

        $leave->update([
            'status' => 'cancelled',
        ]);

        Activity::create([
            'user_id'      => auth()->id(),
            'description'  => 'cancelled_leave',
            'subject_type' => Leave::class,
            'subject_id'   => $leave->id,
            'properties'   => [
                'details' => "تم إلغاء طلب الإجازة #{$leave->id} لـ {$leave->employee->name}",
            ],
        ]);

        return redirect()->route('leaves.index')
            ->with('success', "تم إلغاء طلب الإجازة لـ {$leave->employee->name} بنجاح.");
    }

    // ====================== DESTROY (DELETE) ======================
    public function destroy(Leave $leave)
    {
        $this->authorize('delete', $leave);

        $authenticatedUser = auth()->user();

        // التأكد من أن الموظف يحذف طلبه الخاص فقط
        if ($authenticatedUser->hasRole('employee') && $authenticatedUser->employee->id !== $leave->employee_id) {
            abort(403, 'لا يمكنك حذف طلب إجازة لموظف آخر.');
        }

        if (!in_array($leave->status, ['pending'])) {
            return redirect()->route('leaves.index')->with('error', 'لا يمكن حذف هذا الطلب.');
        }

        $leave->delete();

        Activity::create([
            'user_id'      => auth()->id(),
            'description'  => 'deleted_leave',
            'subject_type' => Leave::class,
            'subject_id'   => $leave->id,
            'properties'   => [
                'details' => "تم حذف طلب الإجازة #{$leave->id} لـ {$leave->employee->name}",
            ],
        ]);

        return redirect()->route('leaves.index')
            ->with('success', "تم حذف طلب الإجازة لـ {$leave->employee->name} بنجاح.");
    }

    // ====================== SHOW ======================
    /**
     * عرض تفاصيل إجازة.
     * يضمن أن الموظف لا يمكنه عرض تفاصيل إجازة لشخص آخر.
     */
    public function show($id)
    {
        $leave = Leave::with('employee')->findOrFail($id);
        $this->authorize('view', $leave);

        $authenticatedUser = auth()->user();

        // التأكد من أن الموظف يعرض تفاصيل لطلبه الخاص فقط
        if ($authenticatedUser->hasRole('employee') && $authenticatedUser->employee->id !== $leave->employee_id) {
            abort(403, 'لا يمكنك عرض تفاصيل إجازة لموظف آخر.');
        }

        $employee = $leave->employee;
        $hireDate = Carbon::parse($employee->hire_date);
        $now = Carbon::now();

        if ($hireDate->greaterThan($now)) {
            $monthsWorked = 0;
        } else {
            $monthsWorked = $hireDate->diffInMonths($now);
            if ($now->day >= $hireDate->day) {
                $monthsWorked += 1;
            }
        }

        $monthsWorked = max(0, min($monthsWorked, 12));

        $usedDays = $employee->leaves()
            ->whereIn('status', ['approved', 'modified'])
            ->ofCodes(['annual', 'emergency'])
            ->sum(DB::raw('COALESCE(deductible_days, days_count)'));

        $granted = round(($employee->monthly_leave_days_allowed ?? 0) * $monthsWorked, 2);
        $remaining = $granted - $usedDays;

        return view('leaves.show', compact('leave', 'employee', 'granted', 'usedDays', 'remaining'));
    }

    // ====================== GET LEAVE BALANCE (JSON) ======================
    public function getLeaveBalance($employeeId)
    {
        $employee = Employee::findOrFail($employeeId);
        $hireDate = Carbon::parse($employee->hire_date);
        $now = Carbon::now();

        if ($hireDate->greaterThan($now)) {
            $monthsWorked = 0;
        } else {
            $monthsWorked = $hireDate->diffInMonths($now);
            if ($now->day >= $hireDate->day) {
                $monthsWorked += 1;
            }
        }

        $monthsWorked = max(0, min($monthsWorked, 12));

        $usedDays = $employee->leaves()
            ->whereIn('status', ['approved', 'modified'])
            ->whereIn('leave_type', ['annual', 'emergency'])
            ->sum(DB::raw('COALESCE(deductible_days, days_count)'));

        $granted = round(($employee->monthly_leave_days_allowed ?? 0) * $monthsWorked, 2);
        $remaining = $granted - $usedDays;

        return response()->json([
            'granted'   => $granted,
            'used'      => $usedDays,
            'remaining' => $remaining,
        ]);
    }

    /**
     * عرض دليل أنواع الإجازات
     */
    public function guide()
    {
        $this->authorize('viewAny', Leave::class);
        
        return view('leaves.guide');
    }
}