<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\LeaveBalanceChange;
use App\Models\Employee;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Carbon\Carbon;

class LeaveBalanceChangeController extends Controller
{
    public function __construct()
    {
        $this->authorizeResource(LeaveBalanceChange::class, 'leave_balance_change');
    }

    public function index(): View
    {
        $changes = LeaveBalanceChange::with('employee')->orderByDesc('effective_from')->paginate(30);
        return view('leave_balance_changes.index', compact('changes'));
    }

    public function create(): View
    {
        $employees = Employee::orderBy('name')->get();
        return view('leave_balance_changes.create', compact('employees'));
    }

    public function store(Request $request): RedirectResponse
    {
        $this->validate($request, [
            'employee_id' => 'required|exists:employees,id',
            'kind' => 'required|in:monthly,annual',
            'days_change' => 'required|integer',
            'effective_from' => 'required|date',
            'note' => 'nullable|string|max:1000',
        ]);

        LeaveBalanceChange::create([
            'employee_id' => $request->employee_id,
            'kind' => $request->kind,
            'days_change' => (int)$request->days_change,
            'effective_from' => Carbon::parse($request->effective_from)->toDateString(),
            'note' => $request->note,
            'created_by' => auth()->id(),
        ]);

        return redirect()->route('leave-balance-changes.index')->with('success', 'تم إضافة تعديل الرصيد بنجاح.');
    }

    public function edit(LeaveBalanceChange $leave_balance_change): View
    {
        $employees = Employee::orderBy('name')->get();
        return view('leave_balance_changes.edit', ['change' => $leave_balance_change, 'employees' => $employees]);
    }

    public function update(Request $request, LeaveBalanceChange $leave_balance_change): RedirectResponse
    {
        $this->validate($request, [
            'employee_id' => 'required|exists:employees,id',
            'kind' => 'required|in:monthly,annual',
            'days_change' => 'required|integer',
            'effective_from' => 'required|date',
            'note' => 'nullable|string|max:1000',
        ]);

        $leave_balance_change->update([
            'employee_id' => $request->employee_id,
            'kind' => $request->kind,
            'days_change' => (int)$request->days_change,
            'effective_from' => Carbon::parse($request->effective_from)->toDateString(),
            'note' => $request->note,
        ]);

        return redirect()->route('leave-balance-changes.index')->with('success', 'تم تحديث تعديل الرصيد بنجاح.');
    }

    public function destroy(LeaveBalanceChange $leave_balance_change): RedirectResponse
    {
        $leave_balance_change->delete();
        return redirect()->route('leave-balance-changes.index')->with('success', 'تم حذف تعديل الرصيد بنجاح.');
    }
}
