<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\HolidayYear;
use App\Models\Holiday;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;

class HolidayYearController extends Controller
{
    public function index(): View
    {
        // Ensure years from 2025 to 2100 exist so the UI can show the full range
        $start = 2025;
        $end = 2100;
        for ($y = $start; $y <= $end; $y++) {
            HolidayYear::firstOrCreate(['year' => $y]);
        }

        $dbYears = HolidayYear::orderBy('year')->get()->keyBy('year');

        $display = [];
        for ($y = $start; $y <= $end; $y++) {
            if (isset($dbYears[$y])) {
                $display[] = $dbYears[$y];
            } else {
                // placeholder object to display empty year row
                $obj = new HolidayYear();
                $obj->id = null;
                $obj->year = $y;
                $obj->name = null;
                $display[] = $obj;
            }
        }

        // sort descending for display consistency
        $display = collect($display)->sortByDesc('year');

        return view('holiday_years.index', ['years' => $display]);
    }

    /**
     * Generate holiday years up to a given end year (default 2100).
     */
    public function generate(Request $request): RedirectResponse
    {
        $end = (int) $request->input('end_year', 2100);
        $start = (int) $request->input('start_year', date('Y'));
        if ($end < $start) {
            return redirect()->back()->with('error', 'سنة النهاية يجب أن تكون أكبر من أو تساوي سنة البداية.');
        }

        for ($y = $start; $y <= $end; $y++) {
            HolidayYear::updateOrCreate(['year' => $y], ['name' => null]);
        }

        return redirect()->route('holiday-years.index')->with('success', "تم إنشاء سنوات العطلات من $start إلى $end.");
    }

    public function create(): View
    {
        return view('holiday_years.create');
    }

    public function store(Request $request): RedirectResponse
    {
        $this->validate($request, [
            'year' => 'required|integer|unique:holiday_years,year',
            'name' => 'nullable|string|max:255',
        ]);

        HolidayYear::create($request->only(['year','name']));

        return redirect()->route('holiday-years.index')->with('success', 'سنة العطلات أضيفت.');
    }

    public function show(HolidayYear $holidayYear): View
    {
        $holidays = $holidayYear->holidays()->orderBy('date')->get();
        return view('holiday_years.show', compact('holidayYear','holidays'));
    }

    public function edit(HolidayYear $holidayYear): View
    {
        return view('holiday_years.edit', compact('holidayYear'));
    }

    public function update(Request $request, HolidayYear $holidayYear): RedirectResponse
    {
        $this->validate($request, [
            'year' => 'required|integer|unique:holiday_years,year,'.$holidayYear->id,
            'name' => 'nullable|string|max:255',
        ]);

        $holidayYear->update($request->only(['year','name']));

        return redirect()->route('holiday-years.index')->with('success', 'تم تحديث سنة العطلات.');
    }

    public function destroy(HolidayYear $holidayYear): RedirectResponse
    {
        // Detach year from holidays (keep holidays if desired)
        $holidayYear->holidays()->update(['holiday_year_id' => null]);
        $holidayYear->delete();
        return redirect()->route('holiday-years.index')->with('success', 'تم حذف سنة العطلات.');
    }
}
