<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Holiday;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Carbon\Carbon;

class HolidayController extends Controller
{
    public function __construct()
    {
        $this->authorizeResource(Holiday::class, 'holiday');
    }

    public function index(Request $request): View
    {
        $query = Holiday::query();
        
        // Filter by year if provided
        if ($request->filled('year')) {
            $year = (int) $request->input('year');
            $query->whereYear('date', $year);
        }
        
        // Filter by month if provided (requires year to be meaningful)
        if ($request->filled('month') && $request->filled('year')) {
            $month = (int) $request->input('month');
            $query->whereMonth('date', $month);
        }
        
        $holidays = $query->orderBy('date', 'desc')->paginate(20);
        return view('holidays.index', compact('holidays'));
    }

    public function create(): View
    {
        // provide list of holiday years to optionally attach the holiday
        $years = \App\Models\HolidayYear::orderBy('year', 'desc')->get();
        return view('holidays.create', compact('years'));
    }

    public function store(Request $request): RedirectResponse
    {
        // If user provides month/day and optionally a holiday_year_id, compose full date
        $this->validate($request, [
            'month' => 'required|integer|min:1|max:12',
            'day' => 'required|integer|min:1|max:31',
            'name' => 'nullable|string|max:255',
            'is_paid' => 'sometimes|boolean',
            'holiday_year_id' => 'nullable|exists:holiday_years,id',
        ]);

        // determine year: from selected holiday_year or current year
        $year = null;
        if ($request->filled('holiday_year_id')) {
            $hy = \App\Models\HolidayYear::find($request->input('holiday_year_id'));
            $year = $hy ? $hy->year : null;
        }
        $year = $year ?: Carbon::now()->year;

        // validate the composed date
        $month = (int) $request->input('month');
        $day = (int) $request->input('day');
        if (!checkdate($month, $day, $year)) {
            return back()->withInput()->withErrors(['date' => 'التاريخ غير صالح (اليوم/الشهر/السنة غير متوافق).']);
        }

        $date = Carbon::createFromDate($year, $month, $day)->toDateString();

        Holiday::create([
            'date' => $date,
            'name' => $request->name,
            // ensure boolean true/false (checkbox may be absent when unchecked)
            'is_paid' => $request->boolean('is_paid'),
            'holiday_year_id' => $request->input('holiday_year_id') ?: null,
        ]);

        return redirect()->route('holidays.index')->with('success', 'تم إضافة العطلة بنجاح.');
    }

    public function edit(Holiday $holiday): View
    {
        return view('holidays.edit', compact('holiday'));
    }

    public function update(Request $request, Holiday $holiday): RedirectResponse
    {
        $this->validate($request, [
            'month' => 'required|integer|min:1|max:12',
            'day' => 'required|integer|min:1|max:31',
            'name' => 'nullable|string|max:255',
            'is_paid' => 'sometimes|boolean',
            'holiday_year_id' => 'nullable|exists:holiday_years,id',
        ]);

        // determine year: from selected holiday_year or current year
        $year = null;
        if ($request->filled('holiday_year_id')) {
            $hy = \App\Models\HolidayYear::find($request->input('holiday_year_id'));
            $year = $hy ? $hy->year : null;
        }
        $year = $year ?: Carbon::now()->year;

        $month = (int) $request->input('month');
        $day = (int) $request->input('day');
        if (!checkdate($month, $day, $year)) {
            return back()->withInput()->withErrors(['date' => 'التاريخ غير صالح (اليوم/الشهر/السنة غير متوافق).']);
        }

        $date = Carbon::createFromDate($year, $month, $day)->toDateString();

        $holiday->update([
            'date' => $date,
            'name' => $request->name,
            // ensure boolean true/false
            'is_paid' => $request->boolean('is_paid'),
            'holiday_year_id' => $request->input('holiday_year_id') ?: null,
        ]);

        return redirect()->route('holidays.index')->with('success', 'تم تحديث العطلة بنجاح.');
    }

    public function destroy(Holiday $holiday): RedirectResponse
    {
        $holiday->delete();
        return redirect()->route('holidays.index')->with('success', 'تم حذف العطلة بنجاح.');
    }
}
