<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Employee;
use App\Models\EmployeeSalaryHistory;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class EmployeeSalaryHistoryController extends Controller
{
    public function index(Employee $employee)
    {
        $this->authorize('view', $employee);
        $histories = $employee->salaryHistories()->orderByDesc('effective_from')->get();
        return view('employees.partials.salary_histories', compact('employee', 'histories'));
    }

    public function store(Request $request, Employee $employee)
    {
        $this->authorize('update', $employee);
        $v = Validator::make($request->all(), [
            'effective_from' => 'required|date',
            'base_salary' => 'required|numeric|min:0',
            'incentives' => 'nullable|numeric|min:0',
            'overtime_rate' => 'nullable|numeric|min:0',
            'monthly_leave_entitlement' => 'nullable|numeric|min:0',
            'annual_leave_entitlement' => 'nullable|numeric|min:0',
            'component_names' => 'nullable|array',
            'component_names.*' => 'nullable|string|max:255',
            'component_values' => 'nullable|array',
            'component_values.*' => 'nullable|numeric|min:0',
            'note' => 'nullable|string|max:1000',
        ]);

        if ($v->fails()) {
            return redirect()->back()->withErrors($v)->withInput();
        }

        $data = $v->validated();
        
        // تحويل effective_from إلى بداية الشهر
        $effectiveFrom = Carbon::parse($data['effective_from'])->startOfMonth();
        
        // بناء مصفوفة المكونات من الحقول المنفصلة
        $components = [];
        if (!empty($data['component_names']) && !empty($data['component_values'])) {
            $names = $data['component_names'];
            $values = $data['component_values'];
            
            foreach ($names as $index => $name) {
                if (!empty($name) && isset($values[$index]) && $values[$index] > 0) {
                    $components[trim($name)] = (float) $values[$index];
                }
            }
        }

        EmployeeSalaryHistory::create([
            'employee_id' => $employee->id,
            'effective_from' => $effectiveFrom->toDateString(),
            'base_salary' => $data['base_salary'],
            'incentives' => $data['incentives'] ?? 0,
            'overtime_rate' => $data['overtime_rate'] ?? null,
            'monthly_leave_entitlement' => $data['monthly_leave_entitlement'] ?? null,
            'annual_leave_entitlement' => $data['annual_leave_entitlement'] ?? null,
            'components' => $components,
            'created_by' => auth()->id(),
            'note' => $data['note'] ?? null,
        ]);

        return redirect()->back()->with('success', 'تم إضافة سجل التعويض بنجاح. سيتم تطبيقه من بداية الشهر المحدد.');
    }

    public function edit(Employee $employee, EmployeeSalaryHistory $history)
    {
        $this->authorize('update', $employee);
        return view('employees.partials.salary_histories', compact('employee', 'history'));
    }

    public function update(Request $request, Employee $employee, EmployeeSalaryHistory $history)
    {
        $this->authorize('update', $employee);
        $v = Validator::make($request->all(), [
            'effective_from' => 'required|date',
            'base_salary' => 'required|numeric|min:0',
            'incentives' => 'nullable|numeric|min:0',
            'overtime_rate' => 'nullable|numeric|min:0',
            'monthly_leave_entitlement' => 'nullable|numeric|min:0',
            'annual_leave_entitlement' => 'nullable|numeric|min:0',
            'component_names' => 'nullable|array',
            'component_names.*' => 'nullable|string|max:255',
            'component_values' => 'nullable|array',
            'component_values.*' => 'nullable|numeric|min:0',
            'note' => 'nullable|string|max:1000',
        ]);

        if ($v->fails()) {
            return redirect()->back()->withErrors($v)->withInput();
        }

        $data = $v->validated();
        
        // تحويل effective_from إلى بداية الشهر
        $effectiveFrom = Carbon::parse($data['effective_from'])->startOfMonth();
        
        // بناء مصفوفة المكونات من الحقول المنفصلة
        $components = [];
        if (!empty($data['component_names']) && !empty($data['component_values'])) {
            $names = $data['component_names'];
            $values = $data['component_values'];
            
            foreach ($names as $index => $name) {
                if (!empty($name) && isset($values[$index]) && $values[$index] > 0) {
                    $components[trim($name)] = (float) $values[$index];
                }
            }
        }

        $history->update([
            'effective_from' => $effectiveFrom->toDateString(),
            'base_salary' => $data['base_salary'],
            'incentives' => $data['incentives'] ?? 0,
            'overtime_rate' => $data['overtime_rate'] ?? null,
            'monthly_leave_entitlement' => $data['monthly_leave_entitlement'] ?? null,
            'annual_leave_entitlement' => $data['annual_leave_entitlement'] ?? null,
            'components' => $components,
            'note' => $data['note'] ?? null,
        ]);

        return redirect()->back()->with('success', 'تم تحديث سجل التعويض بنجاح.');
    }

    public function destroy(Employee $employee, EmployeeSalaryHistory $history)
    {
        $this->authorize('update', $employee);
        $history->delete();
        return redirect()->back()->with('success', 'تم حذف سجل التعويض.');
    }
}
