<?php

namespace App\Http\Controllers;

use App\Models\Employee;
use App\Models\Department;
use App\Models\Position;
use App\Models\Setting;
use App\Models\Payroll; // ✅ هذا ما تحتاجه

use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Carbon\Carbon;
use App\Http\Requests\EmployeeRequest;
use App\Models\LeaveType;
use App\Models\Leave;
use App\Models\Holiday;
use App\Models\Attendance;
use App\Models\Deduction;

class EmployeeController extends Controller
{
    /**
     * عرض قائمة الموظفين مع إمكانية البحث والتصفية.
     *
     * @return \Illuminate\View\View
     */
    public function index(): View
    {
        $this->authorize('viewAny', Employee::class);

        $authenticatedUser = auth()->user();
        $isEmployee = $authenticatedUser->hasRole('employee');

        $query = Employee::with(['department', 'position']);

        if ($isEmployee && $authenticatedUser->employee) {
            $query->where('id', $authenticatedUser->employee->id);
        } else {
            // للمديرين: الافتراضي عرض الموظفين النشطين فقط
            // لكن إذا حدد حالة معينة، استخدمها
            if ($status = request('status')) {
                $query->where('status', $status);
            } else {
                $query->where('status', 'active');
            }
            
            if ($search = request('search')) {
                $query->where(function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('employee_id', 'like', "%{$search}%")
                      ->orWhere('email', 'like', "%{$search}%");
                });
            }
            if ($departmentId = request('department_id')) {
                $query->where('department_id', $departmentId);
            }
        }

        $employees = $query->orderBy('name')->paginate(15)->appends(request()->query());
        $departments = Department::orderBy('name')->get();
        $positions = Position::orderBy('name')->get();
        $currency_name = get_currency_code();

        // حساب الإحصائيات
        $totalEmployees = Employee::count();
        $activeEmployees = Employee::where('status', 'active')->count();
        $inactiveEmployees = Employee::where('status', 'inactive')->count();
        $pendingEmployees = Employee::where('status', 'pending')->count();

        // حساب مجموع الأوفر تايم لآخر 12 شهراً لكل موظف في الصفحة الحالية
        $sinceDate = Carbon::now()->subYear()->toDateString();
        foreach ($employees as $emp) {
            try {
                $hours = $emp->attendances()->where('date', '>=', $sinceDate)->get()->sum(fn($a) => $a->getOvertimeHours());
                $emp->overtime_hours_last_12m = round($hours, 2);
                // respect employee's overtime_paid flag
                if (($emp->overtime_paid ?? true) && ($emp->overtime_hourly_rate ?? 0) > 0) {
                    $emp->overtime_pay_last_12m = round($hours * (float) ($emp->overtime_hourly_rate ?? 0), 2);
                } else {
                    $emp->overtime_pay_last_12m = 0.00;
                }
            } catch (\Throwable $_) {
                $emp->overtime_hours_last_12m = 0;
                $emp->overtime_pay_last_12m = 0;
            }
        }

        return view('employees.index', compact('employees', 'departments', 'positions', 'currency_name', 'totalEmployees', 'activeEmployees', 'inactiveEmployees', 'pendingEmployees'));
    }

    /**
     * عرض نموذج إنشاء موظف جديد.
     *
     * @return \Illuminate\View\View
     */
    public function create(): View
    {
        $this->authorize('create', Employee::class);

        $departments = Department::orderBy('name')->get();
        $positions = Position::orderBy('name')->get();
        $currency_name = get_currency_code();

        $defaultComponents = [];
        $firstEmployee = Employee::orderBy('created_at')->first();
        if ($firstEmployee) {
            for ($i = 1; $i <= 7; $i++) {
                $defaultComponents["name_$i"] = old("component_names.$i", $firstEmployee->{"component_name_$i"});
                $defaultComponents["value_$i"] = old("component_values.$i", $firstEmployee->{"component_$i"});
            }
            $defaultPaidOffAmount = old('default_paid_off_amount', $firstEmployee->default_paid_off_amount ?? 0.00);
        } else {
            for ($i = 1; $i <= 7; $i++) {
                $defaultComponents["name_$i"] = old("component_names.$i");
                $defaultComponents["value_$i"] = old("component_values.$i");
            }
            $defaultPaidOffAmount = old('default_paid_off_amount', 0.00);
        }

        return view('employees.create', compact('departments', 'positions', 'currency_name', 'defaultComponents', 'defaultPaidOffAmount'));
    }

    /**
     * تخزين موظف جديد في قاعدة البيانات.
     *
     * @param EmployeeRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(EmployeeRequest $request): RedirectResponse
    {
        $this->authorize('create', Employee::class);

        $data = $request->validated();

        if ($request->birth_date) {
            $data['age'] = Carbon::parse($request->birth_date)->age;
        }

        // Ensure annual_entitlement has a default value
        if (!isset($data['annual_entitlement']) || is_null($data['annual_entitlement'])) {
            $data['annual_entitlement'] = 0;
        }

        $employee = Employee::create($data);

        // save default paid off amount if provided
        if ($request->filled('default_paid_off_amount')) {
            $employee->default_paid_off_amount = (float) $request->input('default_paid_off_amount');
            $employee->save();
        }

        // set accrue_leaves default if present in request
        if ($request->has('accrue_leaves')) {
            $employee->accrue_leaves = (bool) $request->input('accrue_leaves');
            $employee->save();
        }

        // set deduct_if_underworked if present
        if ($request->has('deduct_if_underworked')) {
            $employee->deduct_if_underworked = (bool) $request->input('deduct_if_underworked');
            $employee->save();
        }

        // set overtime_paid if present
        if ($request->has('overtime_paid')) {
            $employee->overtime_paid = (bool) $request->input('overtime_paid');
            $employee->save();
        }

        // Recalculate overtime totals immediately so UI and reports reflect the new flag
        try {
            $employee->recalcOvertimeTotals();
            // Ensure pending payrolls reflect the new overtime flag immediately
            try {
                $employee->applyOvertimePaidFlagToPendingPayrolls();
            } catch (\Throwable $_) {
                // ignore
            }
        } catch (\Throwable $_) {
            // ignore calc errors
        }

        // save annual_entitlement and weekly_off_days if provided
        if ($request->filled('annual_entitlement')) {
            $employee->annual_entitlement = (int) $request->input('annual_entitlement');
        }
        if ($request->has('weekly_off_days')) {
            $employee->weekly_off_days = $request->input('weekly_off_days');
        }
        $employee->save();

        $componentNames = $request->input('component_names', []);
        $componentValues = $request->input('component_values', []);

        for ($i = 1; $i <= 7; $i++) {
            $employee->{"component_name_$i"} = $componentNames[$i - 1] ?? null;
            $employee->{"component_$i"} = (float) ($componentValues[$i - 1] ?? 0);
        }
        $employee->save();

        if ($request->hasFile('photo')) {
            $employee->update(['photo' => $request->file('photo')->store('employees/photos', 'public')]);
        }

        if ($request->hasFile('cv_files')) {
            $cvPaths = [];
            foreach ($request->file('cv_files') as $file) {
                $cvPaths[] = $file->store('employees/cvs', 'public');
            }
            $employee->update(['cv_files' => $cvPaths]);
        }

        // Link to user account if provided
        if ($request->filled('user_id')) {
            $user = \App\Models\User::find($request->input('user_id'));
            if ($user) {
                // clear any previous employee linked to this user
                if ($user->employee_id && $user->employee_id !== $employee->id) {
                    // detach previous employee->user relationship if exists
                    $previous = \App\Models\Employee::find($user->employee_id);
                    if ($previous) {
                        $previous->user_id = null;
                        $previous->save();
                    }
                }

                $user->employee_id = $employee->id;
                $user->save();

                $employee->user_id = $user->id;
                $employee->save();
            }
        }

        return redirect()->route('employees.index')
                         ->with('success', "تم إنشاء الموظف بنجاح برقم: <strong>{$employee->employee_id}</strong>");
    }

    /**
     * عرض تفاصيل موظف محدد.
     *
     * @param Employee $employee
     * @return \Illuminate\View\View
     */
    public function show(Employee $employee): View
    {
        $this->authorize('view', $employee);

        $employee->load([
            'department:id,name',
            'position:id,name',
            'user',
            'salaryComponents' => fn($q) => $q->withPivot('value')
        ]);

        $leaves = $employee->leaves()->orderBy('created_at', 'desc')->paginate(10, ['*'], 'leave_page');
        $loans = $employee->loans()->with('installments')->orderBy('created_at', 'desc')->paginate(10, ['*'], 'loan_page');
        $deductions = $employee->deductions()->orderBy('created_at', 'desc')->paginate(10, ['*'], 'deduction_page');
        $attendances = $employee->attendances()->orderBy('date', 'desc')->paginate(10, ['*'], 'attendance_page');
        $payrolls = $employee->payrolls()->orderBy('year', 'desc')->orderBy('month', 'desc')->get();

        $leaveMonthlyDetails = $employee->getMonthlyLeaveAccrualDetails();
        $accrualYear = request('accrual_year');
        $accrualMonth = request('accrual_month');
        if ($accrualYear || $accrualMonth) {
            $leaveMonthlyDetails = array_filter($leaveMonthlyDetails, function ($d) use ($accrualYear, $accrualMonth) {
                return (!$accrualYear || $d['year'] == $accrualYear) && (!$accrualMonth || $d['month'] == $accrualMonth);
            });
        }

        $payrollSummary = $this->calculatePayrollSummary($employee);

        $currency_name = get_currency_code();
        $leaveTypesMap = [
            'annual' => 'سنوية', 'sick' => 'مرضية', 'emergency' => 'طارئة',
            'maternity' => 'وضع', 'paternity' => 'أبوة', 'unpaid' => 'بدون راتب'
        ];


// ✅ إضافة هذا السطر قبل return مباشرةً
$paidPayrolls = Payroll::where('employee_id', $employee->id)
    ->where('status', 'paid')
    ->with('payrollSalaryComponents')
    ->orderBy('year', 'desc')
    ->orderBy('month', 'desc')
    ->get();




        return view('employees.show', compact(
            'employee', 'leaves', 'loans', 'deductions', 'attendances',
            'payrolls', 'leaveMonthlyDetails', 'payrollSummary',
            'currency_name', 'leaveTypesMap', 'paidPayrolls'
        ));
    }

    /**
     * عرض تقويم شهري لحالة الحضور/العطلات/الإجازات للموظف.
     */
    public function calendar(Request $request, Employee $employee)
    {
        $this->authorize('view', $employee);

        $month = (int) $request->input('month', now()->month);
        $year = (int) $request->input('year', now()->year);

        // Create proper date range for the month
        $start = Carbon::create($year, $month, 1);
        $end = $start->copy()->endOfMonth();

        // Extend the range to include full weeks (from start of week to end of week)
        $start = $start->copy()->startOfWeek(Carbon::MONDAY); // Start from Monday of the first week
        $end = $end->copy()->endOfWeek(Carbon::SUNDAY); // End on Sunday of the last week

        // holidays in extended range
        $holidays = Holiday::whereBetween('date', [$start->format('Y-m-d'), $end->format('Y-m-d')])->get()->keyBy('date');

        // leaves overlapping
        $leaves = Leave::where('employee_id', $employee->id)
            ->whereNotIn('status', ['rejected', 'cancelled'])
            ->where(function ($q) use ($start, $end) {
                $q->whereBetween('start_date', [$start->format('Y-m-d'), $end->format('Y-m-d')])
                  ->orWhereBetween('end_date', [$start->format('Y-m-d'), $end->format('Y-m-d')])
                  ->orWhere(function ($q2) use ($start, $end) {
                      $q2->where('start_date', '<=', $start->format('Y-m-d'))
                         ->where('end_date', '>=', $end->format('Y-m-d'));
                  });
            })->get();

        $attendances = Attendance::where('employee_id', $employee->id)
            ->whereBetween('date', [$start->format('Y-m-d'), $end->format('Y-m-d')])->get();
        
        // Create a map of attendances keyed by date string for faster lookup
        $attendancesByDate = [];
        foreach ($attendances as $att) {
            $dateKey = Carbon::parse($att->date)->format('Y-m-d');
            $attendancesByDate[$dateKey] = $att;
        }

        // weekly offs from employee (array of ISO day numbers or names)
        $weeklyOffDays = $employee->weekly_off_days ?? [];

        // Map ISO day numbers to day names
        $dayNameMap = [
            1 => 'monday',
            2 => 'tuesday',
            3 => 'wednesday',
            4 => 'thursday',
            5 => 'friday',
            6 => 'saturday',
            7 => 'sunday',
        ];

        $days = [];
        for ($d = $start->copy(); $d->lte($end); $d->addDay()) {
            $date = $d->format('Y-m-d');
            $isHoliday = $holidays->has($date);
            $dayName = $dayNameMap[$d->dayOfWeekIso];
            $isWeeklyOff = in_array($dayName, (array)$weeklyOffDays);
            // Check if day falls within a leave period
            $dayLeave = $leaves->first(function ($l) use ($date) {
                return Carbon::parse($l->start_date)->format('Y-m-d') <= $date && Carbon::parse($l->end_date)->format('Y-m-d') >= $date;
            });
            
            // If day falls within a leave but is also a holiday/weekly off, mark it as such (not counted against leave)
            if ($dayLeave && ($isHoliday || $isWeeklyOff)) {
                // This day is covered by leave but shouldn't be counted (it's a holiday or weekly off within the leave period)
                $dayLeave = null;
            }
            
            $attendance = $attendancesByDate[$date] ?? null;

            $status = 'present';
            $reason = '';
            $hasAttendanceRecord = false;

            // If there's an attendance record and the day is a holiday or weekly off,
            // mark it as 'present_on_holiday' so views can display a special badge.
            if ($attendance) {
                $hasAttendanceRecord = true;
                if ($isHoliday || $isWeeklyOff) {
                    if ($isHoliday) {
                        $status = 'present_on_official_holiday';
                        $reason = $holidays->get($date)?->title ?? 'عطلة رسمية';
                    } else {
                        $status = 'present_on_weekly_off';
                        $reason = __('calendar.statuses.present_on_weekly_off');
                    }
                } elseif ($attendance->leave_id && $attendance->leave) {
                    $status = 'leave';
                    $reason = $attendance->leave->reason ?? 'إجازة';
                } else {
                    $status = 'present';
                    $reason = '';
                }
            } elseif ($isHoliday) {
                $status = 'holiday';
                $reason = $holidays->get($date)?->title ?? 'عطلة رسمية';
            } elseif ($dayLeave) {
                $status = 'leave';
                $reason = $dayLeave->reason ?? 'إجازة معتمدة';
            } elseif ($isWeeklyOff) {
                $status = 'weekly_off';
                $reason = __('calendar.statuses.weekly_off');
            } else {
                // No attendance record and not a holiday/weekly off/leave
                $status = 'not_present';
                $reason = '';
            }

            $days[] = [
                'date' => $date,
                'label' => $d->format('j'),
                'status' => $status,
                'reason' => $reason,
                'leave' => $dayLeave,
                'attendance' => $attendance,
                'is_other_month' => $d->month != $month, // Flag to mark days from other months
                'has_attendance_record' => $hasAttendanceRecord,
            ];
        }

        // Debug: Log calendar data
        \Log::debug('Calendar days loaded', [
            'employee_id' => $employee->id,
            'month' => $month,
            'year' => $year,
            'total_days' => count($days),
            'date_range' => $start->format('Y-m-d') . ' to ' . $end->format('Y-m-d'),
            'holidays_count' => $holidays->count(),
            'leaves_count' => $leaves->count(),
            'attendances_count' => count($attendances),
            'attendances_by_date_keys' => array_keys($attendancesByDate),
            'weekly_off_days' => $weeklyOffDays,
        ]);

        return view('employees.calendar', compact('employee', 'month', 'year', 'days'));
    }

    /**
     * تنفيذ إجراء على يوم محدد (AJAX): منح إجازة، وضع غياب، أو خصم.
     */
    public function calendarAction(Request $request, Employee $employee)
    {
        $this->authorize('update', $employee);

        $request->validate([
            'date' => 'required|date',
            'action' => 'required|string|in:grant_leave,mark_absent,apply_deduction,send_reminder',
            'deduction_amount' => 'nullable|numeric|min:0'
        ]);

        $date = Carbon::parse($request->input('date'))->format('Y-m-d');
        $action = $request->input('action');

        if ($action === 'grant_leave') {
            // Use same logic as AttendanceController::markUnauthorizedLeave
            $from = $request->input('from', 'monthly'); // monthly|annual
            $kind = $from;

            // Prevent duplicate attendance for the date unless forced by admin
            $exists = Attendance::where('employee_id', $employee->id)->whereDate('date', $date)->exists();
            $force = (bool) $request->input('force', false);
            if ($exists && !$force) {
                return response()->json(['status' => 'error', 'message' => 'سجل الحضور موجود بالفعل لهذا اليوم. استخدم خيار التعزيز (force) لتطبيق الإجازة والخصم مع الإبقاء على سجل الحضور.'], 422);
            }

            // Use LeaveBalanceService to check available balance for BOTH kinds
            $svc = new \App\Services\LeaveBalanceService();
            
            $annualAvailable = 0;
            $monthlyAvailable = 0;
            
            try {
                $annualAvailable = $svc->availableBalance($employee, 'annual', Carbon::parse($date));
                \Log::info('Calendar annual balance calculated', [
                    'employee_id' => $employee->id,
                    'date' => $date,
                    'annual_available' => $annualAvailable,
                ]);
            } catch (\Throwable $e) {
                \Log::error('LeaveBalanceService annual balance failed: ' . $e->getMessage(), [
                    'employee_id' => $employee->id,
                    'exception' => $e,
                ]);
                $annualAvailable = 0;
            }
            
            try {
                $monthlyAvailable = $svc->availableBalance($employee, 'monthly', Carbon::parse($date));
                \Log::info('Calendar monthly balance calculated', [
                    'employee_id' => $employee->id,
                    'date' => $date,
                    'monthly_available' => $monthlyAvailable,
                ]);
            } catch (\Throwable $e) {
                \Log::error('LeaveBalanceService monthly balance failed: ' . $e->getMessage(), [
                    'employee_id' => $employee->id,
                    'exception' => $e,
                ]);
                $monthlyAvailable = 0;
            }

            \Log::info('Calendar grant_leave balance check BEFORE decision', [
                'employee_id' => $employee->id,
                'date' => $date,
                'annual_available' => $annualAvailable,
                'monthly_available' => $monthlyAvailable,
                'selected_from' => $from,
            ]);
            
            // Determine which balance to use
            $available = ($kind === 'annual') ? $annualAvailable : $monthlyAvailable;
            $kindLabel = ($kind === 'annual') ? 'السنوي' : 'الشهري';
            
            \Log::info('Calendar initial balance check', [
                'employee_id' => $employee->id,
                'kind' => $kind,
                'available' => $available,
                'is_sufficient' => $available >= 1,
            ]);
            
            // If selected balance is not enough, try the other one
            if ($available < 1) {
                $otherKind = ($kind === 'annual') ? 'monthly' : 'annual';
                $otherAvailable = ($otherKind === 'annual') ? $annualAvailable : $monthlyAvailable;
                
                \Log::info('Calendar trying other balance', [
                    'employee_id' => $employee->id,
                    'other_kind' => $otherKind,
                    'other_available' => $otherAvailable,
                    'is_sufficient' => $otherAvailable >= 1,
                ]);
                
                if ($otherAvailable >= 1) {
                    $kind = $otherKind;
                    $available = $otherAvailable;
                    $kindLabel = ($kind === 'annual') ? 'السنوي' : 'الشهري';
                    
                    \Log::info('Calendar SWITCHED to other balance', [
                        'employee_id' => $employee->id,
                        'new_kind' => $kind,
                        'new_available' => $available,
                    ]);
                }
            }
            
            \Log::info('Calendar FINAL decision', [
                'employee_id' => $employee->id,
                'final_kind' => $kind,
                'final_available' => $available,
                'will_deduct_from_balance' => $available >= 1,
            ]);
            
            // Create a Leave record if balance exists and deduct a single day
            if ($available >= 1) {
                $leaveType = \App\Models\LeaveType::where('code', $kind)->first();
                if (!$leaveType) {
                    return response()->json(['status' => 'error', 'message' => 'نوع الإجازة غير معرف في النظام.'], 500);
                }

                $leave = Leave::create([
                    'employee_id' => $employee->id,
                    'leave_type' => $leaveType->code,
                    'leave_type_id' => $leaveType->id,
                    'start_date' => $date,
                    'end_date' => $date,
                    'days_count' => 1,
                    'deductible_days' => 1,  // ✅ استخدام deductible_days (الاسم الذي تتوقعه applyApproval)
                    'reason' => 'غياب بدون إذن — خصم من الرصيد ' . $kindLabel,
                    'status' => 'approved',
                    'approved_at' => now(),
                    'approved_by' => auth()->id(),
                ]);

                try {
                    $svc->applyApproval($leave, auth()->id());
                } catch (\Throwable $e) {
                    \Log::error('applyApproval failed (calendar): ' . $e->getMessage());
                }

                // Create attendance record linked to leave
                Attendance::create([
                    'employee_id' => $employee->id,
                    'date' => $date,
                    'status' => 'absent',
                    'leave_id' => $leave->id ?? null,
                    'notes' => 'غياب بدون إذن — خصم من الرصيد ' . $kindLabel,
                ]);

                // Activity log
                try {
                    if (class_exists('\App\\Models\\Activity')) {
                        \App\Models\Activity::create([
                            'description' => 'unauthorized_leave_deducted_from_balance',
                            'properties' => [
                                'employee_id' => $employee->id,
                                'date' => $date,
                                'kind' => $kind,
                                'leave_id' => $leave->id ?? null,
                                'deducted_days' => 1,
                                'created_by' => auth()->id(),
                            ],
                        ]);
                    }
                } catch (\Throwable $e) {
                    \Log::error('Activity create failed (calendar): ' . $e->getMessage());
                }

                return response()->json([
                    'status' => 'ok',
                    'action' => 'grant_leave',
                    'new_status' => 'leave',
                    'new_badge' => 'إجازة + خصم رصيد',
                    'message' => 'تم خصم يوم من رصيد ' . $kindLabel . ' للموظف.',
                    'leave' => $leave
                ]);
            }

            // If no available balance, create monetary deduction and unpaid leave
            $snapshotDate = Carbon::parse($date);
            $totalSalary = $employee->getTotalSalaryWithComponentsForDate($snapshotDate);
            $dailyRate = $totalSalary / 30.0;
            $amount = round($dailyRate, 2);

            $reasonNoBalance = 'غياب بدون إذن — تم خصم مقابل مادي';

            $ded = Deduction::create([
                'employee_id' => $employee->id,
                'type' => 'monetary',
                'description' => 'خصم مقابل غياب بدون إذن - ' . $date,
                'amount' => $amount,
                'deduction_date' => $date,
                'status' => 'applied',
                'is_monthly' => false,
                'frequency' => 'once',
                'meta' => ['reason' => 'absent_without_permission', 'created_by' => auth()->id()],
            ]);

            // create unpaid leave to link to attendance
            $ltUnpaid = \App\Models\LeaveType::where('code', 'unpaid')->first();
            $unpaidLeave = null;
            if ($ltUnpaid) {
                $unpaidLeave = Leave::create([
                    'employee_id' => $employee->id,
                    'leave_type' => 'unpaid',
                    'leave_type_id' => $ltUnpaid->id,
                    'start_date' => $date,
                    'end_date' => $date,
                    'days_count' => 1,
                    'reason' => $reasonNoBalance,
                    'status' => 'approved',
                    'approved_at' => now(),
                    'approved_by' => auth()->id(),
                    'is_unpaid_deduction_applied' => true,
                ]);

                try {
                    if (class_exists('\App\\Models\\Activity')) {
                        \App\Models\Activity::create([
                            'description' => 'unauthorized_leave_unpaid_record_created',
                            'properties' => [
                                'employee_id' => $employee->id,
                                'date' => $date,
                                'leave_id' => $unpaidLeave->id ?? null,
                                'deduction_id' => $ded->id ?? null,
                                'created_by' => auth()->id(),
                            ],
                        ]);
                    }
                } catch (\Throwable $e) {
                    \Log::error('Activity create failed (unpaid leave): ' . $e->getMessage());
                }
            }

            Attendance::create([
                'employee_id' => $employee->id,
                'date' => $date,
                'status' => 'absent',
                'leave_id' => $unpaidLeave->id ?? null,
                'notes' => $reasonNoBalance,
            ]);

            try {
                if (class_exists('\App\\Models\\Activity')) {
                    \App\Models\Activity::create([
                        'description' => 'unauthorized_leave_deduction_created',
                        'properties' => [
                            'employee_id' => $employee->id,
                            'date' => $date,
                            'amount' => $amount,
                            'deduction_id' => $ded->id ?? null,
                            'created_by' => auth()->id(),
                        ],
                    ]);
                }
            } catch (\Throwable $e) {
                \Log::error('Activity create failed (deduction): ' . $e->getMessage());
            }

            return response()->json([
                'status' => 'ok',
                'action' => 'grant_leave',
                'new_status' => 'deducted',
                'new_badge' => 'إجازة + خصم راتب',
                'message' => 'لا يوجد رصيد؛ تم إنشاء استقطاع بمبلغ ' . number_format($amount, 2)
            ]);
        }

        if ($action === 'mark_absent') {
            Attendance::create([
                'employee_id' => $employee->id,
                'date' => $date,
                'status' => 'absent',
            ]);
            return response()->json(['status' => 'ok', 'action' => 'mark_absent']);
        }

        if ($action === 'send_reminder') {
            // Send reminder notification (email + database) to employee's user
            if ($employee->user) {
                $employee->user->notify(
                    new \App\Notifications\AttendanceReminder($employee, $date, auth()->id())
                );
            }
            
            // Log the reminder for audit trail
            \Log::info('Calendar send_reminder', [
                'employee_id' => $employee->id,
                'date' => $date,
                'by' => auth()->id(),
                'notification_sent' => $employee->user ? true : false
            ]);
            
            return response()->json(['status' => 'ok', 'action' => 'send_reminder']);
        }        if ($action === 'apply_deduction') {
            Deduction::create([
                'employee_id' => $employee->id,
                'date' => $date,
                'amount' => $request->input('deduction_amount', 0),
                'reason' => $request->input('reason', 'خصم لغياب بدون إذن'),
            ]);
            return response()->json(['status' => 'ok', 'action' => 'apply_deduction']);
        }

        return response()->json(['status' => 'error', 'message' => 'إجراء غير معروف'], 422);
    }

    /**
     * عرض نموذج تعديل موظف.
     *
     * @param Employee $employee
     * @return \Illuminate\View\View
     */
    public function edit(Employee $employee): View
    {
        $this->authorize('update', $employee);

        $employee->load(['salaryComponents' => fn($q) => $q->withPivot('value')]);
        $departments = Department::orderBy('name')->get();
        $positions = Position::orderBy('name')->get();
        $currency_name = get_currency_code();

        return view('employees.edit', compact('employee', 'departments', 'positions', 'currency_name'));
    }

    /**
     * تحديث بيانات موظف محدد.
     *
     * @param \Illuminate\Http\Request $request
     * @param Employee $employee
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, Employee $employee): RedirectResponse
    {
        $this->authorize('update', $employee);

        $request->validate([
            'name' => 'required|string|max:255',
            'employee_id' => 'required|string|unique:employees,employee_id,' . $employee->id,
            'email' => 'nullable|email|unique:employees,email,' . $employee->id,
            'user_id' => 'nullable|exists:users,id',
            'phone' => 'nullable|string|max:20',
            'position_id' => 'nullable|exists:positions,id',
            'department_id' => 'nullable|exists:departments,id',
            'salary' => 'nullable|numeric|min:0',
            'incentives' => 'nullable|numeric|min:0',
            'hire_date' => 'nullable|date',
            'birth_date' => 'nullable|date|before:today',
            'status' => 'required|in:active,inactive,pending',
            'age' => 'nullable|integer|min:16|max:100',
            'qualification' => 'nullable|string|max:255',
            'national_id' => 'nullable|string|max:50|unique:employees,national_id,' . $employee->id,
            'address' => 'nullable|string|max:500',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'cv_files' => 'nullable|array',
            'cv_files.*' => 'nullable|file|mimes:pdf,doc,docx|max:10240',
            'scheduled_check_in' => 'nullable|date_format:H:i',
            'scheduled_check_out' => 'nullable|date_format:H:i|after:scheduled_check_in',
            'monthly_leave_days_allowed' => 'nullable|integer|min:0|max:30',
            'annual_entitlement' => 'nullable|integer|min:0',
            'overtime_hourly_rate' => 'nullable|numeric|min:0',
            'deduction_hourly_rate' => 'nullable|numeric|min:0',
            'weekly_off_days' => 'nullable|array',
            'weekly_off_days.*' => 'nullable',
            'component_names' => 'array|max:7',
            'component_names.*' => 'string|max:255|nullable',
            'component_values' => 'array|max:7',
            'component_values.*' => 'required_with:component_names.*|numeric|min:0',
        ]);

        $data = $request->only([
            'name', 'employee_id', 'email', 'phone', 'position_id', 'department_id',
            'salary', 'incentives', 'hire_date', 'status', 'qualification',
            'birth_date', 'national_id', 'address', 'scheduled_check_in',
            'scheduled_check_out', 'monthly_leave_days_allowed', 'annual_entitlement', 'weekly_off_days', 'overtime_hourly_rate', 'deduction_hourly_rate', 'accrue_leaves', 'deduct_if_underworked', 'overtime_paid', 'default_paid_off_amount'
        ]);

        if ($request->birth_date) {
            $data['age'] = Carbon::parse($request->birth_date)->age;
        } else {
            $data['age'] = null;
        }

        $employee->update($data);

        // sync linked user (if provided)
        if ($request->has('user_id')) {
            $newUserId = $request->input('user_id') ?: null;
            $oldUser = $employee->user;
            if ($oldUser && $oldUser->id != $newUserId) {
                $oldUser->employee_id = null;
                $oldUser->save();
            }

            if ($newUserId) {
                $newUser = \App\Models\User::find($newUserId);
                if ($newUser) {
                    // if this user was linked to another employee, detach that employee
                    if ($newUser->employee_id && $newUser->employee_id != $employee->id) {
                        $prevEmp = \App\Models\Employee::find($newUser->employee_id);
                        if ($prevEmp) {
                            $prevEmp->user_id = null;
                            $prevEmp->save();
                        }
                    }

                    $newUser->employee_id = $employee->id;
                    $newUser->save();
                    $employee->user_id = $newUser->id;
                    $employee->save();
                }
            } else {
                // clearing link
                $employee->user_id = null;
                $employee->save();
            }
        }

        // ensure weekly_off_days saved correctly (as array)
        if ($request->has('weekly_off_days')) {
            $employee->weekly_off_days = $request->input('weekly_off_days');
            $employee->save();
        }

        // accrue_leaves flag
        if ($request->has('accrue_leaves')) {
            $employee->accrue_leaves = (bool) $request->input('accrue_leaves');
            $employee->save();
        }

        // deduct_if_underworked flag
        if ($request->has('deduct_if_underworked')) {
            $employee->deduct_if_underworked = (bool) $request->input('deduct_if_underworked');
            $employee->save();
        }

        // overtime_paid flag
        if ($request->has('overtime_paid')) {
            $employee->overtime_paid = (bool) $request->input('overtime_paid');
            $employee->save();
        }

        // Recalculate overtime totals immediately so UI and reports reflect the new flag
        try {
            $employee->recalcOvertimeTotals();
            // Ensure pending payrolls reflect the new overtime flag immediately
            try {
                $employee->applyOvertimePaidFlagToPendingPayrolls();
            } catch (\Throwable $_) {
                // ignore
            }
        } catch (\Throwable $_) {
            // ignore calc errors
        }

        $componentNames = $request->input('component_names', []);
        $componentValues = $request->input('component_values', []);

        for ($i = 1; $i <= 7; $i++) {
            $employee->{"component_name_$i"} = $componentNames[$i - 1] ?? null;
            $employee->{"component_$i"} = (float) ($componentValues[$i - 1] ?? 0);
        }
        $employee->save();

        if ($request->hasFile('photo')) {
            if ($employee->photo) Storage::disk('public')->delete($employee->photo);
            $employee->update(['photo' => $request->file('photo')->store('employees/photos', 'public')]);
        }

        if ($request->hasFile('cv_files')) {
            $newPaths = [];
            foreach ($request->file('cv_files') as $file) {
                $newPaths[] = $file->store('employees/cvs', 'public');
            }
            $employee->update([
                'cv_files' => array_merge($employee->cv_files ?? [], $newPaths)
            ]);
        }

        return redirect()->route('employees.show', $employee->id)
                         ->with('success', 'تم تحديث بيانات الموظف بنجاح.');
    }

    /**
     * حذف موظف محدد.
     *
     * @param Employee $employee
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Employee $employee): RedirectResponse
    {
        $this->authorize('delete', $employee);

        if ($employee->photo) {
            Storage::disk('public')->delete($employee->photo);
        }

        if ($employee->cv_files) {
            foreach ($employee->cv_files as $file) {
                if (!Str::startsWith($file, ['http://', 'https://'])) {
                    Storage::disk('public')->delete($file);
                }
            }
        }

        $employee->delete();

        return redirect()->route('employees.index')
                         ->with('success', 'تم حذف الموظف بنجاح.');
    }

    /**
     * حذف ملف سيرة ذاتية محدد.
     *
     * @param \Illuminate\Http\Request $request
     * @param Employee $employee
     * @return \Illuminate\Http\RedirectResponse
     */
    public function removeCvFile(Request $request, Employee $employee): RedirectResponse
    {
        $this->authorize('update', $employee);

        $fileName = $request->input('file');
        $cvFiles = $employee->cv_files ?? [];

        if (in_array($fileName, $cvFiles)) {
            if (!Str::startsWith($fileName, ['http://', 'https://'])) {
                Storage::disk('public')->delete($fileName);
            }
            $employee->update(['cv_files' => array_values(array_filter($cvFiles, fn($f) => $f !== $fileName))]);
            return redirect()->back()->with('success', 'تم حذف ملف السيرة الذاتية بنجاح.');
        }

        return redirect()->back()->with('error', 'الملف غير موجود.');
    }

    // ====================== PRINT FUNCTIONS ======================

    public function printSingle(Employee $employee)
    {
        $this->authorize('view', $employee);

        $employee->load('salaryComponents');
        $leaveMonthlyDetails = $employee->getMonthlyLeaveAccrualDetails();
        $leaves = $employee->leaves()->orderBy('created_at', 'desc')->get();
        $loans = $employee->loans()->with('installments')->orderBy('created_at', 'desc')->get();
        $deductions = $employee->deductions()->orderBy('created_at', 'desc')->get();
        $payrolls = $employee->payrolls()->orderBy('year', 'desc')->orderBy('month', 'desc')->get();
        $attendances = $employee->attendances()->orderBy('date', 'desc')->get();
        $payrollSummary = $this->calculatePayrollSummary($employee);

        $leaveTypesMap = [
            'annual' => 'سنوية', 'sick' => 'مرضية', 'emergency' => 'طارئة',
            'maternity' => 'وضع', 'paternity' => 'أبوة', 'unpaid' => 'بدون راتب'
        ];
        $currency_name = get_currency_code();

        return view('employees.print-single', compact(
            'employee', 'leaveMonthlyDetails', 'leaves', 'loans',
            'deductions', 'payrolls', 'attendances', 'payrollSummary',
            'leaveTypesMap', 'currency_name'
        ));
    }

    public function printPayrolls(Employee $employee)
    {
        $this->authorize('view', $employee);
        $payrolls = $employee->payrolls()->orderBy('year', 'desc')->orderBy('month', 'desc')->get();
        $currency_name = get_currency_code();
        return view('employees.print-payrolls', compact('employee', 'payrolls', 'currency_name'));
    }

    public function printPayrollSummaryMonth(Employee $employee)
    {
        $this->authorize('view', $employee);
        $payrollSummary = $this->calculatePayrollSummary($employee);
        $currency_name = get_currency_code();
        return view('employees.print-payroll-summary-month', compact('employee', 'payrollSummary', 'currency_name'));
    }

    public function printLeaves(Employee $employee)
    {
        $this->authorize('view', $employee);
        $leaves = $employee->leaves()->orderBy('created_at', 'desc')->get();
        $leaveTypesMap = [
            'annual' => 'سنوية', 'sick' => 'مرضية', 'emergency' => 'طارئة',
            'maternity' => 'وضع', 'paternity' => 'أبوة', 'unpaid' => 'بدون راتب'
        ];
        $currency_name = get_currency_code();
        return view('employees.print-leaves', compact('employee', 'leaves', 'leaveTypesMap', 'currency_name'));
    }

    public function printLoans(Employee $employee)
    {
        $this->authorize('view', $employee);
        $loans = $employee->loans()->with('installments')->orderBy('created_at', 'desc')->get();
        $currency_name = get_currency_code();
        return view('employees.print-loans', compact('employee', 'loans', 'currency_name'));
    }

    public function printDeductions(Employee $employee)
    {
        $this->authorize('view', $employee);
        $deductions = $employee->deductions()->orderBy('deduction_date', 'desc')->get();
        $currency_name = get_currency_code();
        return view('employees.print-deductions', compact('employee', 'deductions', 'currency_name'));
    }

    public function printLeaveAccrualDetails(Employee $employee)
    {
        $this->authorize('view', $employee);
        $leaveMonthlyDetails = $employee->getMonthlyLeaveAccrualDetails();
        $currency_name = get_currency_code();
        return view('employees.print-leave-accrual-details', compact('employee', 'leaveMonthlyDetails', 'currency_name'));
    }

    public function printAttendances(Employee $employee)
    {
        $this->authorize('view', $employee);
        $attendances = $employee->attendances()->orderBy('date', 'desc')->get();
        $currency_name = get_currency_code();
        return view('employees.print-attendances', compact('employee', 'attendances', 'currency_name'));
    }

    public function printLeaveAccrual(Employee $employee)
    {
        $this->authorize('view', $employee);
        $leaveMonthlyDetails = $employee->getMonthlyLeaveAccrualDetails();
        $currency_name = get_currency_code();
        return view('employees.print-leave-accrual', compact('employee', 'leaveMonthlyDetails', 'currency_name'));
    }

    // ====================== HELPER METHODS ======================

    private function calculatePayrollSummary(Employee $employee)
    {
        $summary = [];
        $hireDate = Carbon::parse($employee->hire_date);
        $startYear = $hireDate->year;
        $startMonth = $hireDate->month;
        $now = now();
        $endYear = $now->year;
        $endMonth = $now->month;

        for ($y = $startYear; $y <= $endYear; $y++) {
            $mStart = ($y == $startYear) ? $startMonth : 1;
            $mEnd = ($y == $endYear) ? $endMonth : 12;

            for ($m = $mStart; $m <= $mEnd; $m++) {
                // ✅ التحقق من وجود راتب محفوظ - إذا وُجد، استخدم القيم المحفوظة
                $payrollRecord = \App\Models\Payroll::where('employee_id', $employee->id)
                    ->where('year', $y)
                    ->where('month', $m)
                    ->with('payrollSalaryComponents')
                    ->first();
                
                // ✅ إذا كان هناك راتب محفوظ، استخدم القيم المحفوظة (الرواتب القديمة تحتفظ بقيمها)
                if ($payrollRecord) {
                    $grossSalary = ($payrollRecord->basic_salary ?? 0) + ($payrollRecord->incentives ?? 0);
                    // إضافة المكونات الإضافية من الراتب المحفوظ
                    $componentsTotal = $payrollRecord->payrollSalaryComponents
                        ->filter(fn($c) => ($c->value ?? 0) > 0)
                        ->sum('value');
                    $grossSalary += $componentsTotal;
                    $leaveDeduction = $payrollRecord->leave_deduction ?? 0;
                    $deductionsTotal = $payrollRecord->active_deductions ?? 0;
                    $loanInstallmentsTotal = $payrollRecord->loan_installments ?? 0;
                } else {
                    // ✅ إذا لم يكن هناك راتب محفوظ، استخدم سجل التعويض للشهر المحدد
                    $leaveDeduction = $employee->calculateLeaveDeductionForMonth($y, $m);
                    $deductionsTotal = $employee->getTotalActiveDeductionsForMonth($y, $m);
                    $loanInstallmentsTotal = $employee->getTotalLoanInstallmentsForMonth($y, $m);
                    $snapshotDate = \Carbon\Carbon::create($y, $m, 1)->startOfMonth();
                    $grossSalary = $employee->getTotalSalaryWithComponentsForDate($snapshotDate);
                }
                // حساب الأوفر تايم من سجلات الحضور لهذا الشهر
                $overtimeHours = (float) $employee->attendances()
                    ->whereYear('date', $y)
                    ->whereMonth('date', $m)
                    ->get()
                    ->sum(fn($a) => $a->getOvertimeHours());
                // attendance early/late summary
                $earlyHours = (float) $employee->attendances()
                    ->whereYear('date', $y)
                    ->whereMonth('date', $m)
                    ->get()
                    ->sum(fn($a) => $a->getEarlyHours());
                $lateHours = (float) $employee->attendances()
                    ->whereYear('date', $y)
                    ->whereMonth('date', $m)
                    ->get()
                    ->sum(fn($a) => $a->getLateHours());

                // underworked (worked less than scheduled) and its deduction
                $underworkedHours = (float) $employee->attendances()
                    ->whereYear('date', $y)
                    ->whereMonth('date', $m)
                    ->get()
                    ->sum(fn($a) => $a->getUnderworkedHours());
                $underworkedDeduction = (float) $employee->attendances()
                    ->whereYear('date', $y)
                    ->whereMonth('date', $m)
                    ->get()
                    ->sum(fn($a) => $a->getUnderworkedValue());

                // attendanceAdjustment now represents early/late net + (- underworkedDeduction)
                $attendanceAdjustment = (float) $employee->attendances()
                    ->whereYear('date', $y)
                    ->whereMonth('date', $m)
                    ->get()
                    ->sum(fn($a) => $a->getEarlyLateValue()) - $underworkedDeduction;
                // respect overtime_paid flag: if disabled, overtime pay is 0
                if (($employee->overtime_paid ?? true) && ($employee->overtime_hourly_rate ?? 0) > 0) {
                    $overtimePay = $overtimeHours * (float) ($employee->overtime_hourly_rate ?? 0);
                } else {
                    $overtimePay = 0.0;
                }

                $netSalary = $grossSalary + $overtimePay + $attendanceAdjustment - $deductionsTotal - $loanInstallmentsTotal - $leaveDeduction;

                $summary[] = [
                    'year' => $y,
                    'month' => $m,
                    'month_name' => Carbon::create($y, $m, 1)->locale('ar')->monthName,
                    'gross_salary' => (float) $grossSalary,
                    'overtime_pay' => (float) $overtimePay,
                    'early_hours' => (float) $earlyHours,
                    'late_hours' => (float) $lateHours,
                    'underworked_hours' => (float) $underworkedHours,
                    'underworked_deduction' => (float) $underworkedDeduction,
                    'attendance_adjustment' => (float) $attendanceAdjustment,
                    'deductions_total' => (float) $deductionsTotal,
                    'loan_installments_total' => (float) $loanInstallmentsTotal,
                    'leave_deduction' => (float) $leaveDeduction,
                    'net_salary' => (float) $netSalary,
                    // ✅ إضافة payroll_id للتحقق من وجود راتب محفوظ
                    'payroll_id' => $payrollRecord ? $payrollRecord->id : null,
                ];
            }
        }

        return $summary;
    }
}

