<?php

namespace App\Http\Controllers;

use App\Models\Deduction;
use App\Models\Employee;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class DeductionController extends Controller
{
    public function index()
{
    $this->authorize('viewAny', Deduction::class);

    $authenticatedUser = auth()->user();

    // ✅ التحقق من وجود العلاقة مع Employee
    if ($authenticatedUser->hasRole('employee') && !$authenticatedUser->employee) {
        // ✅ إرجاع كائن Pagination فارغ (ليس Collection) لتجنب خطأ hasPages()
        $emptyPaginator = Deduction::whereRaw('1 = 0')->paginate(15);
        return view('deductions.index', [
            'deductions' => $emptyPaginator,
            'employees' => collect(),
            'stats' => [
                'total' => 0,
                'monthly' => 0,
                'annual' => 0,
                'cancelled' => 0,
            ],
            'error' => 'Your employee profile is not linked to your user account. Please contact HR.'
        ]);
    }

    if ($authenticatedUser->hasRole('employee')) {
        $query = Deduction::with('employee')->where('employee_id', $authenticatedUser->employee->id);
    } else {
        $query = Deduction::with('employee');
    }

    // ✅ البحث العام
    if (request('search')) {
        $search = request('search');
        $query->whereHas('employee', function ($q) use ($search) {
            $q->where('name', 'like', "%{$search}%")
              ->orWhere('employee_id', 'like', "%{$search}%");
        });
    }

        // ✅ الفلاتر
    if (!$authenticatedUser->hasRole('employee')) {
        if (request('employee_id')) {
            $query->where('employee_id', request('employee_id'));
        }

            if (request('deduction_type')) {
            $query->where('type', request('deduction_type'));
        }

            // frequency filter: accept several synonyms
            if (request('frequency')) {
                $f = request('frequency');
                if ($f === 'monthly') {
                    // support legacy boolean flag too
                    $query->where(function($q){ $q->where('frequency', 'monthly')->orWhere('is_monthly', true); });
                } elseif (in_array($f, ['once', 'one_time'])) {
                    $query->where(function($q){ $q->whereNull('frequency')->orWhere('frequency', 'once')->orWhere('frequency', 'one_time'); })->where('is_monthly', false);
                } elseif (in_array($f, ['annual', 'yearly'])) {
                    $query->where('frequency', 'yearly');
                }
            }

        if (request('status')) {
            $query->where('status', request('status'));
        }
    }

    $deductions = $query->latest()->paginate(15);
    $employees = $authenticatedUser->hasRole('employee')
        ? collect([$authenticatedUser->employee])
        : Employee::orderBy('name')->get();

    // ✅ الإحصائيات — تصنيف إلى: one_time, monthly, yearly
    $buildCountQuery = fn($baseQuery) => [
        'one_time' => (clone $baseQuery)->where(function($q){ $q->whereNull('frequency')->orWhere('frequency', 'once')->orWhere('frequency','one_time'); })->where('is_monthly', false)->count(),
        'monthly' => (clone $baseQuery)->where(function($q){ $q->where('frequency','monthly')->orWhere('is_monthly', true); })->count(),
        'yearly' => (clone $baseQuery)->where('frequency','yearly')->count(),
        'cancelled' => (clone $baseQuery)->where('status','cancelled')->count(),
        'total' => (clone $baseQuery)->count(),
    ];

    if ($authenticatedUser->hasRole('employee')) {
        $stats = $buildCountQuery($query);
    } else {
        $stats = $buildCountQuery(Deduction::query());
    }

    $currency_name = get_currency_code();

    return view('deductions.index', compact(
        'deductions',
        'employees',
        'stats',
        'currency_name'
    ));
}

    
    public function create()
    {
        $this->authorize('create', Deduction::class);
        $employees = Employee::orderBy('name')->get();
        return view('deductions.create', compact('employees'));
    }

    public function store(Request $request)
    {
        $this->authorize('create', Deduction::class);

        $validatedData = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'deduction_type' => 'required|in:cash_deduction,in_kind_deduction,taxes,social_insurance,health_insurance,penalties,deductions,loans,subscriptions,other_deductions,tax,insurance,loan,fine,other,cash,in_kind,deduction,penalty,subscription,absence',
            'amount' => 'required|numeric|min:0.01|max:100000',
            'frequency' => 'required|in:once,monthly,yearly,one_time,annual',
            'notes' => 'nullable|string|max:500',
            'deduction_date' => 'required|date',
            'end_date' => 'nullable|date|after_or_equal:deduction_date',
            'status' => 'nullable|in:pending,applied,cancelled',
        ]);

        // normalize frequency values: support 'annual'/'yearly' and 'one_time'/'once'
        $freq = $validatedData['frequency'] ?? 'once';
        if ($freq === 'annual') $freq = 'yearly';
        if ($freq === 'one_time') $freq = 'once';

        $isMonthly = ($freq === 'monthly');

        $modelData = [
            'employee_id' => $validatedData['employee_id'],
            'type' => $validatedData['deduction_type'],
            'description' => $validatedData['notes'] ?? '',
            'amount' => $validatedData['amount'],
            'deduction_date' => $validatedData['deduction_date'],
            'status' => $validatedData['status'] ?? 'pending',
            'frequency' => $freq,
            'is_monthly' => ($freq === 'monthly') ? true : false,
            'end_date' => $validatedData['end_date'] ?? null,
            // 'method' removed: field deprecated and hidden from forms
            'meta' => [
                'created_by' => [
                    'id' => auth()->id(),
                    'name' => auth()->user()?->name,
                ],
                'original_frequency' => $freq,
                'ip' => request()->ip(),
                'user_agent' => substr(request()->userAgent() ?? '', 0, 255),
            ],
        ];

        try {
            // create single record for any frequency: monthly/yearly/once
            Deduction::create($modelData);

            $message = $isMonthly ? '✅ تم إنشاء سلسلة الاستقطاعات الشهرية بنجاح' : '✅ تم إضافة الاستقطاع بنجاح';
            return redirect()->route('deductions.index')->with('success', $message);
        } catch (\Exception $e) {
            \Log::error('Error creating deduction(s): ' . $e->getMessage());
            return redirect()->back()->withInput()->with('error', 'حدث خطأ أثناء إنشاء الاستقطاع(ات): ' . $e->getMessage());
        }
    }

    public function show(Deduction $deduction)
    {
        $this->authorize('view', $deduction);
        $deduction->load('employee'); // تحميل معلومات الموظف

        // --- تعديل: جلب سجلات الاستقطاعات الشهرية ---
        $relatedMonthlyDeductions = collect(); // افتراضيًا قائمة فارغة
        if ($deduction->is_monthly) {
            // نفترض أن سجلات الاستقطاع الشهري المرتبطة تُحدد بـ employee_id و type و period (مثلاً من description أو معرف فريد)
            // لتبسيط المثال، نبحث حسب employee_id و type و is_monthly=true و status='applied'
            $relatedMonthlyDeductions = Deduction::where('employee_id', $deduction->employee_id)
                ->where('type', $deduction->type)
                ->where('is_monthly', true)
                ->where('status', 'applied') // فقط السجلات التي تم تطبيقها (الخصم الفعلي)
                ->orderBy('deduction_date')
                ->get();
        }
        // ---

        $currency_name = get_currency_code(); // يمكنك تغييره حسب النظام

        return view('deductions.show', compact('deduction', 'relatedMonthlyDeductions', 'currency_name'));
    }

    public function edit(Deduction $deduction)
    {
        $this->authorize('update', $deduction);
        $employees = Employee::orderBy('name')->get();
        $deduction->load('employee');
        return view('deductions.edit', compact('deduction', 'employees'));
    }

    public function update(Request $request, Deduction $deduction)
    {
        $this->authorize('update', $deduction);

        // ✅ تم تحديث أسماء الحقول في قواعد التحقق لتتطابق مع 'name' في edit.blade.php
        $validatedData = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'type' => 'required|in:cash_deduction,in_kind_deduction,taxes,social_insurance,health_insurance,penalties,deductions,loans,subscriptions,other_deductions,tax,insurance,loan,fine,other,cash,in_kind,deduction,penalty,subscription,absence', // ✅ تغيير من 'deduction_type' إلى 'type'
            'amount' => 'required|numeric|min:0.01|max:100000',
            'frequency' => 'required|in:once,monthly,yearly,one_time,annual', // ✅ هذا مطابق
            'notes' => 'nullable|string|max:500',
            'deduction_date' => 'required|date',
            'end_date' => 'nullable|date|after_or_equal:deduction_date',
            'status' => 'nullable|in:pending,applied,cancelled',
        ]);

        // ✅ تحديث الكود لاستخدام 'type' بدلاً من 'deduction_type'
        // normalize frequency
        $freq = $validatedData['frequency'] ?? 'once';
        if ($freq === 'annual') $freq = 'yearly';
        if ($freq === 'one_time') $freq = 'once';

        // normalize frequency
        $isMonthly = ($freq === 'monthly');

        $modelData = [
            'employee_id' => $validatedData['employee_id'],
            'type' => $validatedData['type'], // ✅ تغيير من $validatedData['deduction_type'] إلى $validatedData['type']
            'description' => $validatedData['notes'] ?? '',
            'amount' => $validatedData['amount'],
            'deduction_date' => $validatedData['deduction_date'],
            'status' => $validatedData['status'] ?? 'pending',
            'frequency' => $freq,
            'is_monthly' => ($freq === 'monthly'), // ✅ استخدام 'frequency' كما هو
            'end_date' => $validatedData['end_date'] ?? null,
            // 'method' removed: field deprecated and hidden from forms
        ];

        // merge existing meta and add update info
        $existingMeta = $deduction->meta ?? [];
        $existingMeta['last_updated_by'] = [
            'id' => auth()->id(),
            'name' => auth()->user()?->name,
            'at' => now()->toDateTimeString(),
        ];
        $modelData['meta'] = $existingMeta;

        $deduction->update($modelData);

        return redirect()->route('deductions.index')->with('success', '✅ تم تحديث الاستقطاع بنجاح');
    }

    public function destroy(Deduction $deduction)
    {
        $this->authorize('delete', $deduction);
        $deduction->delete();
        return redirect()->route('deductions.index')->with('success', '🗑️ تم حذف الاستقطاع بنجاح');
    }

    // ====================== PRINT EMPLOYEE DEDUCTIONS ======================
    /**
     * عرض صفحة الطباعة لسجل استقطاعات موظف محدد.
     *
     * @param  int  $employeeId
     * @return \Illuminate\Http\Response
     */
    public function printEmployee($employeeId)
    {
        $employee = Employee::findOrFail($employeeId);
        // جلب سجلات الاستقطاعات للفترة المطلوبة (مثلاً آخر 6 أشهر)
        $startDate = Carbon::now()->subMonths(6)->startOfMonth();
        $endDate = Carbon::now()->endOfMonth();

        $deductions = $employee->deductions()
            ->whereBetween('created_at', [$startDate, $endDate])
            ->orderBy('created_at', 'desc')
            ->get();

        return view('deductions.print-employee', compact('employee', 'deductions'));
    }

    // ====================== CUSTOM ACTIONS ======================

    public function apply(Deduction $deduction)
    {
        $this->authorize('update', $deduction);
        $deduction->update(['status' => 'applied']);
        return redirect()->back()->with('success', '✅ تم تطبيق الاستقطاع بنجاح');
    }

    public function cancel(Deduction $deduction)
    {
        $this->authorize('update', $deduction);
        $deduction->update(['status' => 'cancelled']);
        return redirect()->back()->with('success', '❌ تم إلغاء الاستقطاع بنجاح');
    }

    public function toggleStatus(Deduction $deduction)
    {
        $this->authorize('update', $deduction);
        $newStatus = $deduction->status === 'applied' ? 'pending' : 'applied';
        $deduction->update(['status' => $newStatus]);
        return response()->json(['success' => true, 'status' => $newStatus]);
    }

    public function getActiveEmployees()
    {
        $this->authorize('viewAny', Deduction::class);
        $employees = Employee::where('status', 'active')->orderBy('name')->get(['id', 'name', 'employee_id']);
        return response()->json($employees);
    }

    public function getDeductionTypes()
    {
        $this->authorize('viewAny', Deduction::class);
        $types = [
            'cash_deduction' => 'مسحوبات نقدية',
            'in_kind_deduction' => 'مسحوبات عينية',
            'taxes' => 'ضرائب',
            'social_insurance' => 'تأمينات اجتماعية',
            'health_insurance' => 'تأمين صحي',
            'penalties' => 'جزاءات',
            'deductions' => 'خصومات',
            'loans' => 'سلف',
            'subscriptions' => 'اشتراكات',
            'other_deductions' => 'استقطاعات أخرى',
            // القيم الأصلية من create.blade.php
            'tax' => 'ضريبة',
            'insurance' => 'تأمين',
            'loan' => 'قرض',
            'fine' => 'غرامة',
            'other' => 'أخرى',
            'cash' => 'نقدي',
            'in_kind' => 'عيني',
            'deduction' => 'خصم',
            'penalty' => 'جزاء',
            'subscription' => 'اشتراك',
            'absence' => 'غياب',
        ];
        return response()->json($types);
    }
}