<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use App\Models\Employee;
use App\Models\Attendance;
use App\Models\Leave;
use App\Models\Loan;
use App\Models\LoanInstallment;
use App\Models\Deduction;
use App\Models\Activity;
use App\Models\Payroll; // ✅ استيراد نموذج Payroll
use App\Models\Department;
use App\Models\Position;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index(Request $request)
    {
        // ✅ التحقق مما إذا كان المستخدم من دور 'employee'
        $authenticatedUser = auth()->user();
        $isEmployee = $authenticatedUser->hasRole('employee');

        // ✅ إذا كان المستخدم موظفًا عاديًا، اعرض له صفحة مخصصة أو أرسله إلى مكان آخر
        if ($isEmployee && $authenticatedUser->employee) {
            // يمكنك توجيهه إلى صفحة خاصة به، مثل ملفه الشخصي أو قائمة طلباته
            // return redirect()->route('employees.show', $authenticatedUser->employee->id);
            // أو يمكنك عرض صفحة "لوحة تحكم موظف" مبسطة
            // في هذا المثال، سنعرض صفحة خاصة للموظف تحتوي على معلوماته الأساسية فقط.
            // قم بإنشاء عرض جديد إذا لزم الأمر، مثل: resources/views/dashboard/employee.blade.php
            // return view('dashboard.employee', compact('authenticatedUser'));

            // أو ببساطة، قم بترسيته إلى صفحة ملفه الشخصي
            return redirect()->route('employees.show', $authenticatedUser->employee->id);
        }

        // ✅ إذا لم يكن موظفًا عاديًا (أي أنه Admin أو HR أو Super-Admin)، استمر في عرض لوحة التحكم الإدارية
        // ⭐ الإحصائيات الأساسية (متوفرة فقط للإدارة)
        $stats = [
            'total_employees' => Employee::count(),
            'present_today' => Attendance::whereDate('date', today())->where('status', 'present')->count(),
            'pending_leaves' => Leave::where('status', 'pending')->count(),
            'absent_today' => Attendance::whereDate('date', today())->where('status', 'absent')->count(),
            'active_loans' => Loan::where('status', 'active')->count(),
            'overdue_installments' => LoanInstallment::where('status', 'overdue')->count(),
            'pending_deductions' => Deduction::where('status', 'pending')->count(),
        ];

        // 📊 إحصائيات متقدمة للحضور (متوفرة فقط للإدارة)
        $today = Carbon::today();
        $presentToday = Attendance::whereDate('date', $today)->where('status', 'present')->count();
        $lateToday = Attendance::whereDate('date', $today)->where('status', 'late')->count();
        // حساب عدد الغياب: الموظفين النشطين الذين ليس لديهم أي سجل حضور اليوم
        // ولا يملكون إجازة معتمدة تغطي التاريخ
        $recordedEmployeeIds = Attendance::whereDate('date', $today)->pluck('employee_id')->unique()->filter()->values()->toArray();

        $absentToday = \App\Models\Employee::where('status', 'active')
            ->whereNotIn('id', $recordedEmployeeIds ? $recordedEmployeeIds : [0])
            ->get()
            ->filter(function ($emp) use ($today) {
                $hasApprovedLeave = Leave::where('employee_id', $emp->id)
                    ->where('status', 'approved')
                    ->whereDate('start_date', '<=', $today)
                    ->whereDate('end_date', '>=', $today)
                    ->exists();
                return !$hasApprovedLeave;
            })->count();

        // 🕒 متوسط وقت الحضور (متوفر فقط للإدارة)
        $avgCheckIn = Attendance::whereDate('date', $today)
            ->whereNotNull('check_in')
            ->selectRaw('AVG(TIME_TO_SEC(check_in)) as avg_seconds')
            ->first();

        $averageCheckInTime = '00:00';
        if ($avgCheckIn && $avgCheckIn->avg_seconds) {
            $totalSeconds = (int) $avgCheckIn->avg_seconds;
            $hours = floor($totalSeconds / 3600);
            $minutes = floor(($totalSeconds % 3600) / 60);
            $averageCheckInTime = sprintf('%02d:%02d', $hours, $minutes);
        }

        // 📈 نسب الحضور (متوفرة فقط للإدارة)
        $totalEmployees = Employee::count();
        $attendanceRate = $totalEmployees > 0 ? round(($presentToday / $totalEmployees) * 100, 1) : 0;
        $lateRate = $totalEmployees > 0 ? round(($lateToday / $totalEmployees) * 100, 1) : 0;
        $absentRate = $totalEmployees > 0 ? round(($absentToday / $totalEmployees) * 100, 1) : 0;

        // 📋 آخر 4 أنشطة (مبنية على جدول activities) (متوفرة فقط للإدارة)
        $recentActivities = Activity::with(['user'])
            ->latest()
            ->limit(4)
            ->get();

        // 🔔 الإشعارات الأخيرة (الفعلية) مع ترقيم (Pagination)
        $notifications = auth()->user()->notifications()->latest()->paginate(10);

        // 📊 آخر 5 رواتب (متوفرة فقط للإدارة)
        $recentPayrolls = Payroll::with(['employee'])
            ->orderBy('updated_at', 'desc') // ✅ استخدام updated_at لعرض آخر التحديثات (مثل الدفع)
            ->take(5)
            ->get();

        // 🗓️ آخر 5 طلبات إجازات (متوفرة فقط للإدارة)
        $recentLeaves = Leave::with(['employee'])
            ->latest()
            ->take(5)
            ->get();

        // 💰 آخر 5 استقطاعات (متوفرة فقط للإدارة)
        $recentDeductions = Deduction::with(['employee'])
            ->latest()
            ->take(5)
            ->get()
            ->map(function ($deduction) {
                // Compute a locale-aware label for the deduction type.
                $raw = $deduction->type ?? '';

                // 1) If the type is a known key like 'tax' or 'social_insurance', try translation key
                $slug = trim(strtolower((string) $raw));
                $slug = preg_replace('/[^a-z0-9_]+/i', '_', $slug);
                $slug = trim($slug, '_');

                $transKey = 'dashboard.deduction_type_' . ($slug ?: 'other');
                if (\Illuminate\Support\Facades\Lang::has($transKey)) {
                    $deduction->type_label = __($transKey);
                } else {
                    // 2) try matching raw stored value against translation values (best-effort)
                    try {
                        $all = trans('dashboard');
                        $matched = null;
                        foreach ($all as $tkey => $tval) {
                            if (is_string($tval) && mb_strtolower(trim($tval)) == mb_strtolower(trim($raw))) {
                                if (str_starts_with($tkey, 'deduction_type_')) {
                                    $matched = $tval;
                                    break;
                                }
                            }
                        }
                        if ($matched) {
                            $deduction->type_label = $matched;
                        } else {
                            // 3) fallback: prettify the raw value (replace underscores with spaces)
                            $deduction->type_label = $raw ? str_replace('_', ' ', $raw) : __('dashboard.deduction_type_other');
                        }
                    } catch (\Throwable $e) {
                        $deduction->type_label = $raw ? str_replace('_', ' ', $raw) : __('dashboard.deduction_type_other');
                    }
                }

                return $deduction;
            });

        // 🏦 آخر 5 قروض (متوفرة فقط للإدارة)
        $recentLoans = Loan::with(['employee'])
            ->latest()
            ->take(5)
            ->get()
            ->map(function ($loan) { // <-- هذا { مفتوح لدالة map
                // ✅ تحويل نوع القرض إلى اللغة العربية
                $loan->loan_type_arabic = [
                    'personal' => 'شخصي',
                    'car' => 'سيارة',
                    'home' => 'سكن',
                    'education' => 'تعليم',
                    'medical' => 'طبي',
                    'emergency' => 'طوارئ',
                    'business' => 'تجاري',
                    'other' => 'أخرى',
                ][$loan->loan_type] ?? $loan->loan_type;

                return $loan;
            }); // <-- هذا } يغلق دالة map

        // ✅ حسابات ملخص الرواتب لهذا الشهر (متوفرة فقط للإدارة)
        $currentMonth = now()->month;
        $currentYear = now()->year;

        // ✅ إجمالي الرواتب لهذا الشهر (المدفوعة + غير المدفوعة)
        $totalAmountThisMonth = Payroll::where('month', $currentMonth)
            ->where('year', $currentYear)
            ->sum('net_salary');

        // ✅ عدد الرواتب غير المدفوعة (قيد الدفع) لهذا الشهر
        $pendingPayrollsThisMonth = Payroll::where('month', $currentMonth)
            ->where('year', $currentYear)
            ->where('status', 'pending')
            ->count();

        // ✅ حسابات ملخص الإجازات لهذا الشهر (متوفرة فقط للإدارة)
        // ✅ إجمالي الإجازات (جميع الإجازات التي بدأت هذا الشهر)
        $totalLeavesThisMonth = Leave::where('start_date', '>=', Carbon::create($currentYear, $currentMonth, 1)->startOfMonth())
            ->where('start_date', '<=', Carbon::create($currentYear, $currentMonth, 1)->endOfMonth())
            ->count();

        // ✅ الإجازات غير المعتمدة (غير مقبولة ولا مرفوضة) لهذا الشهر
        $unapprovedLeavesThisMonth = Leave::where('start_date', '>=', Carbon::create($currentYear, $currentMonth, 1)->startOfMonth())
            ->where('start_date', '<=', Carbon::create($currentYear, $currentMonth, 1)->endOfMonth())
            ->where('status', '!=', 'approved')
            ->where('status', '!=', 'rejected')
            ->count();

        // ✅ التعامل مع البحث (إذا كان هناك استعلام) (متوفر فقط للإدارة)
        $searchQuery = trim((string) $request->input('search'));

        // إذا وُجد استعلام، ابني نتائج بحث سريعة عبر الموارد الأساسية
        $searchResults = [];
        if ($searchQuery !== '') {
            $q = $searchQuery;

            // Employees: match name, email, or id (partial match)
            $searchResults['employees'] = Employee::where('name', 'like', "%{$q}%")
                ->orWhere('email', 'like', "%{$q}%")
                ->orWhere('id', 'like', "%{$q}%")
                ->take(12)
                ->get();

            // Payrolls: match employee name, payroll id, or net_salary (partial match)
            $searchResults['payrolls'] = Payroll::with('employee')
                ->whereHas('employee', function ($b) use ($q) {
                    $b->where('name', 'like', "%{$q}%");
                })
                ->orWhere('id', 'like', "%{$q}%")
                ->orWhereRaw("CAST(net_salary AS CHAR) LIKE ?", ["%{$q}%"])
                ->take(10)
                ->get();

            // Leaves: match employee name, leave id
            $searchResults['leaves'] = Leave::with('employee')
                ->whereHas('employee', function ($b) use ($q) {
                    $b->where('name', 'like', "%{$q}%");
                })
                ->orWhere('id', 'like', "%{$q}%")
                ->take(8)
                ->get();

            // Deductions: match employee name, deduction id, or amount
            $searchResults['deductions'] = Deduction::with('employee')
                ->whereHas('employee', function ($b) use ($q) {
                    $b->where('name', 'like', "%{$q}%");
                })
                ->orWhere('id', 'like', "%{$q}%")
                ->orWhereRaw("CAST(amount AS CHAR) LIKE ?", ["%{$q}%"])
                ->take(8)
                ->get();

            // Loans: match employee name, loan id, or amount
            $searchResults['loans'] = Loan::with('employee')
                ->whereHas('employee', function ($b) use ($q) {
                    $b->where('name', 'like', "%{$q}%");
                })
                ->orWhere('id', 'like', "%{$q}%")
                ->orWhereRaw("CAST(amount AS CHAR) LIKE ?", ["%{$q}%"])
                ->take(8)
                ->get();

            // Activities: description or properties (stringified) match
            $searchResults['activities'] = Activity::with('user')
                ->where('description', 'like', "%{$q}%")
                ->orWhereRaw("JSON_EXTRACT(properties, '$') LIKE ?", ["%{$q}%"]) // best-effort
                ->take(8)->get();

            // Notifications: search the notifications table by data JSON containing query
            $searchResults['notifications'] = collect(DB::table('notifications')
                ->where('data', 'like', "%{$q}%")
                ->orderBy('created_at', 'desc')
                ->take(8)
                ->get());

            // Attendance: match by employee name or attendance id
            $searchResults['attendances'] = Attendance::with('employee')
                ->whereHas('employee', function ($b) use ($q) {
                    $b->where('name', 'like', "%{$q}%");
                })
                ->orWhere('id', 'like', "%{$q}%")
                ->take(8)
                ->get();

            // Departments & Positions
            $searchResults['departments'] = Department::where('name', 'like', "%{$q}%")->take(8)->get();
            $searchResults['positions'] = Position::where('name', 'like', "%{$q}%")->take(8)->get();
        }

        return view('dashboard.index', compact( // <-- هذا { مفتوح لدالة compact
            'stats',
            'presentToday',
            'lateToday',
            'absentToday',
            'averageCheckInTime',
            'attendanceRate',
            'lateRate',
            'absentRate',
            'recentActivities',
            'notifications',
            'recentPayrolls',
            'recentLeaves',
            'recentDeductions',
            'recentLoans',
            'searchQuery',
            'searchResults',
            // ✅ إضافة المتغيرات الجديدة للدشبورد
            'totalAmountThisMonth',
            'pendingPayrollsThisMonth',
            'totalLeavesThisMonth',
            'unapprovedLeavesThisMonth'
        )); // <-- هذا } يغلق دالة compact
    } // <-- هذا } يغلق دالة index
}