<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Foundation\Auth\RegistersUsers;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class RegisterController extends Controller
{
    use RegistersUsers;

    protected $redirectTo = '/home';

    public function __construct()
    {
        $this->middleware('guest');
    }

    protected function validator(array $data)
    {
        return Validator::make($data, [
            'name'     => ['required', 'string', 'max:255'],
            'email'    => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
            'role_id'  => ['required', 'integer', 'exists:roles,id'],
        ]);
    }

    protected function create(array $data)
    {
        // إنشاء المستخدم مباشرة بدون أدوار
        return DB::transaction(function () use ($data) {
            $user = User::create([
                'name'     => $data['name'],
                'email'    => $data['email'],
                'password' => Hash::make($data['password']),
                'status'   => 'active', // تفعيل المستخدم مباشرة
            ]);

            \Log::info("تم إنشاء مستخدم جديد: {$user->email}");

            // إذا تم تمرير role_id أثناء التسجيل، اربط الدور بالمستخدم
            if (isset($data['role_id'])) {
                try {
                    $roleId = (int) $data['role_id'];
                    $user->roles()->attach($roleId);
                } catch (\Throwable $e) {
                    // لا تفشل التسجيل بسبب خطأ ربط الدور أثناء الاختبارات
                    \Log::warning('Failed to attach role during registration: ' . $e->getMessage());
                }
            }

            return $user;
        });
    }

    public function showRegistrationForm()
    {
        // عرض فورم التسجيل مع قائمة الأدوار المتوفرة
        $roles = \App\Models\Role::orderBy('name')->get();
        return view('auth.register', compact('roles'));
    }

    /**
     * Handle a registration request for the application.
     */
    public function register(Request $request)
    {
        $this->validator($request->all())->validate();

        $user = $this->create($request->all());

        // تسجيل دخول المستخدم تلقائياً بعد التسجيل
        $this->guard()->login($user);

        // إعادة توجيه مع رسالة نجاح
        return $this->registered($request, $user)
                        ?: redirect($this->redirectPath())
                            ->with('success', 'تم إنشاء حسابك بنجاح! مرحباً بك في النظام.');
    }

    /**
     * The user has been registered.
     */
    protected function registered(Request $request, $user)
    {
        // إشعار ترحيبي بسيط (تجنّب إنشاء الإشعار أثناء تشغيل الـ tests إذا كانت بنية الجدول مختلفة)
        if (!app()->runningUnitTests()) {
            $user->notifications()->create([
                'type' => 'App\\Notifications\\WelcomeNotification',
                'data' => [
                    'message' => 'مرحباً بك في النظام! تم تفعيل حسابك بنجاح.',
                ]
            ]);
        }

        return null; // يكمل بالسلوك الافتراضي
    }
}
