<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\Employee;
use App\Models\Attendance;
use Carbon\Carbon;

class AttendanceImportController extends Controller
{
    public function showForm()
    {
        $this->authorize('import', Attendance::class);
        return view('attendance.import');
    }

    public function import(Request $request)
    {
        $this->authorize('import', Attendance::class);

        $validator = Validator::make($request->all(), [
            'file' => 'required|file|mimes:csv,txt',
            'identifier' => 'required|string', // which column to use to find employee (employee_id or national_id)
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $path = $request->file('file')->getRealPath();
        $identifier = $request->input('identifier');

        $rows = array_map('str_getcsv', file($path));
        if (count($rows) < 1) return redirect()->back()->with('error', 'ملف فارغ أو غير صالح');

        $headers = array_map('trim', $rows[0]);
        $created = 0;
        for ($i = 1; $i < count($rows); $i++) {
            $row = $rows[$i];
            if (count($row) < 1) continue;
            $data = array_combine($headers, $row);

            // Expect columns: identifier (employee_id), date (Y-m-d), check_in (H:i), check_out (H:i)
            $identValue = $data[$identifier] ?? null;
            $dateStr = $data['date'] ?? null;
            $checkIn = $data['check_in'] ?? null;
            $checkOut = $data['check_out'] ?? null;

            if (!$identValue || !$dateStr) continue;

            /** @var Employee|null $emp */
            $emp = Employee::where('employee_id', $identValue)->orWhere('national_id', $identValue)->first();
            if (!$emp) continue;

            try {
                $date = Carbon::createFromFormat('Y-m-d', $dateStr)->toDateString();
            } catch (\Throwable $_) {
                continue;
            }

            $attendance = Attendance::firstOrNew([
                'employee_id' => $emp->id,
                'date' => $date,
            ]);

            if ($checkIn) {
                $attendance->check_in = Carbon::parse($date . ' ' . $checkIn)->toDateTimeString();
            }
            if ($checkOut) {
                $attendance->check_out = Carbon::parse($date . ' ' . $checkOut)->toDateTimeString();
            }

            // set status based on check_in
            if ($attendance->check_in && !$attendance->check_out) {
                $attendance->status = 'present';
            } elseif ($attendance->check_in && $attendance->check_out) {
                $attendance->status = 'present';
            } else {
                $attendance->status = 'absent';
            }

            $attendance->employee_id = $emp->id;
            $attendance->save();
            $created++;
        }

        return redirect()->back()->with('success', "Imported {$created} attendance rows.");
    }
}
