<?php

namespace App\Http\Controllers;

use App\Models\Attendance;
use App\Models\Employee;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class AttendanceController extends Controller
{
    // ====================== INDEX ======================
    public function index()
    {
        // ✅ التحقق من الصلاحية العامة لعرض قائمة الحضور
        // وفقًا لـ AttendancePolicy::viewAny، يُسمح للموظف العادي برؤية القائمة
        $this->authorize('viewAny', Attendance::class);

        // ✅ جلب المستخدم الحالي
        $authenticatedUser = auth()->user();
        $isEmployee = $authenticatedUser->hasRole('employee');

        // ✅ تحديد استعلام سجلات الحضور
        $query = Attendance::with('employee');

        // ✅ تطبيق الفلتر إذا كان المستخدم موظفًا عاديًا ومرتبطًا بسجل موظف
        if ($isEmployee && $authenticatedUser->employee) {
            $query->where('employee_id', $authenticatedUser->employee->id);
        }

        // ⚠️ تطبيق الفلترات الإضافية (مثل البحث، التاريخ، الحالة، القسم) فقط إذا لم يكن المستخدم موظفًا عاديًا
        // لأن الفلترات تُستخدم عادةً من قبل الإدارة لعرض بيانات موظفين آخرين
        if (!$isEmployee) {
            if ($search = request('search')) {
                $query->whereHas('employee', function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('employee_id', 'like', "%{$search}%");
                });
            }

            if ($date = request('date')) {
                $query->whereDate('date', $date);
            }

            if ($status = request('status')) {
                $query->where('status', $status);
            }

            // --- إضافة فلترة حسب قسم الموظف ---
            if ($department = request('department')) {
                $query->whereHas('employee.department', function ($q) use ($department) {
                    $q->where('name', 'like', "%{$department}%");
                });
            }
            // --- النهاية ---
        }

        // --- إضافة فلترة حسب الشهر والسنة (متاحة للجميع، ولكن تُطبق على نتائج المستخدم فقط إذا كان موظفًا) ---
        if (request('month')) {
            $query->whereMonth('date', request('month'));
        }

        if (request('year')) {
            $query->whereYear('date', request('year'));
        }
        // --- النهاية ---

        // ✅ جلب البيانات مع التصفية
        $attendances = $query->latest()->paginate(15);

        // Prepare a map of approved leaves for the attendances displayed (keyed by "employeeId|date")
        $leaveKeys = [];
        foreach ($attendances as $att) {
            $key = $att->employee_id . '|' . ($att->date instanceof \Carbon\Carbon ? $att->date->toDateString() : (string) $att->date);
            $leaveKeys[$key] = ['employee_id' => $att->employee_id, 'date' => $att->date instanceof \Carbon\Carbon ? $att->date->toDateString() : (string) $att->date];
        }

        $leavesMap = [];
        if (!empty($leaveKeys)) {
            $employeeIds = array_unique(array_column($leaveKeys, 'employee_id'));
            // fetch leaves that cover the exact date for these employees
            $leaves = \App\Models\Leave::whereIn('employee_id', $employeeIds)
                ->where('status', 'approved')
                ->get();

            foreach ($leaves as $lv) {
                $start = $lv->start_date instanceof \Carbon\Carbon ? $lv->start_date->toDateString() : (string) $lv->start_date;
                $end = $lv->end_date instanceof \Carbon\Carbon ? $lv->end_date->toDateString() : (string) $lv->end_date;
                // for each date in the attendance list, check coverage
                foreach ($leaveKeys as $k => $v) {
                    if ($v['employee_id'] != $lv->employee_id) continue;
                    $d = $v['date'];
                    if ($d >= $start && $d <= $end) {
                        // store detailed leave info (prefer latest if multiple)
                        $leavesMap[$k] = [
                            'id' => $lv->id,
                            'reason' => $lv->reason ?? null,
                            'leave_type' => $lv->leave_type ?? null,
                            'is_unpaid' => $lv->is_unpaid_deduction_applied ?? false,
                        ];
                    }
                }
            }
        }

        // ✅ تحديد قائمة الموظفين لقائمة الفلترة (متوفرة فقط للإدارة)
        // 🔹 التعديل الأساسي: منع تمرير null إلى collect([...])
        $employees = collect();
        if ($isEmployee) {
            // إذا كان الموظف مرتبطًا بسجل employee، نضيفه فقط
            if ($authenticatedUser->employee) {
                $employees = collect([$authenticatedUser->employee]);
            } else {
                // إذا لم يكن مرتبطًا (مثل وجود user بدون employee)، نبقي القائمة فارغة
                $employees = collect();
            }
        } else {
            // للإدارة: جميع الموظفين
            $employees = Employee::orderBy('name')->get();
        }

        // ✅ قائمة الأقسام المُستمدة من الموظفين الآمنين
        $departments = $employees->pluck('department')->unique();

        // ⭐ إحصائيات اليوم (محدودة بصلاحية المستخدم)
        $today = now()->toDateString();

        // ✅ تطبيق الفلتر على الإحصائيات أيضًا إذا كان المستخدم موظفًا
        $todayQuery = Attendance::whereDate('date', $today);
        if ($isEmployee && $authenticatedUser->employee) {
            $todayQuery->where('employee_id', $authenticatedUser->employee->id);
        }

        $presentToday = $todayQuery->where('status', 'present')->count();
        $lateToday = $todayQuery->where('status', 'late')->count();
        // عدد الغياب المعلَن في صفوف الحضور (قد لا يعكس من لم يسجلوا دخولًا)
        $absentToday = $todayQuery->where('status', 'absent')->count();

        // حسابات إضافية: الموظفين الذين لم يُسجل لهم حضور اليوم
        $recordedEmployeeIds = Attendance::whereDate('date', $today)->pluck('employee_id')->unique()->filter()->values()->toArray();

        $noAttendanceEmployees = \App\Models\Employee::where('status', 'active')
            ->whereNotIn('id', $recordedEmployeeIds ? $recordedEmployeeIds : [0])
            ->get();

        // من هؤلاء، من لديهم إجازة معتمدة تغطي التاريخ
        $onLeaveCount = 0;
        $noAttendanceList = [];
        foreach ($noAttendanceEmployees as $emp) {
            $leave = \App\Models\Leave::where('employee_id', $emp->id)
                ->where('status', 'approved')
                ->whereDate('start_date', '<=', $today)
                ->whereDate('end_date', '>=', $today)
                ->orderBy('id', 'desc')
                ->first();

            $onLeave = (bool) $leave;
            if ($onLeave) $onLeaveCount++;

            // determine weekly off for this employee on $today
            $dayKey = \Carbon\Carbon::parse($today)->format('l');
            $dayKey = strtolower($dayKey); // sunday, monday, ...
            $isWeeklyOff = false;
            try {
                $weekly = $emp->weekly_off_days ?? [];
                if (is_array($weekly) && in_array($dayKey, $weekly)) {
                    $isWeeklyOff = true;
                }
            } catch (\Throwable $_) {
                $isWeeklyOff = false;
            }

            // check for official holiday on this date
            $holiday = \App\Models\Holiday::whereDate('date', $today)->first();
            $isHoliday = (bool) $holiday;

            $noAttendanceList[] = [
                'id' => $emp->id,
                'name' => $emp->name,
                'department' => $emp->department?->name ?? null,
                'on_leave' => $onLeave,
                'leave_id' => $leave ? $leave->id : null,
                'leave_reason' => $leave ? $leave->reason : null,
                'leave_type' => $leave ? $leave->leave_type : null,
                'leave_start' => $leave ? ($leave->start_date instanceof \Carbon\Carbon ? $leave->start_date->toDateString() : (string) $leave->start_date) : null,
                'leave_end' => $leave ? ($leave->end_date instanceof \Carbon\Carbon ? $leave->end_date->toDateString() : (string) $leave->end_date) : null,
                'is_weekly_off' => $isWeeklyOff,
                'is_holiday' => $isHoliday,
                'holiday_name' => $holiday ? $holiday->name : null,
            ];
        }

        $absentWithoutLeaveCount = max(0, count($noAttendanceList) - $onLeaveCount);
        $notRecordedCount = count($noAttendanceList);

        // ⭐ متوسط وقت الحضور (محدود بصلاحية المستخدم)
        $avgCheckInQuery = Attendance::whereDate('date', $today)->whereNotNull('check_in');
        if ($isEmployee && $authenticatedUser->employee) {
            $avgCheckInQuery->where('employee_id', $authenticatedUser->employee->id);
        }

        $avgCheckIn = $avgCheckInQuery->selectRaw('AVG(TIME_TO_SEC(check_in)) as avg_seconds')->first();

        $averageCheckInTime = '00:00';
        if ($avgCheckIn && $avgCheckIn->avg_seconds) {
            $totalSeconds = (int) $avgCheckIn->avg_seconds;
            $hours = floor($totalSeconds / 3600);
            $minutes = floor(($totalSeconds % 3600) / 60);
            $averageCheckInTime = sprintf('%02d:%02d', $hours, $minutes);
        }

        return view('attendance.index', compact(
            'attendances',
            'employees',
            'departments',
            'presentToday',
            'lateToday',
            'absentToday',
            'averageCheckInTime',
            'notRecordedCount',
            'absentWithoutLeaveCount',
            'onLeaveCount',
            'noAttendanceList'
        ))->with('leavesMap', $leavesMap);
    }

    // ====================== CREATE ======================
    public function create()
    {
        $employees = Employee::orderBy('name')->get();
        return view('attendance.create', compact('employees'));
    }

    // ====================== STORE ======================
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|exists:employees,id',
            'date' => 'required|date',
            'scheduled_check_in' => 'nullable|date_format:H:i',
            'scheduled_check_out' => 'nullable|date_format:H:i',
            'check_in' => 'nullable|date_format:H:i',
            'check_out' => 'nullable|date_format:H:i',
            'status' => 'required|in:present,absent,late,early_departure,present_on_official_holiday,present_on_weekly_off',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // احسب الحالة إذا لم تُحدد
        $status = $request->status;
        if (!$status) {
            if ($request->check_in) {
                $status = 'present';
            } else {
                $status = 'absent';
            }
        }

        $data = $request->only([
            'employee_id',
            'date',
            'scheduled_check_in',
            'scheduled_check_out',
            'check_in',
            'check_out',
            'notes'
        ]);

        // Determine if the date is an official holiday or weekly off for this employee
        $employee = Employee::findOrFail($request->employee_id);
        $dateOnly = Carbon::parse($data['date'])->toDateString();
        $isHoliday = \App\Models\Holiday::whereDate('date', $dateOnly)->exists();
        $dayNameMap = [1=>'monday',2=>'tuesday',3=>'wednesday',4=>'thursday',5=>'friday',6=>'saturday',7=>'sunday'];
        $dayName = Carbon::parse($dateOnly)->dayOfWeekIso;
        $isWeeklyOff = in_array($dayNameMap[$dayName], (array)($employee->weekly_off_days ?? []));

        // If status implies present, map to holiday-specific statuses when needed
        if (in_array($status, ['present'])) {
            if ($isHoliday) {
                $status = 'present_on_official_holiday';
                if (empty($data['notes'])) $data['notes'] = __('calendar.statuses.present_on_official_holiday');
            } elseif ($isWeeklyOff) {
                $status = 'present_on_weekly_off';
                if (empty($data['notes'])) $data['notes'] = __('calendar.statuses.present_on_weekly_off');
            }
        }

        $data['status'] = $status;

        // احسب مدة العمل الفعلية
        if ($data['check_in'] && $data['check_out']) {
            $in = Carbon::createFromFormat('H:i', $data['check_in']);
            $out = Carbon::createFromFormat('H:i', $data['check_out']);
            $duration = $in->diff($out);
            $data['work_duration'] = $duration->format('%H:%I:%S');
        }

        // احسب المدة المجدولة
        if ($data['scheduled_check_in'] && $data['scheduled_check_out']) {
            $in = Carbon::createFromFormat('H:i', $data['scheduled_check_in']);
            $out = Carbon::createFromFormat('H:i', $data['scheduled_check_out']);
            $duration = $in->diff($out);
            $data['scheduled_work_duration'] = $duration->format('%H:%I:%S');
        }

        Attendance::create($data);

        return redirect()->route('attendance.index')
            ->with('success', 'تم تسجيل الحضور بنجاح.');
    }

    // ====================== SHOW ======================
    public function show(Attendance $attendance)
    {
        $attendance->load('employee');
        $relatedAttendances = Attendance::where('employee_id', $attendance->employee_id)
            ->where('id', '!=', $attendance->id)
            ->orderBy('date', 'desc')
            ->limit(10)
            ->get();

        return view('attendance.show', compact('attendance', 'relatedAttendances'));
    }

    // ====================== EDIT ======================
    public function edit(Attendance $attendance)
    {
        $employees = Employee::orderBy('name')->get();
        $attendance->load('employee');
        return view('attendance.edit', compact('attendance', 'employees'));
    }

    // ====================== UPDATE ======================
    public function update(Request $request, Attendance $attendance)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|exists:employees,id',
            'date' => 'required|date',
            'scheduled_check_in' => 'nullable|date_format:H:i',
            'scheduled_check_out' => 'nullable|date_format:H:i',
            'check_in' => 'nullable|date_format:H:i',
            'check_out' => 'nullable|date_format:H:i',
            'status' => 'required|in:present,absent,late,early_departure,present_on_official_holiday,present_on_weekly_off',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $data = $request->only([
            'employee_id',
            'date',
            'scheduled_check_in',
            'scheduled_check_out',
            'check_in',
            'check_out',
            'notes'
        ]);

        // احسب الحالة
        $status = $request->status;
        if (!$status) {
            if ($data['check_in']) {
                $status = 'present';
            } else {
                $status = 'absent';
            }
        }

        // map present -> holiday-specific if date matches
        $employee = Employee::findOrFail($request->employee_id);
        $dateOnly = Carbon::parse($data['date'])->toDateString();
        $isHoliday = \App\Models\Holiday::whereDate('date', $dateOnly)->exists();
        $dayNameMap = [1=>'monday',2=>'tuesday',3=>'wednesday',4=>'thursday',5=>'friday',6=>'saturday',7=>'sunday'];
        $dayName = Carbon::parse($dateOnly)->dayOfWeekIso;
        $isWeeklyOff = in_array($dayNameMap[$dayName], (array)($employee->weekly_off_days ?? []));
        if (in_array($status, ['present'])) {
            if ($isHoliday) {
                $status = 'present_on_official_holiday';
                if (empty($data['notes'])) $data['notes'] = __('calendar.statuses.present_on_official_holiday');
            } elseif ($isWeeklyOff) {
                $status = 'present_on_weekly_off';
                if (empty($data['notes'])) $data['notes'] = __('calendar.statuses.present_on_weekly_off');
            }
        }
        $data['status'] = $status;

        // احسب مدة العمل
        if ($data['check_in'] && $data['check_out']) {
            $in = Carbon::createFromFormat('H:i', $data['check_in']);
            $out = Carbon::createFromFormat('H:i', $data['check_out']);
            $duration = $in->diff($out);
            $data['work_duration'] = $duration->format('%H:%I:%S');
        }

        // احسب المدة المجدولة
        if ($data['scheduled_check_in'] && $data['scheduled_check_out']) {
            $in = Carbon::createFromFormat('H:i', $data['scheduled_check_in']);
            $out = Carbon::createFromFormat('H:i', $data['scheduled_check_out']);
            $duration = $in->diff($out);
            $data['scheduled_work_duration'] = $duration->format('%H:%I:%S');
        }

        $attendance->update($data);

        return redirect()->route('attendance.index')
            ->with('success', 'تم تحديث سجل الحضور بنجاح.');
    }

    // ====================== DESTROY ======================
    public function destroy(Attendance $attendance)
    {
        $attendance->delete();

        return redirect()->route('attendance.index')
            ->with('success', 'تم حذف سجل الحضور بنجاح.');
    }

    // ====================== QUICK CHECK IN ======================
    public function quickCheckIn(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|exists:employees,id',
            'date' => 'nullable|date|date_format:Y-m-d',
            'check_in' => 'nullable|date_format:H:i',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ]);
        }

        // السماح بتسجيل حضور أي يوم ماضي أو الحالي
        $date = $request->has('date') && $request->filled('date') 
            ? $request->date 
            : now()->toDateString();
        
        $employee = Employee::findOrFail($request->employee_id);

        // تحقق من وجود سجل حضور لهذا اليوم
        $existing = Attendance::where('employee_id', $employee->id)
            ->whereDate('date', $date)
            ->first();

        if ($existing) {
            return response()->json([
                'success' => false,
                'message' => 'تم تسجيل الحضور مسبقًا لهذا اليوم.'
            ]);
        }

        $scheduledIn = $employee->scheduled_check_in ?? '09:00';
        $scheduledOut = $employee->scheduled_check_out ?? '17:00';
        
        // قبول check_in من الـ form إذا تم توفيره
        $checkInTime = $request->has('check_in') && $request->filled('check_in')
            ? $request->check_in
            : now()->format('H:i');

        $data = [
            'employee_id' => $employee->id,
            'date' => $date,
            'check_in' => $checkInTime,
            'scheduled_check_in' => $scheduledIn,
            'scheduled_check_out' => $scheduledOut,
            'scheduled_work_duration' => $this->calculateWorkDuration($scheduledIn, $scheduledOut),
            'status' => 'present',
            // Do not write a default note for quick check-in; leave notes null unless provided
        ];

        Attendance::create($data);

        return response()->json([
            'success' => true,
            'message' => 'تم تسجيل الحضور بنجاح!'
        ]);
    }

    // ====================== QUICK CHECK OUT ======================
    public function quickCheckOut(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|exists:employees,id',
            'date' => 'nullable|date|date_format:Y-m-d',
            'check_out' => 'nullable|date_format:H:i',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ]);
        }

        // السماح بتسجيل انصراف أي يوم ماضي أو الحالي
        $date = $request->has('date') && $request->filled('date') 
            ? $request->date 
            : now()->toDateString();
        
        $attendance = Attendance::where('employee_id', $request->employee_id)
            ->whereDate('date', $date)
            ->first();

        if (!$attendance) {
            return response()->json([
                'success' => false,
                'message' => 'لم يتم تسجيل الحضور بعد.'
            ]);
        }

        if ($attendance->check_out) {
            return response()->json([
                'success' => false,
                'message' => 'تم تسجيل الانصراف مسبقًا.'
            ]);
        }

        // قبول check_out من الـ form إذا تم توفيره
        $checkOutTime = $request->has('check_out') && $request->filled('check_out')
            ? $request->check_out
            : now()->format('H:i');

        $attendance->update([
            'check_out' => $checkOutTime,
            'status' => 'present', // يمكن تغييرها لاحقًا بناءً على التأخير
        ]);

        return response()->json([
            'success' => true,
            'message' => 'تم تسجيل الانصراف بنجاح!'
        ]);
    }

    // ====================== NOTIFY MISSING EMPLOYEE ======================
    public function notifyMissing(Request $request)
    {
        $this->validate($request, [
            'employee_id' => 'required|exists:employees,id',
            'date' => 'nullable|date',
        ]);

        $employee = Employee::find($request->employee_id);
        if (!$employee) {
            return response()->json(['success' => false, 'message' => 'الموظف غير موجود.'], 404);
        }

        $user = $employee->user;
        if (!$user) {
            return response()->json(['success' => false, 'message' => 'لا يوجد حساب مستخدم مرتبط بهذا الموظف. لا يمكن إرسال الإشعار.'], 422);
        }

        try {
            $initiator = auth()->user();
            \Notification::send($user, new \App\Notifications\AbsentReminderNotification($request->date, $initiator?->name ?? null));

            // سجل النشاط (اختياري)
            if (class_exists('\App\Models\Activity')) {
                try {
                    \App\Models\Activity::create([
                        'description' => 'sent_absent_reminder',
                        'properties' => ['employee_id' => $employee->id, 'date' => $request->date ?: now()->toDateString(), 'initiator' => $initiator?->id ?? null],
                    ]);
                } catch (\Throwable $e) {
                    // swallow
                }
            }

            return response()->json(['success' => true, 'message' => 'تم إرسال التذكير بنجاح.']);
        } catch (\Throwable $e) {
            \Log::error('notifyMissing failed: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => 'حدث خطأ أثناء إرسال التذكير.'], 500);
        }
    }

    /**
     * Return JSON list of employees without attendance records for a given date.
     * Accepts `date` (YYYY-MM-DD). If not provided, uses today.
     */
    public function noAttendanceList(Request $request)
    {
        try {
            $this->authorize('viewAny', Attendance::class);

            $this->validate($request, ['date' => 'nullable|date']);

            $date = $request->input('date') ? Carbon::parse($request->input('date'))->toDateString() : now()->toDateString();

            $recordedEmployeeIds = Attendance::whereDate('date', $date)->pluck('employee_id')->unique()->filter()->values()->toArray();

            $noAttendanceEmployees = \App\Models\Employee::where('status', 'active')
                ->whereNotIn('id', $recordedEmployeeIds ? $recordedEmployeeIds : [0])
                ->get();

            $noAttendanceList = [];
            $onLeaveCount = 0;
            foreach ($noAttendanceEmployees as $emp) {
                $leave = \App\Models\Leave::where('employee_id', $emp->id)
                    ->where('status', 'approved')
                    ->whereDate('start_date', '<=', $date)
                    ->whereDate('end_date', '>=', $date)
                    ->orderBy('id', 'desc')
                    ->first();

                $onLeave = (bool) $leave;
                if ($onLeave) $onLeaveCount++;

                // determine weekly off for this employee on $date
                $dayKey = \Carbon\Carbon::parse($date)->format('l');
                $dayKey = strtolower($dayKey);
                $isWeeklyOff = false;
                try {
                    $weekly = $emp->weekly_off_days ?? [];
                    if (is_array($weekly) && in_array($dayKey, $weekly)) {
                        $isWeeklyOff = true;
                    }
                } catch (\Throwable $_) {
                    $isWeeklyOff = false;
                }

                // check for official holiday on this date
                $holiday = \App\Models\Holiday::whereDate('date', $date)->first();
                $isHoliday = (bool) $holiday;

                $noAttendanceList[] = [
                    'id' => $emp->id,
                    'name' => $emp->name,
                    'department' => $emp->department?->name ?? null,
                    'on_leave' => $onLeave,
                    'leave_id' => $leave ? $leave->id : null,
                    'leave_reason' => $leave ? $leave->reason : null,
                    'leave_type' => $leave ? $leave->leave_type : null,
                    'leave_start' => $leave ? ($leave->start_date instanceof \Carbon\Carbon ? $leave->start_date->toDateString() : (string) $leave->start_date) : null,
                    'leave_end' => $leave ? ($leave->end_date instanceof \Carbon\Carbon ? $leave->end_date->toDateString() : (string) $leave->end_date) : null,
                    'is_weekly_off' => $isWeeklyOff,
                    'is_holiday' => $isHoliday,
                    'holiday_name' => $holiday ? $holiday->name : null,
                ];
            }

            return response()->json([
                'date' => $date,
                'count' => count($noAttendanceList),
                'on_leave_count' => $onLeaveCount,
                'list' => $noAttendanceList
            ]);
        } catch (\Illuminate\Auth\Access\AuthorizationException $e) {
            return response()->json(['success' => false, 'message' => 'غير مصرح بالوصول إلى بيانات الحضور.'], 403);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage(), 'errors' => $e->errors()], 422);
        } catch (\Throwable $e) {
            \Log::error('noAttendanceList error: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => 'حدث خطأ أثناء جلب البيانات.'], 500);
        }
    }

    /**
     * Mark an employee as absent without permission for today (or given date),
     * then deduct from monthly or annual leave balance if available. If balance
     * is exhausted, create a monetary deduction based on employee total salary.
     */
    public function markUnauthorizedLeave(Request $request, Employee $employee)
    {
        $this->authorize('create', Attendance::class);

        $this->validate($request, [
            'kind' => 'required|in:monthly,annual',
            'date' => 'nullable|date'
        ]);

        $date = $request->input('date') ? Carbon::parse($request->input('date'))->toDateString() : now()->toDateString();

        // prevent duplicate attendance record for same day unless forced
        $exists = Attendance::where('employee_id', $employee->id)->whereDate('date', $date)->exists();
        $force = (bool) $request->input('force', false);
        if ($exists && !$force) {
            return response()->json(['success' => false, 'message' => 'سجل الحضور موجود بالفعل لهذا اليوم. استخدم خيار التعزيز (force) لتطبيق الخصم مع الإبقاء على سجل الحضور.'], 422);
        }

        $kind = $request->input('kind'); // monthly or annual

        // use LeaveBalanceService to check available balance and apply approval
        $svc = new \App\Services\LeaveBalanceService();
        $available = $svc->availableBalance($employee, $kind, Carbon::parse($date));

        // create a Leave record if balance exists and deduct a single day
        if ($available >= 1) {
            $leaveType = \App\Models\LeaveType::where('code', $kind)->first();
            if (!$leaveType) {
                return response()->json(['success' => false, 'message' => 'نوع الإجازة غير معرف في النظام.'], 500);
            }

            $leave = \App\Models\Leave::create([
                'employee_id' => $employee->id,
                // store the code (consistent with other flows)
                'leave_type' => $leaveType->code,
                'leave_type_id' => $leaveType->id,
                'start_date' => $date,
                'end_date' => $date,
                'days_count' => 1,
                'reason' => 'غياب بدون إذن — خصم من الرصيد ' . $kind,
                'status' => 'approved',
                'approved_at' => now(),
                'approved_by' => auth()->id(),
            ]);

            // apply approval to create LeaveBalanceChange and snapshot
            try {
                $svc->applyApproval($leave, auth()->id());
            } catch (\Throwable $e) {
                \Log::error('applyApproval failed: ' . $e->getMessage());
            }

            // Activity log: unauthorized leave deducted from balance
            try {
                if (class_exists('\App\\Models\\Activity')) {
                    \App\Models\Activity::create([
                        'description' => 'unauthorized_leave_deducted_from_balance',
                        'properties' => [
                            'employee_id' => $employee->id,
                            'date' => $date,
                            'kind' => $kind,
                            'leave_id' => $leave->id ?? null,
                            'deducted_days' => 1,
                            'created_by' => auth()->id(),
                        ],
                    ]);
                }
            } catch (\Throwable $e) {
                \Log::error('Activity create failed (balance deduction): ' . $e->getMessage());
            }

            // if an attendance exists and force=true, link the leave to it and keep the attendance status (e.g. present)
            if ($exists) {
                $att = Attendance::where('employee_id', $employee->id)->whereDate('date', $date)->first();
                if ($att) {
                    $att->leave_id = $leave->id ?? null;
                    $att->notes = $att->notes ? ($att->notes . ' | ' . 'خصم من الرصيد ' . $kind) : ('خصم من الرصيد ' . $kind);
                    $att->save();
                }
            } else {
                // create an attendance record marking absent/leave (link to created leave)
                Attendance::create([
                    'employee_id' => $employee->id,
                    'date' => $date,
                    'status' => 'absent',
                    'leave_id' => $leave->id ?? null,
                    'notes' => 'غياب بدون إذن — خصم من الرصيد ' . $kind,
                ]);
            }

            return response()->json(['success' => true, 'message' => 'تم خصم يوم من رصيد ' . $kind . ' للموظف.']);
        }

        // if no available balance, create monetary deduction for one day
        $snapshotDate = \Carbon\Carbon::parse($date);
        $totalSalary = $employee->getTotalSalaryWithComponentsForDate($snapshotDate);
        $dailyRate = $totalSalary / 30.0;
        $amount = round($dailyRate, 2);

        $ded = \App\Models\Deduction::create([
            'employee_id' => $employee->id,
            'type' => 'monetary',
            'description' => 'خصم مقابل غياب بدون إذن - ' . $date,
            'amount' => $amount,
            'deduction_date' => $date,
            'status' => 'applied',
            'is_monthly' => false,
            'frequency' => 'once',
            'meta' => ['reason' => 'absent_without_permission', 'created_by' => auth()->id()],
        ]);

        // create attendance record as absent
        // create unpaid leave first so we can link it to attendance
        $unpaidLeave = null;
        $ltUnpaid = \App\Models\LeaveType::where('code', 'unpaid')->first();
        if ($ltUnpaid) {
            $unpaidLeave = \App\Models\Leave::create([
                'employee_id' => $employee->id,
                'leave_type' => 'unpaid',
                'leave_type_id' => $ltUnpaid->id,
                'start_date' => $date,
                'end_date' => $date,
                'days_count' => 1,
                'reason' => 'غياب بدون إذن — تم خصم مقابل مادي',
                'status' => 'approved',
                'approved_at' => now(),
                'approved_by' => auth()->id(),
                'is_unpaid_deduction_applied' => true,
            ]);

            // Activity log: unpaid leave created to reflect deduction
            try {
                if (class_exists('\App\\Models\\Activity')) {
                    \App\Models\Activity::create([
                        'description' => 'unauthorized_leave_unpaid_record_created',
                        'properties' => [
                            'employee_id' => $employee->id,
                            'date' => $date,
                            'leave_id' => $unpaidLeave->id ?? null,
                            'deduction_id' => $ded->id ?? null,
                            'created_by' => auth()->id(),
                        ],
                    ]);
                }
            } catch (\Throwable $e) {
                \Log::error('Activity create failed (unpaid leave): ' . $e->getMessage());
            }
        }

        if ($exists) {
            // link deduction/unpaid leave to existing attendance and keep presence
            $att = Attendance::where('employee_id', $employee->id)->whereDate('date', $date)->first();
            if ($att) {
                $att->leave_id = $unpaidLeave->id ?? null;
                $att->notes = $att->notes ? ($att->notes . ' | ' . 'خصم مقابل مادي') : 'خصم مقابل مادي';
                $att->save();
            }
        } else {
            Attendance::create([
                'employee_id' => $employee->id,
                'date' => $date,
                'status' => 'absent',
                'leave_id' => $unpaidLeave->id ?? null,
                'notes' => 'غياب بدون إذن — تم خصم مقابل مادي',
            ]);
        }
        // Activity log: deduction created
        try {
            if (class_exists('\App\\Models\\Activity')) {
                    \App\Models\Activity::create([
                    'description' => 'unauthorized_leave_deduction_created',
                    'properties' => [
                        'employee_id' => $employee->id,
                        'date' => $date,
                        'amount' => $amount,
                        'deduction_id' => $ded->id ?? null,
                        'created_by' => auth()->id(),
                    ],
                ]);
            }
        } catch (\Throwable $e) {
            \Log::error('Activity create failed (deduction): ' . $e->getMessage());
        }

        // Also create a Leave record (unpaid) so it appears in Leaves list / employee view
        $ltUnpaid = \App\Models\LeaveType::where('code', 'unpaid')->first();
        if ($ltUnpaid) {
            $unpaidLeave = \App\Models\Leave::create([
                'employee_id' => $employee->id,
                'leave_type' => 'unpaid',
                'leave_type_id' => $ltUnpaid->id,
                'start_date' => $date,
                'end_date' => $date,
                'days_count' => 1,
                'reason' => 'غياب بدون إذن — تم خصم مقابل مادي',
                'status' => 'approved',
                'approved_at' => now(),
                'approved_by' => auth()->id(),
                'is_unpaid_deduction_applied' => true,
            ]);

            // Activity log: unpaid leave created to reflect deduction
            try {
                if (class_exists('\App\\Models\\Activity')) {
                    \App\Models\Activity::create([
                        'description' => 'unauthorized_leave_unpaid_record_created',
                        'properties' => [
                            'employee_id' => $employee->id,
                            'date' => $date,
                            'leave_id' => $unpaidLeave->id ?? null,
                            'deduction_id' => $ded->id ?? null,
                            'created_by' => auth()->id(),
                        ],
                    ]);
                }
            } catch (\Throwable $e) {
                \Log::error('Activity create failed (unpaid leave): ' . $e->getMessage());
            }
        }

        return response()->json(['success' => true, 'message' => 'لا يوجد رصيد؛ تم إنشاء استقطاع بمبلغ ' . number_format($amount, 2) . '']);
    }

    // ====================== HELPERS ======================
    private function calculateWorkDuration($checkIn, $checkOut)
    {
        if (!$checkIn || !$checkOut) {
            return null;
        }

        try {
            $in = \Carbon\Carbon::createFromFormat('H:i', $checkIn);
            $out = \Carbon\Carbon::createFromFormat('H:i', $checkOut);
            $duration = $in->diff($out);
            return $duration->format('%H:%I:%S');
        } catch (\Exception $e) {
            return null;
        }
    }

    private function attendanceExists($employeeId, $date)
    {
        return Attendance::where('employee_id', $employeeId)
            ->whereDate('date', $date)
            ->exists();
    }

    // ====================== PRINT EMPLOYEE ATTENDANCE ======================
    /**
     * عرض صفحة الطباعة لسجل حضور موظف محدد.
     *
     * @param  int  $employeeId
     * @return \Illuminate\Http\Response
     */
    public function printEmployee($employeeId)
    {
        $employee = Employee::findOrFail($employeeId);
        // جلب سجلات الحضور للفترة المطلوبة (مثلاً آخر 6 أشهر)
        $startDate = Carbon::now()->subMonths(6)->startOfMonth();
        $endDate = Carbon::now()->endOfMonth();

        $attendances = $employee->attendances()
            ->whereBetween('date', [$startDate, $endDate])
            ->orderBy('date', 'desc')
            ->get();

        return view('attendance.print-employee', compact('employee', 'attendances'));
    }
}