<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Asset;
use App\Models\Employee;
use Illuminate\Support\Facades\Storage;

class AssetController extends Controller
{
    public function index()
    {
        $this->authorize('viewAny', Asset::class);
        
        $query = Asset::with('assignedTo');

        // تصفية البيانات
        if (request('search')) {
            $search = request('search');
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('serial_number', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        if (request('employee_id')) {
            $query->where('assigned_to', request('employee_id'));
        }

        if (request('category')) {
            $query->where('category', request('category'));
        }

        if (request('status')) {
            $query->where('status', request('status'));
        }

        $assets = $query->latest()->paginate(15);
        
        // قائمة الموظفين لاختيارهم في الفلاتر أو النماذج
        $employees = Employee::orderBy('name')->get();
        
        // إحصائيات سريعة
        $totalAssets     = Asset::count();
        $assignedAssets  = Asset::where('status', 'assigned')->count();
        $availableAssets = Asset::where('status', 'available')->count();
        $totalValue      = Asset::sum('value');

        return view('assets.index', compact(
            'assets', 
            'employees', 
            'totalAssets', 
            'assignedAssets', 
            'availableAssets', 
            'totalValue'
        ));
    }

    public function create()
    {
        $this->authorize('create', Asset::class);

        $employees = Employee::orderBy('name')->get();
        return view('assets.create', compact('employees'));
    }

    public function store(Request $request)
    {
        $this->authorize('create', Asset::class);
        
        $request->validate([
            'name'           => 'required|string|max:255',
            'category'       => 'required|in:electronic,furniture,vehicle,equipment,software',
            'description'    => 'nullable|string|max:500',
            'serial_number'  => 'nullable|string|max:100',
            'value'          => 'required|numeric|min:0',
            'purchase_date'  => 'nullable|date',
            'warranty_expiry'=> 'nullable|date|after:purchase_date',
            'location'       => 'nullable|string|max:255',
            'status'         => 'required|in:available,assigned,maintenance,retired,damaged',
            'photo'          => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048'
        ]);

        $data = $request->except('photo');

        // رفع الصورة إذا وُجدت
        if ($request->hasFile('photo')) {
            $data['photo'] = $request->file('photo')->store('assets', 'public');
        }

        Asset::create($data);

        return redirect()->route('assets.index')
                         ->with('success', 'تم إضافة الأصل بنجاح');
    }

    public function show(Asset $asset)
    {
        $this->authorize('view', $asset);

        $asset->load(['assignedTo', 'assignedTo.user']);
        return view('assets.show', compact('asset'));
    }

    public function edit(Asset $asset)
    {
        $this->authorize('update', $asset);

        $employees = Employee::orderBy('name')->get();
        $asset->load(['assignedTo', 'assignedTo.user']);

        return view('assets.edit', compact('asset', 'employees'));
    }

    public function update(Request $request, Asset $asset)
    {
        $this->authorize('update', $asset);
        
        $request->validate([
            'name'           => 'required|string|max:255',
            'category'       => 'required|in:electronic,furniture,vehicle,equipment,software',
            'description'    => 'nullable|string|max:500',
            'serial_number'  => 'nullable|string|max:100',
            'value'          => 'required|numeric|min:0',
            'purchase_date'  => 'nullable|date',
            'warranty_expiry'=> 'nullable|date|after:purchase_date',
            'location'       => 'nullable|string|max:255',
            'status'         => 'required|in:available,assigned,maintenance,retired,damaged',
            'photo'          => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048'
        ]);

        $data = $request->except('photo');

        // رفع صورة جديدة وحذف القديمة إن وُجدت
        if ($request->hasFile('photo')) {
            if ($asset->photo) {
                Storage::disk('public')->delete($asset->photo);
            }
            $data['photo'] = $request->file('photo')->store('assets', 'public');
        }

        $asset->update($data);

        return redirect()->route('assets.index')
                         ->with('success', 'تم تحديث الأصل بنجاح');
    }

    public function destroy(Asset $asset)
    {
        $this->authorize('delete', $asset);

        if ($asset->photo) {
            Storage::disk('public')->delete($asset->photo);
        }

        $asset->delete();

        return redirect()->route('assets.index')
                         ->with('success', 'تم حذف الأصل بنجاح');
    }

    public function assign(Request $request, Asset $asset)
    {
        $this->authorize('assign', $asset);

        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'assigned_at' => 'required|date',
            'notes'       => 'nullable|string|max:500'
        ]);

        $asset->update([
            'assigned_to'      => $request->employee_id,
            'status'           => 'assigned',
            'assigned_at'      => $request->assigned_at,
            'assignment_notes' => $request->notes
        ]);

        return redirect()->route('assets.index')
                         ->with('success', 'تم إسناد الأصل للموظف بنجاح');
    }

    public function unassign(Asset $asset)
    {
        $this->authorize('unassign', $asset);

        $asset->update([
            'assigned_to'      => null,
            'status'           => 'available',
            'assigned_at'      => null,
            'assignment_notes' => null
        ]);

        return redirect()->route('assets.index')
                         ->with('success', 'تم إلغاء إسناد الأصل بنجاح');
    }

    public function maintain(Asset $asset)
    {
        $this->authorize('maintain', $asset);

        $asset->update([
            'status'                 => 'maintenance',
            'maintenance_started_at' => now()
        ]);

        return redirect()->route('assets.index')
                         ->with('success', 'تم تحويل الأصل إلى الصيانة');
    }

    public function retire(Asset $asset)
    {
        $this->authorize('retire', $asset);

        $asset->update([
            'status'     => 'retired',
            'retired_at' => now()
        ]);

        return redirect()->route('assets.index')
                         ->with('success', 'تم إيقاف الأصل');
    }
}
