<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\User;
use App\Models\Permission;
use App\Models\Role;

class PermissionController extends Controller
{
    public function index()
    {
        $q = request('q');
        $query = User::with(['roles','permissionsRel'])->orderBy('id');
        if ($q) {
            $query->where(function($qr) use ($q) {
                $qr->where('name', 'like', "%{$q}%")->orWhere('email', 'like', "%{$q}%");
            });
        }

        $users = $query->paginate(25)->withQueryString();

        // Normalize relations to Collections to avoid array/collection inconsistencies
        foreach ($users as $user) {
            $user->roles = collect($user->roles ?? []);
            $user->permissionsRel = collect($user->permissionsRel ?? []);
            $user->deniedPermissions = collect($user->deniedPermissions ?? []);
            // ensure each role.permissions is a collection
            foreach ($user->roles as $role) {
                if (is_object($role)) {
                    $role->permissions = collect($role->permissions ?? []);
                } else {
                    // array role
                    $role['permissions'] = collect($role['permissions'] ?? []);
                }
            }
        }

        if (request()->wantsJson()) {
            return response()->json($users);
        }

        return view('admin.manage_permissions', compact('users'));
    }

    public function update(Request $request, $userId)
    {
        // Expect permissions to be sent as permissions[<id>] = allow|deny|default
        $request->validate([
            'permissions' => 'array',
            'role' => 'nullable|string'
        ]);

        $input = $request->input('permissions', []);

        // Safety rule: prevent super admin from removing their own 'manage_permissions' (deny)
        $auth = auth()->user();
        $managePerm = Permission::where('name', 'manage_permissions')->first();
        $preventRemovalMessage = null;
        if ($auth && $auth->id == $userId && $managePerm) {
            if (isset($input[$managePerm->id]) && $input[$managePerm->id] === 'deny') {
                // override to allow for self
                $input[$managePerm->id] = 'allow';
                $preventRemovalMessage = 'You cannot deny your own manage_permissions; it was kept.';
            }
        }

        DB::transaction(function () use ($userId, $input, $request) {
            // Clear existing explicit grants and denies
            DB::table('user_permission')->where('user_id', $userId)->delete();
            DB::table('user_permission_denied')->where('user_id', $userId)->delete();

            $grants = [];
            $denies = [];
            foreach ($input as $permId => $state) {
                $permId = (int) $permId;
                if ($state === 'allow') {
                    $grants[] = ['user_id' => $userId, 'permission_id' => $permId];
                } elseif ($state === 'deny') {
                    $denies[] = ['user_id' => $userId, 'permission_id' => $permId];
                }
            }

            if (! empty($grants)) {
                DB::table('user_permission')->insert($grants);
            }
            if (! empty($denies)) {
                DB::table('user_permission_denied')->insert($denies);
            }

            // Optionally update role if present
            $roleName = $request->input('role');
            if ($roleName !== null) {
                $roleId = DB::table('roles')->where('name', $roleName)->value('id');
                if ($roleId) {
                    DB::table('user_role')->where('user_id', $userId)->delete();
                    DB::table('user_role')->insert(['user_id' => $userId, 'role_id' => $roleId]);
                }
            }
        });

        if ($preventRemovalMessage) {
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json(['message' => $preventRemovalMessage], 200);
            }
            return redirect()->route('admin.permissions.index')->with('success', 'Permissions updated')->with('warning', $preventRemovalMessage);
        }

        if ($request->ajax() || $request->wantsJson()) {
            return response()->json(['message' => 'Permissions updated'], 200);
        }

        return redirect()->route('admin.permissions.index')->with('success', 'Permissions updated');
    }

    public function edit($userId)
    {
        // Eager-load roles and their permissions so we can show source badges
        $user = User::with(['roles.permissions','permissionsRel','deniedPermissions'])->findOrFail($userId);

        $permissions = Permission::orderBy('name')->get();
        $roles = Role::orderBy('name')->get();

        // Normalize relations to collections for view safety
        $user->roles = collect($user->roles ?? []);
        foreach ($user->roles as $role) {
            if (is_object($role)) $role->permissions = collect($role->permissions ?? []);
            else $role['permissions'] = collect($role['permissions'] ?? []);
        }
        $user->permissionsRel = collect($user->permissionsRel ?? []);
        $user->deniedPermissions = collect($user->deniedPermissions ?? []);

        $userPermissions = $user->permissionsRel->pluck('id')->toArray();
        $userDeniedPermissions = $user->deniedPermissions->pluck('id')->toArray();
        $userRole = $user->roles->first()?->name ?? null;

        return view('admin.edit_permissions', compact('user', 'permissions', 'roles', 'userPermissions', 'userDeniedPermissions', 'userRole'));
    }
}
