<?php
// app/Helpers/SettingsHelper.php

// تأكد من استيراد نموذج الإعدادات (Setting) إذا كنت تستخدم Eloquent
use App\Models\Setting;

/**
 * دالة لجلب قيمة إعداد معين من قاعدة البيانات.
 * * @param string $key مفتاح الإعداد المخزن في الجدول.
 * @param mixed $default القيمة الافتراضية إذا لم يتم العثور على الإعداد.
 * @return mixed
 */
if (!function_exists('get_setting')) {
    function get_setting(string $key, $default = null) {
        // نستخدم استعلام Eloquent العادي لجلب القيمة
        // يجب التأكد من وجود جدول 'settings' ونموذج 'Setting'
        try {
            $setting = Setting::where('key', $key)->first();
            return $setting ? $setting->value : $default;
        } catch (\Exception $e) {
            // في حال فشل الاتصال بقاعدة البيانات أو عدم وجود جدول Setting
            // نعود للقيمة الافتراضية ونطبع الخطأ في الـ Console للمطور
            error_log("Failed to load setting '{$key}': " . $e->getMessage());
            return $default;
        }
    }
}

/**
 * وظيفة مساعدة لجلب اسم مكون الراتب الديناميكي من بيانات الموظف.
 * هذا يحل مشكلة الدالة المفقودة (Call to undefined function).
 *
 * @param object|array $employee كائن الموظف أو مصفوفة خصائصه.
 * @param string $componentKey مفتاح المكون (مثال: 'component_1').
 * @return string|null اسم المكون (مثال: 'component_name_1')، أو قيمة خالية إذا لم يتم العثور عليه.
 */
if (!function_exists('get_salary_component_name')) {
    function get_salary_component_name($employee, $componentKey)
    {
        // نستخرج فهرس المكون (مثل '1' من 'component_1')
        $index = str_replace('component_', '', $componentKey);
        // نحدد المفتاح المقابل لاسم المكون (مثال: 'component_name_1')
        $nameKey = 'component_name_' . $index;

        // الوصول الآمن إلى قيمة الخاصية سواء كانت كائنًا أو مصفوفة
        $value = is_array($employee) ? ($employee[$nameKey] ?? null) : ($employee->{$nameKey} ?? null);

        // إرجاع القيمة إذا لم تكن فارغة
        return !empty($value) ? $value : null;
    }
}


/**
 * دالة لجلب رمز العملة الافتراضية (مثلاً SAR, EGP, USD).
 * تعتمد على قيمة 'default_currency' المخزنة في الإعدادات.
 * * @return string رمز العملة.
 */
if (!function_exists('get_currency_code')) {
    function get_currency_code() {
        // تأكد من إرجاع الكود بأحرف كبيرة لعرض موحد (EGP, USD, ...)
        return strtoupper((string) get_setting('default_currency', 'EGP'));
    }
}

/**
 * دالة لجلب اسم العملة الكامل (مثلاً "جنيه مصري" بدلاً من "EGP").
 *
 * @return string اسم العملة الكامل أو رمزها إذا لم يتم العثور على الاسم.
 */
if (!function_exists('get_currency_name')) {
    function get_currency_name() {
        $currencyCode = get_currency_code(); // استخدام الدالة المحدثة
        $currencyNames = [
            'EGP' => 'جنيه مصري',
            'USD' => 'دولار أمريكي',
            'EUR' => 'يورو',
            'SAR' => 'ريال سعودي',
            'AED' => 'درهم إماراتي',
            'QAR' => 'ريال قطري',
            // ... (بقية العملات) ...
            'GBP' => 'جنيه إسترليني',
            'JPY' => 'ين ياباني',
            'AUD' => 'دولار أسترالي',
            'CAD' => 'دولار كندي',
            'CHF' => 'فرنك سويسري',
            'CNY' => 'يوان صيني',
            'SEK' => 'كرونة سويدية',
            'NOK' => 'كرونة نرويجية',
            'DKK' => 'كرونة دانمركية',
            'PLN' => 'زلوتي بولندي',
            'INR' => 'روبية هندية',
            'BRL' => 'ريال برازيلي',
            'MXN' => 'بيزو مكسيكي',
            'ZAR' => 'راند جنوب أفريقي',
            'TRY' => 'ليرة تركية',
            'RUB' => 'روبل روسي',
            'KRW' => 'ون كوري جنوبي',
            'SGD' => 'دولار سنغافوري',
            'HKD' => 'دولار هونغ كونغ',
            'NZD' => 'دولار نيوزيلندي',
            'THB' => 'بات تايلاندي',
            'IDR' => 'روبية إندونيسية',
            'MYR' => 'رينغيت ماليزي',
            'PHP' => 'بيزو فلبيني',
            'VND' => 'دونغ فيتنامي',
            'KWD' => 'دينار كويتي',
            'BHD' => 'دينار بحريني',
            'OMR' => 'ريال عماني',
            'JOD' => 'دينار أردني',
            'ILS' => 'شيكل إسرائيلي',
            'PKR' => 'روبية باكستانية',
            'BDT' => 'تكا بنغلاديشية',
            'LKR' => 'روبية سيلانكية',
            'MAD' => 'درهم مغربي',
            'TND' => 'دينار تونسي',
            'DZD' => 'دينار جزائري',
            'LYD' => 'دينار ليبي',
            'SDG' => 'جنيه سوداني',
            'NGN' => 'نايرا نيجيري',
            'GHS' => 'سيدي غاني',
            'TZS' => 'شيلينغ تنزاني',
            'KES' => 'شيلينغ كيني',
            'UGX' => 'شيلينغ أوغندي',
            'ETB' => 'بير إثيوبي',
            'AOA' => 'كوانزا أنغولي',
            'ZMW' => 'كواشا زامبي',
            'MWK' => 'كواشا ملاوي',
            'BWP' => 'بولا بوتسوانا',
            'MUR' => 'روبية موريشيوسية',
            'SCR' => 'روبية سيشلية',
            'CVE' => 'إسكودو كابو فيريان',
            'STN' => 'دوبرا ساو تومي وبرينسيبي',
            'GMD' => 'داﻻسي غامبي',
            'GNF' => 'فرنك غيني',
            'LRD' => 'دولار ليبيري',
            'SLL' => 'ليون سيراليوني',
            'CDF' => 'فرنك كونغولي',
            'RWF' => 'فرنك رواندي',
            'BIF' => 'فرنك بوروندي',
            'DJF' => 'فرنك جيبوتي',
            'SOS' => 'شيلينغ صومالي',
            'KMF' => 'فرنك كوموري',
            'MRU' => 'أوقية موريتانية',
            'XAF' => 'فرنك CFA (BEAC)',
            'XOF' => 'فرنك CFA (BCEAO)',
            'XPF' => 'فرنك CFP',
            // أضف أي عملات أخرى تحتاجها هنا
        ];
        // ملاحظة: هذه الدالة كانت تعيد اسم العملة الكامل.
        // لتوحيد العرض في كامل التطبيق حسب طلب المستخدم، نُرجع رمز العملة (مثل EGP, USD)
        // إذا كنت بحاجة إلى الاسم الكامل في مكان ما، استخدم get_currency_full_name().
        return $currencyCode;
    }
}

if (!function_exists('get_currency_full_name')) {
    function get_currency_full_name() {
        $currencyCode = get_currency_code();
        $currencyNames = [
            'EGP' => 'جنيه مصري',
            'USD' => 'دولار أمريكي',
            'EUR' => 'يورو',
            'SAR' => 'ريال سعودي',
            'AED' => 'درهم إماراتي',
            'QAR' => 'ريال قطري',
            // ... بقية القائمة كما كانت سابقًا ...
            'GBP' => 'جنيه إسترليني',
            'JPY' => 'ين ياباني',
            'AUD' => 'دولار أسترالي',
            'CAD' => 'دولار كندي',
            'CHF' => 'فرنك سويسري',
            'CNY' => 'يوان صيني',
        ];

        return $currencyNames[$currencyCode] ?? $currencyCode;
    }
}

/**
 * دالة لتنسيق المبلغ المالي مع رمز أو اسم العملة.
 *
 * @param float $amount المبلغ المراد تنسيقه.
 * @param bool $showCode إذا كانت 'true' تُظهر رمز العملة، وإلا تُظهر اسمها الكامل.
 * @return string المبلغ المنسق مع اسم/رمز العملة.
 */
if (!function_exists('format_currency')) {
    /**
     * تنسيق المبلغ مع عرض رمز العملة بشكل افتراضي (EGP, USD).
     * استخدم المعامل الثاني false لعرض الاسم الكامل عبر get_currency_full_name().
     */
    function format_currency($amount, $showCode = true) {
        $currencyCode = get_currency_code();

        $formattedAmount = number_format((float)$amount, 2, '.', ','); 

        if ($showCode) {
            $currencyDisplay = $currencyCode;
        } else {
            $currencyDisplay = get_currency_full_name();
        }

        return $formattedAmount . ' ' . $currencyDisplay;
    }
}

/**
 * وظيفة مساعدة لجلب اسم مكون الراتب الثابت (مثل الراتب الأساسي، أو الحوافز).
 * تستخدم لضمان توحيد الاسم الظاهر للمستخدم في جميع واجهات العرض والطباعة.
 *
 * [✅ تم إضافة هذه الدالة لتطبيق تعديل "الحوافز" عالميًا في الكود]
 *
 * @param string $componentKey مفتاح المكون (مثل 'salary', 'allowances').
 * @return string الاسم العربي للمكون.
 */
if (!function_exists('get_static_salary_component_name')) {
    function get_static_salary_component_name(string $componentKey): string
    {
        $names = [
            'salary' => 'الراتب الأساسي',
            'allowances' => 'الحوافز', // ✅ التعديل المطلوب هنا: من 'البدلات' إلى 'الحوافز'
            // يمكنك إضافة المزيد من المكونات الثابتة هنا إذا لزم الأمر
            'leave_deduction' => 'خصم الإجازة',
        ];

        return $names[$componentKey] ?? $componentKey;
    }
}


if (!function_exists('position_label')) {
    /**
     * Return a translated label for a Position model if available.
     * Falls back to the position's name when no translation exists.
     * Usage in Blade: `position_label($employee->position)`
     *
     * @param  mixed $position Eloquent Position model or null
     * @return string|null
     */
    function position_label($position)
    {
        if (!$position) return null;

        $id = null;
        $name = null;
        if (is_object($position)) {
            $id = $position->id ?? null;
            $name = $position->name ?? null;
        } elseif (is_array($position)) {
            $id = $position['id'] ?? null;
            $name = $position['name'] ?? null;
        }

        // Try translation by position id first (recommended approach)
        if ($id && \Illuminate\Support\Facades\Lang::has('positions.' . $id)) {
            return trans('positions.' . $id);
        }

        // Try translation by slugified name (if you maintain translations by slug)
        if ($name) {
            $slug = \Illuminate\Support\Str::slug($name, '_');
            if (\Illuminate\Support\Facades\Lang::has('positions.' . $slug)) {
                return trans('positions.' . $slug);
            }
        }

        // Fallback to raw name
        return $name;
    }
}


// ===============================================
// تذكير مهم: لحل المشكلة بشكل كامل، يجب إضافة مسار 
// هذا الملف إلى قسم 'autoload/files' في composer.json 
// ثم تشغيل الأمر: composer dump-autoload
// ===============================================