<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class TrimRoleUsers extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'role:trim-users {--limit=10} {--keep-pattern}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Trim user-role pivots so each role has at most N users (default 10). Does not delete users.';

    public function handle()
    {
        $limit = (int) $this->option('limit');
        $usePattern = (bool) $this->option('keep-pattern');

        $roles = DB::table('roles')->select('id', 'name')->get();

        foreach ($roles as $role) {
            $this->line("Processing role: {$role->name} (id={$role->id})");

            // Two possible pivot table names exist in this project: user_role and user_roles
            $pivots = ['user_roles', 'user_role'];

            foreach ($pivots as $pivot) {
                if (! Schema::hasTable($pivot)) {
                    continue;
                }

                $this->line(" - Checking pivot table: $pivot");

                // Fetch all users attached to this role
                $query = DB::table($pivot)
                    ->where('role_id', $role->id)
                    ->join('users', $pivot . '.user_id', '=', 'users.id')
                    ->select($pivot . '.user_id', 'users.email');

                // If pivot has an auto-increment id column, include it for stable ordering
                $hasId = Schema::hasColumn($pivot, 'id');
                if ($hasId) {
                    $query->addSelect($pivot . '.id as pivot_id');
                }

                $rows = $query->get()->toArray();

                $rowsCount = count($rows);
                if ($rowsCount <= $limit) {
                    $this->line("   -> OK ({$rowsCount})");
                    continue;
                }

                // Prefer to keep users with the standard seeded emails: role{n}@example.com
                $patternMatches = [];
                $others = [];

                foreach ($rows as $r) {
                    if (preg_match('/^' . preg_quote($role->name, '/') . '(\d+)@example\.com$/', $r->email, $m)) {
                        $patternMatches[] = ['user_id' => $r->user_id, 'num' => (int) $m[1], 'pivot_id' => $hasId ? $r->pivot_id : null];
                    } else {
                        $others[] = ['user_id' => $r->user_id, 'pivot_id' => $hasId ? $r->pivot_id : null];
                    }
                }

                // Sort pattern matches by the numeric suffix
                usort($patternMatches, function ($a, $b) {
                    return $a['num'] <=> $b['num'];
                });

                // Sort others by pivot_id if available, else by user_id
                usort($others, function ($a, $b) {
                    if ($a['pivot_id'] !== null && $b['pivot_id'] !== null) {
                        return $a['pivot_id'] <=> $b['pivot_id'];
                    }
                    return $a['user_id'] <=> $b['user_id'];
                });

                $keep = [];
                foreach ($patternMatches as $p) {
                    if (count($keep) >= $limit) break;
                    $keep[] = $p['user_id'];
                }

                foreach ($others as $o) {
                    if (count($keep) >= $limit) break;
                    if (! in_array($o['user_id'], $keep, true)) $keep[] = $o['user_id'];
                }

                // Now delete any pivot rows not in $keep
                $allUserIds = array_map(function ($r) { return $r->user_id; }, $rows);
                $toRemove = array_values(array_diff($allUserIds, $keep));

                if (empty($toRemove)) {
                    $this->line('   -> Nothing to remove after selection.');
                    continue;
                }

                $this->line('   -> Removing ' . count($toRemove) . ' pivot(s)');

                // Backup rows before deletion
                $ts = \Carbon\Carbon::now()->format('Ymd_His');
                $backupTable = $pivot . '_backup_' . $ts;

                if (! Schema::hasTable($backupTable)) {
                    Schema::create($backupTable, function ($table) use ($pivot) {
                        $table->id();
                        $table->unsignedBigInteger('user_id');
                        $table->unsignedBigInteger('role_id');
                        $table->unsignedBigInteger('pivot_id')->nullable();
                        $table->timestamp('backed_up_at')->nullable();
                    });
                    $this->line("   -> Created backup table: $backupTable");
                }

                // Build source query for insertion
                $hasId = Schema::hasColumn($pivot, 'id');
                if ($hasId) {
                    $source = DB::table($pivot)
                        ->where('role_id', $role->id)
                        ->whereIn('user_id', $toRemove)
                        ->select($pivot . '.user_id', $pivot . '.role_id', $pivot . '.id as pivot_id');
                } else {
                    $source = DB::table($pivot)
                        ->where('role_id', $role->id)
                        ->whereIn('user_id', $toRemove)
                        ->select($pivot . '.user_id', $pivot . '.role_id', DB::raw('NULL as pivot_id'));
                }

                // insert using backed_up_at value
                $now = Carbon::now()->toDateTimeString();
                $insertQuery = DB::table($backupTable)->insertUsing(
                    ['user_id', 'role_id', 'pivot_id', 'backed_up_at'],
                    $source->selectRaw("{$pivot}.user_id as user_id, {$pivot}.role_id as role_id, " . ($hasId ? "{$pivot}.id as pivot_id" : "NULL as pivot_id") . ", '{$now}' as backed_up_at")
                );

                $this->line("   -> Backed up " . count($toRemove) . " rows into $backupTable");

                // Delete original rows
                DB::table($pivot)->where('role_id', $role->id)->whereIn('user_id', $toRemove)->delete();
            }
        }

        $this->info('Trimming complete.');
        return 0;
    }
}
