<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Employee;
use App\Models\Attendance;
use Carbon\Carbon;

class TestAttendanceImport extends Command
{
    protected $signature = 'test:attendance {--file=storage/app/tmp_attendance.csv}';
    protected $description = 'Run a test import of attendance from a CSV file (for debugging)';

    public function handle()
    {
        $path = base_path($this->option('file'));
        if (!file_exists($path)) {
            $this->error("File not found: {$path}");
            return 1;
        }

        $rows = array_map('str_getcsv', file($path));
        if (count($rows) < 1) {
            $this->error('Empty or invalid CSV');
            return 1;
        }

        $headers = array_map('trim', $rows[0]);
        $created = 0;
        $skipped = 0;

        for ($i = 1; $i < count($rows); $i++) {
            $row = $rows[$i];
            if (count($row) < 1) continue;
            $data = array_combine($headers, $row);

            $identValue = $data['employee_id'] ?? ($data['national_id'] ?? null);
            $dateStr = $data['date'] ?? null;
            $checkIn = $data['check_in'] ?? null;
            $checkOut = $data['check_out'] ?? null;

            if (!$identValue || !$dateStr) {
                $skipped++;
                continue;
            }

            $emp = Employee::where('employee_id', $identValue)->orWhere('national_id', $identValue)->first();
            if (!$emp) {
                $this->line("Skipping row {$i}: employee not found ({$identValue})");
                $skipped++;
                continue;
            }

            try {
                $date = Carbon::createFromFormat('Y-m-d', $dateStr)->toDateString();
            } catch (\Throwable $_) {
                $this->line("Skipping row {$i}: invalid date ({$dateStr})");
                $skipped++;
                continue;
            }

            $attendance = Attendance::firstOrNew([
                'employee_id' => $emp->id,
                'date' => $date,
            ]);

            if ($checkIn) {
                $attendance->check_in = Carbon::parse($date . ' ' . $checkIn)->toDateTimeString();
            }
            if ($checkOut) {
                $attendance->check_out = Carbon::parse($date . ' ' . $checkOut)->toDateTimeString();
            }

            if ($attendance->check_in && !$attendance->check_out) {
                $attendance->status = 'present';
            } elseif ($attendance->check_in && $attendance->check_out) {
                $attendance->status = 'present';
            } else {
                $attendance->status = 'absent';
            }

            $attendance->employee_id = $emp->id;
            $attendance->save();
            $this->info("Row {$i}: saved attendance for employee_id={$emp->id} date={$date} status={$attendance->status}");
            $created++;
        }

        $this->info("Import complete. Created: {$created}, Skipped: {$skipped}");
        return 0;
    }
}
