<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class SyncPermissionsFromCode extends Command
{
    protected $signature = 'sync:permissions {--apply : Actually insert missing permissions into DB}';
    protected $description = 'Scan codebase for permission strings (\'can\', \"@can\") and insert missing ones into permissions table.';

    public function handle()
    {
        $this->info('Scanning project for permission strings...');

        $paths = [base_path('app'), base_path('resources'), base_path('routes'), base_path('tests'), base_path('database')];
        $patterns = [
            "@can\\('([^']+)'\\)",
            "->can\\('([^']+)'\\)",
            "can\\('([^']+)'\\)",
            "Gate::allows\\('([^']+)'\\)",
            "Gate::denies\\('([^']+)'\\)",
            "\$user->can\\('([^']+)'\\)",
        ];

        $found = [];

        foreach ($paths as $path) {
            if (!is_dir($path)) continue;
            $rii = new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator($path));
            foreach ($rii as $file) {
                if ($file->isDir()) continue;
                $fname = $file->getFilename();
                // limit to relevant extensions
                if (!preg_match('/\\.(php|blade\\.php|js|ts|vue)$/i', $fname)) continue;
                $content = @file_get_contents($file->getPathname());
                if ($content === false) continue;
                foreach ($patterns as $pat) {
                    if (preg_match_all("/{$pat}/m", $content, $m)) {
                        foreach ($m[1] as $perm) {
                            $perm = trim($perm);
                            if ($perm === '') continue;
                            $found[$perm] = true;
                        }
                    }
                }
            }
        }

        $perms = array_keys($found);
        sort($perms);

        if (empty($perms)) {
            $this->info('No permission strings found.');
            return 0;
        }

        $this->info('Found '.count($perms).' unique permission strings:');
        foreach ($perms as $p) {
            $this->line(' - '.$p);
        }

        $missing = [];
        foreach ($perms as $p) {
            $exists = DB::table('permissions')->where('name', $p)->exists();
            if (! $exists) $missing[] = $p;
        }

        if (empty($missing)) {
            $this->info('All discovered permissions already exist in DB.');
            return 0;
        }

        $this->info('Missing permissions ('.count($missing).') detected:');
        foreach ($missing as $m) $this->line(' * '.$m);

        if ($this->option('apply')) {
            $now = now();
            // detect which columns exist in the permissions table
            try {
                $tableColumns = Schema::hasTable('permissions') ? Schema::getColumnListing('permissions') : [];
            } catch (\Throwable $e) {
                // fallback: try to get from DB schema builder
                try {
                    $tableColumns = DB::getSchemaBuilder()->getColumnListing('permissions');
                } catch (\Throwable $e2) {
                    $tableColumns = [];
                }
            }

            $insert = [];
            foreach ($missing as $m) {
                $row = [];
                if (in_array('name', $tableColumns)) $row['name'] = $m;
                if (in_array('display_name', $tableColumns)) $row['display_name'] = $m;
                if (in_array('created_at', $tableColumns)) $row['created_at'] = $now;
                if (in_array('updated_at', $tableColumns)) $row['updated_at'] = $now;
                // if permissions table has additional required fields, you may need to extend logic here
                $insert[] = $row;
            }

            if (!empty($insert)) {
                DB::table('permissions')->insert($insert);
                $this->info('Inserted '.count($insert).' missing permissions.');
            } else {
                $this->error('Could not determine columns for `permissions` table or no insertable columns found.');
            }
        } else {
            $this->info('Run `php artisan sync:permissions --apply` to insert the missing permissions into DB.');
        }

        return 0;
    }
}
