<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Payroll;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class RecalculatePayrolls extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'payroll:recalculate {--apply : Actually save changes to DB} {--id= : Recalculate a specific payroll id} {--employee= : Employee id or comma-separated ids to limit} {--since= : ISO date (YYYY-MM-DD) to process payrolls created on/after this date} {--chunk=200 : Chunk size}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Recalculate total_salary and net_salary for payrolls using payroll components. Use --apply to persist changes.';

    public function handle()
    {
        $apply = $this->option('apply');
        $id = $this->option('id');
        $employeeOpt = $this->option('employee');
        $sinceOpt = $this->option('since');
        $chunkSize = (int) $this->option('chunk');

        $query = Payroll::with('payrollSalaryComponents');

        // Apply filters
        if ($id) {
            $query->where('id', $id);
        }

        if ($employeeOpt) {
            // accept comma-separated ids
            $ids = array_filter(array_map('trim', explode(',', $employeeOpt)));
            $ids = array_map('intval', $ids);
            if (!empty($ids)) {
                $query->whereIn('employee_id', $ids);
            }
        }

        if ($sinceOpt) {
            try {
                $sinceDate = Carbon::parse($sinceOpt)->startOfDay();
                $query->where('created_at', '>=', $sinceDate);
            } catch (\Exception $e) {
                $this->error("Invalid --since date format. Use YYYY-MM-DD.");
                return 1;
            }
        }

        $filterSummary = '';
        if ($id) $filterSummary .= " id={$id}";
        if ($employeeOpt) $filterSummary .= " employee={$employeeOpt}";
        if ($sinceOpt) $filterSummary .= " since={$sinceOpt}";

        $this->info('Starting payroll recalc'.($apply ? ' (will apply changes)' : ' (dry-run)') . ($filterSummary ? ' Filters:' . $filterSummary : ''));

        $total = 0;
        $changed = 0;

        $query->chunk($chunkSize, function($rows) use (&$total, &$changed, $apply) {
            foreach ($rows as $p) {
                $total++;
                // compute components sum
                $componentsSum = $p->relationLoaded('payrollSalaryComponents') ? $p->payrollSalaryComponents->sum('value') : $p->payrollSalaryComponents()->sum('value');
                $computedTotal = (float) (($p->basic_salary ?? 0) + ($p->incentives ?? 0) + $componentsSum);
                $deductions = ($p->active_deductions ?? 0) + ($p->loan_installments ?? 0) + ($p->leave_deduction ?? 0) + ($p->deductions ?? 0);
                $computedNet = (float) max(0, $computedTotal - $deductions);

                if (abs($p->total_salary - $computedTotal) > 0.009 || abs($p->net_salary - $computedNet) > 0.009) {
                    $changed++;
                    $this->line("Payroll ID {$p->id} ({$p->month}/{$p->year}): saved_total={$p->total_salary} => computed_total={$computedTotal}, saved_net={$p->net_salary} => computed_net={$computedNet}");
                    if ($apply) {
                        // wrap in transaction for safety per payroll
                        DB::transaction(function() use ($p, $computedTotal, $computedNet) {
                            $p->total_salary = $computedTotal;
                            $p->net_salary = $computedNet;
                            $p->save();
                        });
                    }
                }
            }
        });

        $this->info("Processed: {$total} payroll(s). Changes detected: {$changed}.");

        if (!$apply && $changed > 0) {
            $this->warn('Dry-run only: no changes were saved. Rerun with --apply to persist.');
        }

        return 0;
    }
}
