<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Payroll;
use App\Models\Employee;
use App\Models\Deduction;
use App\Models\Loan;
use App\Models\Leave;
use Carbon\Carbon;

class RecalculatePayroll extends Command
{
    protected $signature = 'payroll:recalculate {payrollId}';
    protected $description = 'إعادة حساب راتب معين بناءً على البيانات الحالية في قاعدة البيانات.';

    public function handle()
    {
        $payrollId = $this->argument('payrollId');
        $payroll = Payroll::find($payrollId);

        if (!$payroll) {
            $this->error("لم يتم العثور على راتب بالرقم {$payrollId}.");
            return;
        }

        $employee = Employee::find($payroll->employee_id);

        if (!$employee) {
            $this->error("لم يتم العثور على موظف بالرقم {$payroll->employee_id}.");
            return;
        }

        $year = $payroll->year;
        $month = $payroll->month;

        // حساب الاستقطاعات النشطة
        $activeDeductions = $this->getActiveDeductionsForPayroll($employee, $year, $month);

        // حساب أقساط القروض
        $loanInstallments = $this->getTotalLoanInstallmentsForPayroll($employee, $year, $month);

        // حساب خصم الإجازات بالتراكمي الشهري من تاريخ التعيين
        $leaveDeduction = $this->calculateLeaveDeductionBasedOnCumulativeFromHireDate($employee, $year, $month);

        // حساب الراتب الصافي
        $basicSalary = $employee->salary ?? 0;
        $allowances = $employee->allowances ?? 0;
        $otherDeductions = $payroll->deductions ?? 0;

        $netSalary = $basicSalary + $allowances - $activeDeductions - $loanInstallments - $leaveDeduction - $otherDeductions;

        $payroll->update([
            'basic_salary' => $basicSalary,
            'allowances' => $allowances,
            'active_deductions' => $activeDeductions,
            'loan_installments' => $loanInstallments,
            'leave_deduction' => $leaveDeduction,
            'deductions' => $otherDeductions,
            'net_salary' => $netSalary,
        ]);

        $this->info("✅ تم إعادة حساب الراتب رقم {$payrollId} بنجاح.");
    }

    private function getActiveDeductionsForPayroll(Employee $employee, $year, $month)
    {
        $startDate = Carbon::create($year, $month, 1)->startOfMonth();
        $endDate = (clone $startDate)->endOfMonth();

        return Deduction::where('employee_id', $employee->id)
            ->where('status', 'applied')
            ->where(function ($q) use ($startDate, $endDate) {
                $q->where(function ($qq) use ($startDate, $endDate) {
                    $qq->whereNull('is_monthly')
                       ->orWhere('is_monthly', false);
                    $qq->whereBetween('deduction_date', [$startDate, $endDate]);
                })
                ->orWhere(function ($qq) use ($startDate) {
                    $qq->where('is_monthly', true)
                       ->whereDate('deduction_date', '<=', $startDate)
                       ->where(function ($qqq) use ($startDate) {
                           $qqq->whereNull('end_date')
                               ->orWhereDate('end_date', '>=', $startDate);
                       });
                });
            })
            ->sum('amount');
    }

    private function getTotalLoanInstallmentsForPayroll(Employee $employee, $year, $month)
    {
        $startDate = Carbon::create($year, $month, 1)->startOfMonth();
        $endDate = (clone $startDate)->endOfMonth();

        return $employee->loans()
            ->where('status', 'active')
            ->with(['installments' => function ($q) use ($startDate, $endDate) {
                $q->whereBetween('due_date', [$startDate, $endDate])
                  ->where(function ($qq) {
                      $qq->whereNull('paid_date')
                         ->orWhere('status', '!=', 'paid');
                  });
            }])
            ->get()
            ->pluck('installments')
            ->flatten()
            ->sum('amount');
    }

    private function getUsedLeaveDaysInMonth(Employee $employee, $year, $month)
    {
        $startOfMonth = Carbon::create($year, $month, 1)->startOfMonth();
        $endOfMonth = (clone $startOfMonth)->endOfMonth();

        $leaves = Leave::where('employee_id', $employee->id)
            ->where('status', 'approved')
            ->where(function ($q) use ($startOfMonth, $endOfMonth) {
                $q->whereBetween('start_date', [$startOfMonth, $endOfMonth])
                  ->orWhereBetween('end_date', [$startOfMonth, $endOfMonth])
                  ->orWhere(function ($qq) use ($startOfMonth, $endOfMonth) {
                      $qq->where('start_date', '<', $startOfMonth)
                         ->where('end_date', '>', $endOfMonth);
                  });
            })->get();

        $days = 0;
        foreach ($leaves as $leave) {
            $s = Carbon::parse($leave->start_date)->max($startOfMonth);
            $e = Carbon::parse($leave->end_date)->min($endOfMonth);
            $days += $e->diffInDays($s) + 1;
        }

        return $days;
    }

    /**
     * حساب خصم الإجازات بالتراكمي الشهري من تاريخ التعيين
     */
    private function calculateLeaveDeductionBasedOnCumulativeFromHireDate(Employee $employee, $year, $month)
    {
        if (!$employee->hire_date) {
            return 0;
        }

        $hireDate = Carbon::parse($employee->hire_date);
        $currentDate = Carbon::create($year, $month, 1)->endOfMonth();

        // عدد الأشهر من التعيين حتى الشهر الحالي
        $monthsWorked = $hireDate->diffInMonths($currentDate) + 1;

        // الأيام الممنوحة حتى الشهر الحالي
        $monthlyGranted = $employee->monthly_leave_days_allowed ?? 0;
        $totalGrantedDays = $monthlyGranted * $monthsWorked;

        // مجموع الأيام المستخدمة من بداية التعيين حتى هذا الشهر
        $totalUsedDays = $employee->leaves()
            ->where('status', 'approved')
            ->whereHas('leaveTypeModel', function ($q) {
                $q->where('code', 'annual');
            })
            ->whereDate('start_date', '<=', $currentDate)
            ->sum('days_count');

        // الرصيد النهائي
        $balance = $totalGrantedDays - $totalUsedDays;

        if ($balance >= 0) {
            return 0; // لا يوجد خصم
        }

        // عدد الأيام الزائدة
        $excessDays = abs($balance);

        // الأجر الشهري الكلي (الأساسي + الحوافز + المكونات الأخرى)
        $totalSalary = ($employee->salary ?? 0) + ($employee->allowances ?? 0);

        foreach (['component_1','component_2','component_3','component_4','component_5'] as $comp) {
            if (!empty($employee->$comp)) {
                $totalSalary += $employee->$comp;
            }
        }

        // حساب الأجر اليومي
        $dailyRate = $totalSalary / 30;

        // حساب الخصم
        $leaveDeduction = $excessDays * $dailyRate;

        return $leaveDeduction;
    }
}
