<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Employee;
use App\Models\Attendance;
use App\Models\Leave;
use App\Models\LeaveType;
use App\Models\Activity;
use Carbon\Carbon;

class MarkAbsentAsLeave extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'attendance:mark-absent {date? : date to check (Y-m-d), default today}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Mark employees with no attendance and no approved leave as approved annual leave (auto)';

    public function handle()
    {
        $dateArg = $this->argument('date');
        $date = $dateArg ? Carbon::createFromFormat('Y-m-d', $dateArg)->startOfDay() : Carbon::today();

        $this->info('Processing absences for ' . $date->toDateString());

        $employees = Employee::where('status', 'active')->get();
        $created = 0;

        // find leave type records
        $annualType = LeaveType::where('code', 'annual')->first();
        $unpaidType = LeaveType::where('code', 'unpaid')->first();

        foreach ($employees as $emp) {
            try {
                // skip if there is attendance for date
                $hasAttendance = Attendance::where('employee_id', $emp->id)
                    ->whereDate('date', $date->toDateString())
                    ->exists();

                if ($hasAttendance) continue;

                // skip if there is an approved leave covering the date
                $hasApprovedLeave = Leave::where('employee_id', $emp->id)
                    ->where('status', 'approved')
                    ->whereDate('start_date', '<=', $date->toDateString())
                    ->whereDate('end_date', '>=', $date->toDateString())
                    ->exists();

                if ($hasApprovedLeave) continue;

                // decide whether to consume annual balance or create unpaid leave
                $availableBalance = 0;
                try {
                    $availableBalance = (int) ($emp->getAvailableBalanceAttribute() ?? 0);
                } catch (\Throwable $_) {
                }

                if ($availableBalance >= 1) {
                    // create annual leave (consumes balance)
                    $leaveData = [
                        'employee_id' => $emp->id,
                        // store the enum/code value expected by the `leaves.leave_type` column
                        'leave_type' => $annualType ? ($annualType->code ?? 'annual') : 'annual',
                        'leave_type_id' => $annualType ? $annualType->id : null,
                        'start_date' => $date->toDateString(),
                        'end_date' => $date->toDateString(),
                        'reason' => 'غياب بدون إذن',
                        'status' => 'approved',
                        'approved_by' => null,
                        'is_unpaid_deduction_applied' => false,
                    ];
                    $leave = Leave::create($leaveData);
                } else {
                    // create unpaid leave so payroll will deduct later
                    $leaveData = [
                        'employee_id' => $emp->id,
                        // store the enum/code value expected by the `leaves.leave_type` column
                        'leave_type' => $unpaidType ? ($unpaidType->code ?? 'unpaid') : 'unpaid',
                        'leave_type_id' => $unpaidType ? $unpaidType->id : null,
                        'start_date' => $date->toDateString(),
                        'end_date' => $date->toDateString(),
                        'reason' => 'غياب بدون إذن',
                        'status' => 'approved',
                        'approved_by' => null,
                        'is_unpaid_deduction_applied' => false,
                    ];
                    $leave = Leave::create($leaveData);

                    // If there's a pending payroll for this employee and month, apply immediate deduction component
                    try {
                        $payroll = \App\Models\Payroll::where('employee_id', $emp->id)
                            ->where('year', $date->year)
                            ->where('month', $date->month)
                            ->where('status', 'pending')
                            ->first();

                        if ($payroll) {
                            $dailyRate = 0;
                            try {
                                $dailyRate = $emp->getTotalSalaryWithComponentsAttribute() / 30;
                            } catch (\Throwable $_) {
                                $dailyRate = 0;
                            }

                            if ($dailyRate > 0) {
                                $payroll->payrollSalaryComponents()->create([
                                    'salary_component_id' => null,
                                    'name' => 'خصم إجازة بدون راتب',
                                    'value' => -1 * round($dailyRate, 2),
                                ]);

                                // mark leave as applied for unpaid deduction
                                $leave->is_unpaid_deduction_applied = true;
                                $leave->deduction_date = Carbon::now()->toDateString();
                                $leave->save();

                                // recalc payroll totals
                                $payroll->calculateTotalSalary();
                                $payroll->calculateNetSalary();
                                $payroll->save();
                            }
                        }
                    } catch (\Throwable $_) {
                    }
                }

                // Activity log
                try {
                    Activity::create([
                        'user_id' => auth()->id() ?? null,
                        'description' => 'auto_mark_absent_as_leave',
                        'subject_type' => Leave::class,
                        'subject_id' => $leave->id,
                        'properties' => [
                            'employee_id' => $emp->id,
                            'employee' => $emp->name,
                            'date' => $date->toDateString(),
                            'reason' => 'غياب بدون إذن (تم إنشاء تلقائيًا)'
                        ],
                    ]);
                } catch (\Throwable $_) {
                }

                $created++;
            } catch (\Throwable $e) {
                $this->error('Failed for emp=' . ($emp->id ?? 'null') . ': ' . $e->getMessage());
                continue;
            }
        }

        $this->info("Done. Created {$created} automatic leaves.");
        return 0;
    }
}
