<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Loan;

class FixLoans extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'loans:fix-amounts';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'تحديث القسط الشهري والقيمة الإجمالية لجميع القروض المخزنة بناءً على المبلغ الأساسي ونسبة الفائدة وعدد الأقساط.';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('جاري بدء تحديث بيانات القروض...');

        // استعلام عن القروض التي لا يحتوي حقل installment_amount فيها على قيمة رقمية صحيحة (أو يساوي 0)
        // ملاحظة: نفترض أن installment_amount هو الحقل الصحيح لحساب القسط الشهري من جدول loans
        $loans = Loan::where(function ($query) {
                $query->whereNull('installment_amount')
                      ->orWhere('installment_amount', 0);
            })
            // ->where('installments_count', '>', 0) // تجنب القسمة على صفر
            ->get();

        if ($loans->isEmpty()) {
            $this->info('لا توجد قروض بحاجة للتحديث.');
            return;
        }

        $progressBar = $this->output->createProgressBar(count($loans));
        $progressBar->start();

        foreach ($loans as $loan) {
            $amount = $loan->amount;
            $interestRate = $loan->interest_rate / 100; // تحويل النسبة المئوية إلى كمية عشرية
            $installmentCount = $loan->installments_count;

            // تجنب القسمة على صفر
            if ($installmentCount <= 0) {
                $this->warn("القرض #{$loan->id} له عدد أقساط غير صحيح ({$installmentCount}). تخطي...");
                continue;
            }

            // حساب الفائدة
            $interestAmount = $amount * $interestRate;

            // حساب المبلغ الإجمالي (الأصل + الفائدة)
            $totalAmount = $amount + $interestAmount;

            // حساب القسط الشهري
            $monthlyInstallment = $totalAmount / $installmentCount;

            // تحديث القرض في قاعدة البيانات
            $loan->update([
                'interest_amount' => $interestAmount,
                'total_amount' => $totalAmount,
                'installment_amount' => $monthlyInstallment,
            ]);

            $progressBar->advance();
        }

        $progressBar->finish();
        $this->newLine();
        $this->info('تم الانتهاء من تحديث بيانات القروض بنجاح!');
    }
}
