<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Leave;
use App\Models\Employee;
use Carbon\Carbon;

class FixLeaveDaysCount extends Command
{
    protected $signature = 'leave:fix-days-count';
    protected $description = 'تحديث حقل days_count في جميع سجلات الإجازات السنوية، وحساب الرصيد السنوي بناءً على عدد الشهور الفعلية في العمل';

    public function handle()
    {
        $this->info('بدء تحديث سجلات الإجازات السنوية وحساب الرصيد...');

        $leaves = Leave::whereHas('leaveTypeModel', function($q){ $q->where('code', 'annual'); })->get();

        foreach ($leaves as $leave) {
            // تحويل التواريخ إلى كائنات Carbon
            $startDate = Carbon::parse($leave->start_date)->startOfDay();
            $endDate = Carbon::parse($leave->end_date)->startOfDay();

            // حساب عدد الأيام
            if ($startDate <= $endDate) {
                $daysCount = $startDate->diffInDays($endDate) + 1;
            } else {
                $daysCount = 0;
            }

            // تحديث السجل
            $leave->update(['days_count' => (int)$daysCount]);

            $this->info("تم تحديث الإجازة السنوية رقم {$leave->id} - الموظف: {$leave->employee_id} - الأيام: {$daysCount}");
        }

        // 🔹 [تحديث] حساب الرصيد السنوي لكل موظف بناءً على عدد الشهور الفعلية في العمل
        $employees = Employee::all();

        foreach ($employees as $employee) {
            if (!$employee->hire_date) {
                $this->warn("الموظف {$employee->id} ({$employee->name}) لا يملك تاريخ توظيف، تم تجاهله.");
                continue;
            }

            $hireDate = Carbon::parse($employee->hire_date)->startOfDay(); // تأكد من بدء اليوم
            $currentDate = Carbon::now()->startOfDay(); // تأكد من بدء اليوم

            // عدد الشهور الفعلية من تاريخ التوظيف حتى الآن
            // استخدام diffInMonths يحسب عدد الشهور الكاملة بين التاريخين
            // +1 لتشمل الشهر الحالي (مثلاً، 1 يناير إلى 31 يناير = 1 شهر، 1 يناير إلى 1 فبراير = 2 شهر)
            $monthsWorked = $hireDate->diffInMonths($currentDate) + 1;

            // لا يمكن أن يتجاوز 12 شهرًا (لأن الرصيد السنوي محدود بـ 12 شهر)
            // هذا يضمن أن الحساب لا يتجاوز الحد الأقصى للسنة الكاملة
            $monthsWorked = min($monthsWorked, 12);

            // حساب الرصيد السنوي الممنوح
            // يُفترض أن monthly_leave_days_allowed هو عدد الأيام في الشهر
            $grantedAnnual = ($employee->monthly_leave_days_allowed ?? 0) * $monthsWorked;

            // حساب عدد الأيام المستخدمة من الإجازات السنوية فقط
            // تأكد من أن status هو 'approved' وleave_type هو 'annual'
            $usedAnnual = $employee->leaves()
                ->where('status', 'approved')
                ->whereHas('leaveTypeModel', function($q){ $q->where('code', 'annual'); })
                ->sum('days_count');

            $availableAnnualBalance = max(0, $grantedAnnual - $usedAnnual);

            $this->info("الموظف {$employee->id} ({$employee->name}) - الشهور في العمل: {$monthsWorked}, الرصيد الممنوح: {$grantedAnnual}, المستخدم: {$usedAnnual}, المتبقي: {$availableAnnualBalance}");
        }

        $this->info('تم تحديث جميع سجلات الإجازات السنوية وحساب الرصيد بنجاح!');
    }
}