<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Attendance;

class CheckAttendanceLeaveConsistency extends Command
{
    protected $signature = 'attendance:check-leave-consistency {--fix : Attempt to fix simple problems (set leave_id=null where invalid)}';
    protected $description = 'Check that attendances with leave_id reference an approved leave that covers the attendance date. Optionally fix simple problems.';

    public function handle()
    {
        $fix = $this->option('fix');

        $this->info('Scanning attendance records for leave_id consistency...');

        $query = Attendance::whereNotNull('leave_id');
        $total = $query->count();
        $this->info("Found {$total} attendance rows with leave_id set.");

        $invalid = 0;

        $query->chunk(200, function($rows) use (&$invalid, $fix) {
            foreach ($rows as $att) {
                try {
                    $lv = \App\Models\Leave::find($att->leave_id);
                    $d = $att->date instanceof \Carbon\Carbon ? $att->date->toDateString() : (string) $att->date;

                    $ok = false;
                    if ($lv && $lv->status === 'approved' && $lv->employee_id == $att->employee_id) {
                        $start = $lv->start_date instanceof \Carbon\Carbon ? $lv->start_date->toDateString() : (string) $lv->start_date;
                        $end = $lv->end_date instanceof \Carbon\Carbon ? $lv->end_date->toDateString() : (string) $lv->end_date;
                        if ($d >= $start && $d <= $end) {
                            $ok = true;
                        }
                    }

                    if (!$ok) {
                        $invalid++;
                        $this->line("Invalid: attendance {$att->id} -> leave_id={$att->leave_id} (emp={$att->employee_id}, date={$d})");
                        if ($fix) {
                            // Simple fix: unset the leave_id so it doesn't falsely point to wrong leave
                            $att->leave_id = null;
                            $att->save();
                            $this->line("  -> Fixed: cleared leave_id on attendance {$att->id}");
                        }
                    }
                } catch (\Throwable $e) {
                    $this->error('Error checking attendance ' . $att->id . ': ' . $e->getMessage());
                }
            }
        });

        $this->info("Scan complete. Invalid mappings found: {$invalid}. Fix applied: " . ($fix ? 'yes' : 'no'));
        return 0;
    }
}
