<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Employee;

class BulkUpdateWeeklyOffDays extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'employees:bulk-weekly-off {--days= : Comma-separated list of days (e.g. friday,saturday)} {--ids= : Comma-separated employee IDs to update} {--all : Update all employees} {--dry-run : Show changes without saving}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Bulk update weekly_off_days for employees (provide --days and either --ids or --all).';

    public function handle()
    {
        $daysOpt = trim($this->option('days') ?? '');
        if ($daysOpt === '') {
            $this->error('Please provide --days=friday,saturday');
            return 1;
        }

        $days = array_values(array_filter(array_map('trim', explode(',', $daysOpt))));
        if (empty($days)) {
            $this->error('No valid days parsed from --days');
            return 1;
        }

        $idsOpt = trim($this->option('ids') ?? '');
        $updateAll = (bool) $this->option('all');

        if (!$updateAll && $idsOpt === '') {
            $this->error('Provide --ids=1,2,3 or --all to update employees');
            return 1;
        }

        $query = Employee::query();
        if ($idsOpt !== '') {
            $ids = array_values(array_filter(array_map('intval', explode(',', $idsOpt))));
            $query->whereIn('id', $ids);
        }

        $count = $query->count();
        $this->info("Employees to update: {$count}");

        if ($count === 0) {
            $this->info('No employees matched.');
            return 0;
        }

        if ($this->option('dry-run')) {
            $this->info('Dry run mode. Showing intended changes:');
            $query->chunk(200, function ($emps) use ($days) {
                foreach ($emps as $e) {
                    $this->line("#{$e->id} {$e->name} => [" . implode(',', $days) . "]");
                }
            });
            return 0;
        }

        $updated = 0;
        $query->chunk(200, function ($emps) use ($days, &$updated) {
            foreach ($emps as $e) {
                $e->weekly_off_days = $days;
                $e->save();
                $updated++;
            }
        });

        $this->info("Updated weekly_off_days for {$updated} employees.");
        return 0;
    }
}
