<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Schema;
use App\Models\Attendance;
use App\Models\Holiday;
use App\Models\Employee;
use Carbon\Carbon;

class BackfillPaidForOffAttendances extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'attendances:mark-paid-for-off {--dry-run} {--limit=0}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Mark attendances as paid_for_off when they fall on weekly-off or official holidays and show presence.';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        if (!Schema::hasColumn('attendances', 'paid_for_off')) {
            $this->error("Column 'paid_for_off' does not exist on `attendances` table. Run migrations first (php artisan migrate).");
            return 1;
        }

        $this->info('Scanning attendances for candidates to mark as paid_for_off...');

        $query = Attendance::where(function ($q) {
            $q->whereNull('paid_for_off')->orWhere('paid_for_off', false);
        })->where(function ($q) {
            $q->where('status', 'present')->orWhereNotNull('check_in');
        });

        $limit = (int) $this->option('limit');
        if ($limit > 0) {
            $query = $query->limit($limit);
        }

        $total = $query->count();
        $this->info("Found {$total} candidate attendances.");

        if ($this->option('dry-run')) {
            $this->info('Dry run enabled — no database changes will be made.');
            return 0;
        }

        $bar = $this->output->createProgressBar($total);
        $bar->start();

        $processed = 0;
        $query->chunk(500, function ($rows) use (&$bar, &$processed) {
            foreach ($rows as $attendance) {
                try {
                    $d = $attendance->date instanceof Carbon ? $attendance->date->toDateString() : (string) $attendance->date;

                    // official holiday
                    $isHoliday = Holiday::whereDate('date', $d)->exists();

                    // employee weekly off
                    $isWeeklyOff = false;
                    try {
                        $emp = Employee::find($attendance->employee_id);
                        if ($emp) {
                            $dayKey = Carbon::parse($d)->format('l');
                            $dayKey = strtolower($dayKey);
                            $weekly = $emp->weekly_off_days ?? [];
                            if (is_array($weekly) && in_array($dayKey, $weekly)) {
                                $isWeeklyOff = true;
                            }
                        }
                    } catch (\Throwable $_) {
                        $isWeeklyOff = false;
                    }

                    if ($isHoliday || $isWeeklyOff) {
                        $attendance->paid_for_off = true;
                        $attendance->save();
                        $processed++;
                    }
                } catch (\Throwable $e) {
                    // skip problematic records
                }
                $bar->advance();
            }
        });

        $bar->finish();
        $this->newLine();
        $this->info("Marked {$processed} attendances as paid_for_off.");

        return 0;
    }
}
