<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Employee;
use App\Models\EmployeeSalaryHistory;
use Carbon\Carbon;

class BackfillEmployeeSalaryHistories extends Command
{
    protected $signature = 'backfill:employee-salaries {--from=2000-01-01}';
    protected $description = 'Backfill employee_salary_histories table from current employees data';

    public function handle()
    {
        $this->info('Starting backfill of employee salary histories...');

        $from = $this->option('from');

        $count = 0;
        Employee::chunk(100, function ($employees) use (&$count, $from) {
            foreach ($employees as $emp) {
                // determine effective_from: use hire_date if present else provided --from
                $effective = $emp->hire_date ? Carbon::parse($emp->hire_date)->startOfMonth()->toDateString() : Carbon::parse($from)->toDateString();

                // normalize to first day of month so changes apply for whole month
                $effective = Carbon::parse($effective)->startOfMonth()->toDateString();

                // prepare components array from known fields if present (component_name_i/component_i)
                $components = [];
                for ($i = 1; $i <= 7; $i++) {
                    $nameField = 'component_name_' . $i;
                    $valueField = 'component_' . $i;
                    if (!empty($emp->{$nameField}) && $emp->{$valueField}) {
                        $components[$emp->{$nameField}] = (float) $emp->{$valueField};
                    }
                }

                EmployeeSalaryHistory::create([
                    'employee_id' => $emp->id,
                    'effective_from' => $effective,
                    'base_salary' => $emp->salary ?? null,
                    'incentives' => $emp->incentives ?? null,
                    'overtime_rate' => $emp->overtime_hourly_rate ?? null,
                    'monthly_leave_entitlement' => $emp->monthly_leave_days_allowed ?? null,
                    'annual_leave_entitlement' => $emp->annual_entitlement ?? null,
                    'components' => $components ?: null,
                    'created_by' => null,
                    'note' => 'Backfill from employees table'
                ]);

                $count++;
            }
        });

        $this->info("Backfill completed. Records created: {$count}");
        return 0;
    }
}
