<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Carbon\Carbon;

class AttendanceSummary extends Command
{
    protected $signature = 'attendance:summary {--date= : date to inspect (Y-m-d), default today}';
    protected $description = 'Print attendance summary for a date and list employees without attendance and without approved leave';

    public function handle()
    {
        $dateOpt = $this->option('date');
        $date = $dateOpt ? Carbon::createFromFormat('Y-m-d', $dateOpt)->startOfDay() : Carbon::today();

        $today = $date->toDateString();

        $totalEmployees = \App\Models\Employee::where('status', 'active')->count();

        $presentCount = \App\Models\Attendance::whereDate('date', $today)->where('status', 'present')->count();

        $recordedEmployeeIds = \App\Models\Attendance::whereDate('date', $today)
            ->pluck('employee_id')
            ->unique()
            ->filter()
            ->values()
            ->toArray();

        $recordedCount = count($recordedEmployeeIds);
        $notRecordedCount = max(0, $totalEmployees - $recordedCount);

        $absent = \App\Models\Employee::where('status', 'active')
            ->whereNotIn('id', $recordedEmployeeIds ? $recordedEmployeeIds : [0])
            ->get()
            ->filter(function ($emp) use ($today) {
                $hasApprovedLeave = \App\Models\Leave::where('employee_id', $emp->id)
                    ->where('status', 'approved')
                    ->whereDate('start_date', '<=', $today)
                    ->whereDate('end_date', '>=', $today)
                    ->exists();
                return !$hasApprovedLeave;
            })
            ->map(function ($e) {
                return ['id' => $e->id, 'name' => $e->name];
            })
            ->values()
            ->all();

        $out = [
            'date' => $today,
            'total_employees' => $totalEmployees,
            'present' => $presentCount,
            'recorded' => $recordedCount,
            'not_recorded' => $notRecordedCount,
            'absent_without_leave_count' => count($absent),
            'absent_samples' => array_slice($absent, 0, 50),
        ];

        $this->line(json_encode($out, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
        return 0;
    }
}
