<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Attendance;
use App\Models\Holiday;
use App\Models\Employee;
use Carbon\Carbon;

class AttendanceBackfillHolidayPay extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'attendance:backfill-holiday-pay {--preview : Run in preview mode without saving changes} {--chunk=200 : Chunk size for processing}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Backfill attendance records: convert past present->present_on_official_holiday/present_on_weekly_off and fill holiday_pay_amount';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $preview = $this->option('preview');
        $chunkSize = (int) $this->option('chunk');

        $this->info('Starting attendance backfill for holiday pay' . ($preview ? ' (preview mode)' : ''));

        $total = 0;
        $updated = 0;
        $rowsPreview = [];

        $query = Attendance::where('status', 'present')->orderBy('id');

        $query->chunkById($chunkSize, function ($attendances) use (&$total, &$updated, &$rowsPreview, $preview) {
            foreach ($attendances as $attendance) {
                $total++;
                try {
                    $date = $attendance->date ? Carbon::parse($attendance->date)->format('Y-m-d') : null;
                    if (!$date) continue;

                    $holiday = Holiday::whereDate('date', $date)->first();

                    $emp = Employee::find($attendance->employee_id);
                    $isWeeklyOff = false;
                    if ($emp) {
                        try {
                            $dayKey = Carbon::parse($date)->format('l');
                            $dayKey = strtolower($dayKey);
                            $weekly = $emp->weekly_off_days ?? [];
                            if (is_array($weekly) && in_array($dayKey, $weekly)) {
                                $isWeeklyOff = true;
                            }
                        } catch (\Throwable $_) {
                            $isWeeklyOff = false;
                        }
                    }

                    if (!$holiday && !$isWeeklyOff) {
                        // not a holiday/weekly-off -> skip
                        continue;
                    }

                    $newStatus = $holiday ? 'present_on_official_holiday' : 'present_on_weekly_off';

                    // compute holiday pay: prefer employee default_paid_off_amount
                    $holidayPay = null;
                    if ($emp) {
                        $defaultPaid = (float) ($emp->default_paid_off_amount ?? 0);
                        if ($defaultPaid > 0) {
                            $holidayPay = round($defaultPaid, 2);
                        }
                    }
                    if ($holidayPay === null) {
                        // fallback to overtime hours * overtime rate
                        $hours = (float) $attendance->getOvertimeHours();
                        $rate = ($emp->overtime_hourly_rate ?? 0);
                        $holidayPay = round($hours * $rate, 2);
                    }

                    $rowsPreview[] = [
                        'id' => $attendance->id,
                        'employee_id' => $attendance->employee_id,
                        'date' => $date,
                        'old_status' => $attendance->status,
                        'new_status' => $newStatus,
                        'holiday_title' => $holiday?->title ?? ($isWeeklyOff ? 'weekly_off' : ''),
                        'holiday_pay' => number_format($holidayPay, 2),
                    ];

                    if (!$preview) {
                        $attendance->status = $newStatus;
                        $attendance->paid_for_off = true;
                        $attendance->holiday_pay_amount = $holidayPay;
                        $attendance->save();
                        $updated++;
                    }
                } catch (\Throwable $e) {
                    $this->error('Failed processing attendance id=' . $attendance->id . ' : ' . $e->getMessage());
                    continue;
                }
            }
        });

        $this->info('Scanned attendances: ' . $total);
        if ($preview) {
            $this->info('Preview rows (first 200):');
            $sample = array_slice($rowsPreview, 0, 200);
            if (!empty($sample)) {
                $this->table(array_keys($sample[0]), $sample);
            } else {
                $this->info('No matching present-on-holiday/weekly-off records found.');
            }
            $this->info('To perform actual update, re-run without --preview');
            return 0;
        }

        $this->info('Updated attendances: ' . $updated);
        $this->info('Done.');
        return 0;
    }
}
